# encoding: utf-8
# license: gpl3p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST

require 'securerandom'
module Iof
  module Data
    CHECK_CONSISTENCY=false
    RECURSIVE_BLACKLIST = {}
    @@namespaces=Hash.new()
    @@namespace_mutex=Mutex.new()

    def self.add_namespace(ns,url)
      @@namespace_mutex.synchronize{
        unless @@namespaces.include?(ns.intern) then
          @@namespaces[ns.intern]=Hash.new()
          @@namespaces[ns.intern][:ns]=ns
          @@namespaces[ns.intern][:name]=ns.upcase
          @@namespaces[ns.intern][:url]=url
          mod=Module.new() do
          end
          @@namespaces[ns.intern][:mod]=mod
          Iof::Data.const_set(@@namespaces[ns.intern][:name],mod)
          mod.const_set("NS",ns)
          mod.const_set("URL",url)
        end
        return @@namespaces[ns.intern][:module]
      }
    end

    def self.namespace_to_url(ns)
      @@namespace_mutex.synchronize{
        return nil if ns.nil?()
        unless @@namespaces.include?(ns.intern) then
          return nil
        else
          return @@namespaces[ns.intern][:url]
        end
      }
    end

    def self.url_to_namespace(url)
      @@namespace_mutex.synchronize{
        return nil if url.nil?()
        result=nil
        @@namespaces.each do |ns,data|
          if data[:url]==url then
            result=data[:ns]
            break
          end
        end
        return result
      }
    end

    def self.namespace_to_name(ns)
      @@namespace_mutex.synchronize{
        return nil if ns.nil?()
        unless @@namespaces.include?(ns.intern) then
          return nil
        else
          return @@namespaces[ns.intern][:name]
        end
      }
    end

    def self.name_to_namespace(name)
      @@namespace_mutex.synchronize{
        return nil if name.nil?()
        result=nil
        @@namespaces.each do |ns,data|
          if data[:name]==name then
            result=data[:ns]
            break
          end
        end
        return result
      }
    end

    def self.namespace_to_module(ns)
      @@namespace_mutex.synchronize{
        return nil if ns.nil?()
        unless @@namespaces.include?(ns.intern) then
          return nil
        else
          return @@namespaces[ns.intern][:mod]
        end
      }
    end

    def self.module_to_namespace(mod)
      @@namespace_mutex.synchronize{
        return nil if mod==self
        return mod::NS
      }
    end

    def self.const_missing(name)
      caller=Ghun::Base.caller_method()
      STDERR.puts "Request for missing constant '#{name}' by method #{caller[2]} (#{caller[0]}:#{caller[1]})"
      return nil
    end

    class Base
      include Ghun::Base::Logging
      @@known=Hash.new()
      @@known_mutex=Monitor.new()
      class << self
        include Ghun::Base::Logging

        def init(type,name=nil)
          init_logging(Ghun::Log::Source::DATA)
          @type=type
          @name=name
          @ontology_name=nil
          @short_name=nil
          @uri=nil
          @ns=nil
          @module=nil
          @options=Hash.new()
          @direct_super=nil
          @sups=Hash.new()
          @subs=Hash.new()
          @@known_mutex.synchronize {
            @@known[@type]=Hash.new() unless @type.nil?() || @@known.include?(@type)
          }
        end
        attr_accessor :module,:short_name,:type,:name,:ontology_name, :uri, :ns, :options, :sups, :subs, :direct_super

        def new_sub(ns,name)
          raise Ghun::Base::AbstractClassError, "'ontology_to_ruby_name' must not be called on Iof::Data::Base" if @type.nil?()
          ruby_name=ontology_to_ruby_name(name)
          object=Class.new(self) do
          end
          object.init(@type,ruby_name)
          object.ns=ns
          object.ontology_name=name.to_s
          object.short_name=ruby_to_short_name(object.name,object.ns)
          object.uri="#{Iof::Data.namespace_to_url(ns.intern)}#{name}"
          object.options=Hash.new()
          object.direct_super=self
          self.sups.each do |namespace,classes|
            classes.each do |n,cls|
              object.sups[namespace]=Hash.new() unless object.sups.include?(namespace)
              object.sups[namespace][n]=cls
            end
          end
          object.sups[self.ns]=Hash.new() unless object.sups.include?(self.ns)
          object.sups[self.ns][self.name.intern]=self
          self.add_sub(ruby_name.intern,object)
          @@known_mutex.synchronize {
            @@known[@type][object.ns]=Hash.new() unless @@known[@type].include?(object.ns)
            @@known[@type][object.ns][ruby_name.intern]=object
          }
          mod=Iof::Data.namespace_to_module(object.ns)
          mod=Iof::Data if mod.nil?()
          mod.const_set(ruby_name,object)
          return mod.const_get(ruby_name)
        end

        def add_sub(name,klass)
          @subs[klass.ns]=Hash.new() unless @subs.include?(klass.ns)
          @subs[klass.ns][name]=klass unless @subs[klass.ns].include?(name)
          self.direct_super.add_sub(name,klass) unless self.direct_super.nil?()
        end

        def is_sub_of?(object,ns)
          c_object=nil
          if object.is_a?(Iof::Data::Base) then
            c_object=object.class.to_s.intern
          elsif object.is_a?(Symbol) || object.is_a?(String) then
            c_object=object.intern
          elsif object < Base
            c_object=object.name.intern
          else
            raise ArgumentError, "#{object.class} is an unexpected type"
          end
          #FIXME
          if @sups.include?(ns) then
            @sups[ns].keys.include?(c_object)
          else
            false
          end
        end

        #FIXME
        def is_super_of?(object,ns)
          c_object=nil
          if object.is_a?(Base) then
            c_object=object.class.to_s.intern
          elsif object.is_a?(Symbol) || object.is_a?(String) then
            c_object=object.intern
          elsif object < Base
            c_object=object.name.intern
          else
            raise ArgumentError, "#{object.class} is an unexpected type"
          end
          if @subs.include?(ns) then
            @subs[ns].keys.include?(c_object)
          else
            false
          end
        end

        def ontology_to_ruby_name(name)
          #ruby classes need uppercase first letter
          new=name.to_s
          new[0]=new[0].upcase
          return new
        end

        def ruby_to_short_name(name,ns)
          #ruby classes need uppercase first letter
          new=name.clone
          new.gsub!(/TCP/,'Tcp')
          new.gsub!(/UDP/,'Udp')
          new.gsub!(/ACL/,'Acl')
          new.gsub!(/IP/,'Ip')
          new.gsub!(/ID/,'Id')
          new.gsub!(/YAML/,'Yaml')
          new.gsub!(/ICMP/,'Icmp')
          new.gsub!(/VSA/,'Vsa')
          new.gsub!(/^Has([0-9A-Z]+[0-9a-z]+)/,'\1')
          new.gsub!(/^Is([0-9A-Z]+[0-9a-z]+)/,'\1')
          new.gsub!(/([A-Z]+)/,'_\1')
          new.downcase!
          new.gsub!(/^_+/,'')
          new="#{ns.to_s.downcase}_#{new}"
          return new
        end

        def generate_random_name()
          SecureRandom.uuid()
        end

        def sanitize_name(name)
          new=""
          new=name.tr(' ','_') unless name.nil?()
#          new.tr!('.','_')
#          new.tr!(':','_')
          new.tr!('^ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789_\-/.:','')
          new=generate_random_name() if new==""
          return new
        end
      end

      def initialize()
        init_logging(Ghun::Log::Source::DATA)
        @name=nil
        if self.class.options[:reasoning_result] then
          error "Trying to instantiate '#{self.class.to_s}', which is marked as reasoner output"
          raise Iof::Data::ReasoningResultUsageError, "Trying to instantiate '#{self.class.to_s}', which is marked as reasoner output"
        elsif self.class.options.include?(:reasoning_result) && self.class.options[:reasoning_result] then
          error "Trying to instantiate '#{self.class.to_s}', which is marked as reasoner output"
          raise Iof::Data::ReasoningResultUsageError, "Trying to instantiate '#{self.class.to_s}', which is marked as reasoner output"
        end
      end

      def generate_random_name()
        self.class.generate_random_name()
      end

      def sanitize_name(name)
        self.class.sanitize_name(name)
      end
    end

    class BaseClass < Base
      class << self
        def init(type=:class,name=nil)
          super(type,name)
          @in_domain_of=Array.new()
          @in_range_of=Array.new()
        end
        attr_accessor :in_domain_of, :in_range_of

        def load(ontology,name)
          #TODO implement me
        end

        def delete(ontology,name,ns)
          debug "Dropping individual '#{name}'"
          if ontology.exists_individual?(name,ns) then
            debug "Thing '#{name}' exists"
            #XXX Broken: work without get_all_object_properties
            properties=ontology.get_all_object_properties(name,Iof::Data::RECURSIVE_BLACKLIST,ns)
            unless properties.nil?()
              debug "Dropping all properties for thing '#{name}'"
              properties.each do |p|
                debug "Dropping property #{p} for thing '#{name}'"
                ontology.drop_property(name, p[:name],ns,p[:ns])
              end
              debug "Dropping connected individuals"
              properties.each do |p|
                p[:values].each do |v|
                  debug "Dropping individual #{v.to_s} connected via object property #{p.to_s}"
                  delete(ontology,v[:name],v[:ns]) unless v.nil?() && ontology.exists_individual?(v[:name],v[:ns])
                end unless p[:values].nil?()
              end
              ontology.drop_individual(name)
            end
          else
            warn "Thing '#{name}' does not exist"
          end
        rescue => e
          error "Failed to drop individual '#{name.to_s}'",e
          return nil
        end

        def search_by(ontology,search_by,values,properties,klass=self)
          names=Array.new()
          search_by.each do |id|
            identifier=Hash.new()
            identifier[:class]=properties[id]
            if identifier.nil?() || identifier[:class]==Iof::Data::BaseClass then
              error "Got no class for name #{id.to_s}"
              next
            elsif identifier[:class].type==:object_property then
              error "Searching by object properties is not supported yet"
              next
            else
              identifier[:uri]=identifier[:class].uri
              identifier[:name]=identifier[:class].name
            end
            classes=Array.new()
            sup=klass.direct_super
            classes << {:ns => @ns, :name => @ontology_name, :uri => @uri }
            until sup.nil?() || sup==Iof::Data::BaseClass do
              classes << {:ns => sup.ns, :name => sup.ontology_name, :uri => sup.uri }
              sup=sup.direct_super
            end
            classes.each do |cls|
              values[id].each do |value|
                begin
                  result=ontology.query("SELECT ?c WHERE {?c a <#{cls[:uri]}> . ?c <#{identifier[:uri]}> \"#{value[:direct]}\"}",true,false,[])
                  result.each do |line|
                    names << {:name => line[:c], :ns => line[:__ns_c]}
                    debug "Found '#{{:name => line[:c], :ns => line[:__ns_c]}}' for '#{value[:direct]}'"
                  end unless result.nil?()
                rescue => e
                  error "Failed to search with identifier '#{identifier[:name]}' and value '#{value}' for individual with class '#{cls[:name]}'", e
                end
              end unless values[id].empty?()
            end
          end
          if names.empty?() then
            #found nothing
            return nil
          elsif names.size()==1 then
            return names[0]
          else
            search_result=nil
            names.each do |n|
              if search_result.nil?() then
                search_result=n
              else
                return nil unless n[:ns]==search_result[:ns] && n[:name]==search_result[:name]
              end
            end
            return search_result
          end
        end

        def from_iof_yaml(yaml,top=true)
          parsed=YAML.load(yaml) if top
          parsed=yaml unless top
          result=nil
          ns=parsed[:ns]
          klass=parsed[:class]
          klasses=parsed[:classes]
          @@known_mutex.synchronize{
            result=@@known[@type][ns][klass].new(klasses)
          }
          parsed[:values].each do |value|
            v=nil
            if value[:native] then
              v=value[:value]
            else
              v=Iof::Data::BaseClass.from_iof_yaml(value[:value],false)
            end
            result.set(value[:key],v)
          end
          return result
        end
      end
      self.init(:class,"BaseClass")

      def initialize(additional_classes=[])
        super()
        @classes=Array.new()
        @primary_class=nil
        @in_domain_of=self.class.in_domain_of.clone
        @in_range_of=self.class.in_range_of.clone
        @properties=Hash.new()
        #used for storage ordering for top level objects
        @category=nil
        if self.class.options[:naming_policy]==:unique || self.class.options[:naming_policy]==:parent_and_unique then
          @name=Iof::Data::Base.generate_random_name()
        else
          @name=nil
        end
        add_class(self.class.name.intern,self.class.ns.intern,true)
        additional_classes.each do |klass|
          add_class(klass)
        end unless additional_classes.nil?()
      end
      attr_reader :name
      attr_accessor :category

      def name()
        @name=Iof::Data::Base.generate_random_name if @name.nil?() && self.class.options[:naming_policy]!=:own && self.class.options[:naming_policy]!=:parent_and_own
        return @name
      end

      def name=(name)
        @name=sanitize_name(name)
        return @name
      end

      def add_class(klass,ns,primary=false)
        c_klass=nil
        if klass.is_a?(BaseClass) then
          c_klass=klass.class.name.intern
        elsif klass.is_a?(Symbol) || klass.is_a?(String) then
          c_klass=klass.intern
        elsif klass.is_a?(Class)
          if klass.respond_to?(:name) then
            c_klass=klass.name.to_s.intern
          else
            c_klass=klass.to_s.intern
          end
        else
          raise ArgumentError, "#{klass.class} is an unexpected type"
        end
        @primary_class=c_klass if primary
        @classes << c_klass unless @classes.include?(c_klass)
        @@known_mutex.synchronize {
          @in_domain_of.concat(@@known[self.class.type][ns][c_klass].in_domain_of.clone)
          @in_range_of.concat(@@known[self.class.type][ns][c_klass].in_range_of.clone)
        }
        update_properties
      end

      def get(key)
        if @values.nil?() || !@values.include?(key) || @values[key].nil?() then
          return nil
        elsif @values[key].is_a?(Array) then
          result=Array.new()
          @values[key].each do |v|
            result << v[:direct] if v.include?(:direct)
          end
          return result unless result.empty?()
        else
          return @values[key][:direct] if @values[key].include?(:direct)
        end
        return nil
      end
      alias :"[]" :get

      def set(key,value)
        if value.is_a?(Array) then
          value.each do |v|
            set(key,v)
          end
        else
          if @properties.include?(key) && !@properties[key].options[:disable_mapping] && !@properties[key].options[:tree_leaving_property] then
            #FIXME load ns
            if value.nil?() then
              if @values[key].is_a?(Array) then
                @values[key]=Array.new()
              else
                @values[key] = nil
              end
            elsif @properties[key].is_in_range?(value,:no) then
              if @values[key].is_a?(Array) then
                #FIXME load ns
                @values[key] << { :direct => value, :ns => :no }
              else
                #FIXME load ns
                @values[key] = { :direct => value, :ns => :no }
              end
            else
              error "Value '#{value}' for property '#{key}' of individual '#{@name}' with class '#{self.class.to_s}' is invalid"
              raise Iof::Data::ConstraintViolationError, "Value '#{value}' for property '#{key}' of individual '#{@name}' with class '#{self.class.to_s}' is invalid"
            end
          elsif @properties.include?(key) && !@properties[key].options[:disable_mapping] && @properties[key].options[:tree_leaving_property] then
            #FIXME load ns
            if value.nil?() then
              if @values[key].is_a?(Array) then
                @values[key]=Array.new()
              else
                @values[key] = nil
              end
            else
              if @values[key].is_a?(Array) then
                #FIXME load ns
                @values[key] << { :indirect => value, :ns => :no }
              else
                #FIXME load ns
                @values[key] = { :indirect => value, :ns => :no }
              end
            end
          else
            if value.respond_to?(:name) then
              warn "Setting value '#{value.name}' for key '#{key.to_s}' without corresponding property"
            else
              warn "Setting value '#{value.inspect}' for key '#{key.to_s}' without corresponding property"
            end
            @values[key]=Array.new() unless @values.include?(key)
            #FIXME load ns
            @values[key] << { :direct => value, :ns => :no }
          end
        end
      end
      alias :"[]=" :set

      def to_iof_yaml(top=true)
        result=Hash.new()
        result[:name]=@name
        result[:class]=@primary_class
        result[:ns]=self.class.ns
        result[:classes]=@classes.clone-[@primary_class]
        result[:values]=Array.new()
        @values.each do |key,value|
          if value.is_a?(Array) then
            value.each do |v|
              next if v.nil?()
              r=Hash.new()
              r[:key]=key
              r[:ns]=v[:ns]
              if v.include?(:direct) && v[:direct].is_a?(Iof::Data::BaseClass) then
                r[:native]=false
                r[:value]=v[:direct].to_iof_yaml(false)
              elsif v.include?(:direct) && !v[:direct].is_a?(Iof::Data::BaseClass)
                r[:native]=true
                r[:value]=v[:direct]
              else
                r[:native]=true
                r[:value]=v[:indirect]
              end
              result[:values] << r
            end unless value.nil?()
          else
            next if value.nil?()
            r=Hash.new()
            r[:key]=key
            r[:ns]=value[:ns]
            if value.include?(:direct) && value[:direct].is_a?(Iof::Data::BaseClass) then
              r[:native]=false
              r[:value]=value[:direct].to_iof_yaml(false)
            elsif value.include?(:direct) && !value[:direct].is_a?(Iof::Data::BaseClass)
              r[:native]=true
              r[:value]=value[:direct]
            else
              r[:native]=true
              r[:value]=value[:indirect]
            end
            result[:values] << r
          end
        end
        return result unless top
        return result.to_yaml if top
      end

      def is_class?(klass,ns)
        c_klass=nil
        if klass.is_a?(BaseClass) then
          c_klass=klass.class.to_s.intern
        elsif klass.is_a?(Symbol) || klass.is_a?(String) then
          c_klass=klass.intern
        elsif klass.is_a?(Class)
          c_klass=klass.to_s.intern
        else
          raise ArgumentError, "#{klass.class} is an unexpected type"
        end
        result=false
        result||=@classes.include?("#{ns}___#{c_klass}")
        result||=self.class.is_sub_of?(c_klass,ns)
      end

      def update_properties()
        @values=Hash.new() unless defined?(@values) && !@values.nil?()
        @properties=Hash.new
        @in_domain_of.each do |cls|
          property=Iof::Data.namespace_to_module(cls[:ns]).const_get(cls[:name])
          @properties[property.short_name.intern]=property unless @properties.include?(property.short_name.intern)
        end
        @properties.each do |name,property|
          if property.options.include?(:reasoning_result) && property.options[:reasoning_result] then
            next
          elsif property.options.include?(:disable_mapping) && property.options[:disable_mapping] then
            next
          elsif property.domain.nil?() || property.range.nil?() then
            next
          elsif property.options.include?(:single_usage) && property.options[:single_usage]
            @values[name]=nil unless @values.include?(name)
          elsif property.options.include?(:single_usage) && !property.options[:single_usage]
            @values[name]=Array.new unless @values.include?(name)
          elsif !property.options.include?(:single_usage)
            @values[name]=Array.new unless @values.include?(name)
          else
            @values[name]=Array.new unless @values.include?(name)
          end
        end
      end

      def store(changed_thing_names,ontology,parent_name,ns,force_name=nil,force_exists=nil,search_by=[:no_identifier],keep_all_data=false)
        exist=nil
        need_name=nil
        ontology_name=nil
        #search by identifier if search_by given and top_level_object
        search=!(search_by.nil?() || search_by.empty?() || !self.class.options[:top_level_object])
        #fail if force_exists=true but no name given
        raise ArgumentError, "Providing name of existing individual for force_exist=true mandatory" if (force_name.nil?() || force_name=="" ) && force_exists==true
        if (force_name.nil?() || force_name=="" )
          #no valid name given
          need_name=true
          ontology_name=""
        else
          #name given
          ontology_name=force_name
          need_name=false
        end

        if !force_name.nil?() && !force_name=="" && force_exists==true then
          exist=true
          need_name=false
          ontology_name=force_name
          search=false
        elsif force_name.nil?() && force_exists==false then
          need_name=true
          exist=false
          search=false
        elsif !force_name.nil?() && force_exists==false then
          need_name=false
          ontology_name=force_name
          #exist=false
        end
        if need_name then
          #generate name
          ontology_name=""
          #ontology_name+="#{parent_name.sub(/_sub[0-9]+$/,'')}___" if !parent_name.nil?() && self.class.options[:naming_policy]==:parent_and_own || self.class.options[:naming_policy]==:parent_and_own_or_unique || self.class.options[:naming_policy]==:parent_and_unique
          ontology_name+="#{parent_name}___" if !parent_name.nil?() && self.class.options[:naming_policy]==:parent_and_own || self.class.options[:naming_policy]==:parent_and_own_or_unique || self.class.options[:naming_policy]==:parent_and_unique
          if @name.nil?() then
            if self.class.options[:naming_policy]==:own || self.class.options[:naming_policy]==:parent_and_own then
              raise Iof::Data::ConstraintViolationError, "Individual has no name and naming policy 'own'"
            else
              @name=Iof::Data::Base.generate_random_name()
              ontology_name+=@name
            end
          else
            ontology_name+=@name
          end
          # overriding force_exists to prevent duplicate names, by enabling name based check
          exist=nil
        end

        if ontology.exists_individual?(ontology_name,ns) then
          exist=true
          #no need for searching
          search=false
        else
          #keep search setting unchanged
          exist=false
        end if exist.nil?() || !exist

        if search then
          #search by given identifier; ontology_name and exists can change here
          values=Hash.new()
          search_by.each do |id|
            values[id]=Array.new()
            if @values[id].is_a?(Array)
              @values[id].each do |v|
                values[id] << v
              end unless @values[id].nil?()
            else
              values[id] << @values[id] unless @values[id].nil?()
            end
          end
          result=search_by(ontology,search_by,values,@properties)
          if result.nil?() || result.empty?() then
            exist=false
          else
            debug "Found individual '#{result[:name]}' as search result for '#{ontology_name}'"
            exist=true
            ontology_name=result[:name]
            ns=result[:ns]
          end
        end

        result=nil
        if exist then
          #individual exists; calling store_helper to update
          result=store_helper(changed_thing_names,ontology,ontology_name,ns,true,keep_all_data)
        else
          #individual does not exist; calling store_helper to create
          result=store_helper(changed_thing_names,ontology,ontology_name,ns,false,false)
        end
        if ontology_name!=result[:name]
          warn "Thing changed name from #{ontology_name} to #{result[:name]}"
          changed_thing_names[ontology_name.intern]=result[:name]
        end unless result.nil?()
        return result
      rescue => e
        error "Failed to store individual #{@name}", e
        return nil
      end

      def clone(with_name=true)
        classes=@classes.clone()-[@primary_class]
        new=nil
        if classes.nil?() || classes.empty?() then
          new=self.class.new()
        else
          new=self.class.new(classes)
        end
        new.name=@name if with_name
        @values.each do |key,value|
          next if value.nil?()
          if value.is_a?(Array) then
            value.each do |v|
              next if v.nil?()
              begin
                if v.include?(:indirect) then
                  new.set(key,v[:indirect].clone)
                elsif v.include?(:direct) then
                  if v[:direct].is_a?(Iof::Data::BaseClass) then
                    new.set(key, v[:direct].clone(with_name))
                  else
                    new.set(key, v[:direct].clone)
                  end
                end
              rescue
                if v.include?(:indirect) then
                  new.set(key,v[:indirect])
                elsif v.include?(:direct) then
                  new.set(key, v[:direct]) if v.include?(:direct)
                end
              end
            end
          else
            begin
              if value.include?(:indirect) then
                new.set(key,value[:indirect].clone)
              elsif value.include?(:direct) then
                if value[:direct].is_a?(Iof::Data::BaseClass) then
                  new.set(key, value[:direct].clone(with_name))
                else
                  new.set(key, value[:direct].clone)
                end
              end
            rescue
              if value.include?(:indirect) then
                new.set(key,value[:indirect])
              elsif value.include?(:direct) then
                new.set(key, value[:direct])
              end
            end
          end
        end
        return new
      rescue => e
        error "Failed to clone #{@name}", e
        return nil
      end

    private

      def search_by(ontology,search_by,values,properties)
        self.class.search_by(ontology,search_by,values,properties)
      end

      def store_helper(changed_thing_names,ontology,raw_name,ns,updating,keep_all_data)
        if updating && keep_all_data then
          debug "Safe updating '#{self.class.to_s}' '#{ns}:#{raw_name}'"
        elsif updating && !keep_all_data then
          debug "Unsafe updating '#{self.class.to_s}' '#{ns}:#{raw_name}'"
        elsif !updating then
          debug "Creating '#{self.class.to_s}' '#{ns}:#{raw_name}'"
        end
        name=nil
        unless updating
          debug "Creating individual '#{self.class.to_s}' '#{ns}:#{raw_name}'"
          begin
            stored=ontology.create_individual(self.class.ontology_name,raw_name,self.class.ns,ns)
            raise StandardError, "Storing for #{raw_name} failed" if stored.nil?()
            name=stored[:name]
          rescue Exception => e
            error "Failed to create #{raw_name}", e
            return nil
          end
        else
          name=raw_name
        end
        @values.each do |attr,value|
          begin
            #skip if while creating or safe updating without data to safe
            if (keep_all_data || !updating) && (value.nil?() || (value.is_a?(Array) && value.empty?()) || (value.is_a?(Array) && value.size==1 && value[0].nil?())) then
              debug "Nothing to store or cleanup for #{attr.to_s}"
              next
            end
            property=nil
            if @properties.include?(attr) then
              property=@properties[attr]
            else
              if value.is_a?(Iof::Data::BaseClass) then
                warn "Attribute '#{attr}' is unknown in ontology; not storing value '#{value.name}' of class #{value.class}"
              elsif value.is_a?(Array) then
                warn "Attribute '#{attr}' is unknown in ontology; not storing '#{value.size}' individuals"
              elsif value.is_a?(Hash) && value.include?(:direct) && value[:direct].is_a?(Iof::Data::BaseClass)
                warn "Attribute '#{attr}' is unknown in ontology; not storing value '#{value[:direct].name}' of class #{value[:direct].class}"
              else
                warn "Attribute '#{attr}' is unknown in ontology; not storing value '#{value}'"
              end
              next
            end

            #clean up if no data given and doing unsafe update
            if property.type==:object_property then
              if property.options[:tree_leaving_property] then
                ontology.drop_object_property(name, property.ontology_name,ns,property.ns)
              else
                existing_value=ontology.get_object_property(name, property.ontology_name,ns,property.ns)
                existing_value=[existing_value] unless existing_value.nil?() || existing_value.is_a?(Array)
                existing_value.each do |ev|
                  Iof::Data::BaseClass.delete(ontology,ev[:name],ev[:ns])
                end unless existing_value.nil?()
              end
            elsif property.type==:datatype_property then
              ontology.drop_datatype_property(name, property.ontology_name,ns,property.ns)
            else
              warn "Storing properties of type #{property.type} not supported yet"
              next
            end if !updating && !keep_all_data && ( value.nil?() || (value.is_a?(Array) && value.empty?()) || (value.is_a?(Array) && value.size==1 && value[0].nil?()))

            values=value
            values=[value] unless value.is_a?(Array)

            #updating/creating multiple properties
            existing=[]
            no_data_update=false
            no_indirect_update=false
            #read existing values if updating
            #simulate no existing values otherwise
            if @properties[attr].type==:object_property && !property.options[:tree_leaving_property] then
              #FIXME will report only properties in default ns
              existing=ontology.get_object_property(name, property.ontology_name,ns,property.ns)
              existing=[existing] unless existing.is_a?(Array)
              existing=[] if existing.nil?()
            elsif @properties[attr].type==:object_property && property.options[:tree_leaving_property] then
              #FIXME will report only properties in default ns
              existing=ontology.get_object_property(name, property.ontology_name,ns,property.ns)
              existing=[existing] unless existing.is_a?(Array)
              existing=[] if existing.nil?()
              values_tmp=Array.new()
              values.each do |v|
                next if v.nil?() || v[:indirect].nil?()
                values_tmp << v[:indirect]
              end
              values_tmp.sort!
              existing_tmp=Array.new()
              existing.each do |v|
                next if v.nil?() || v[:name].nil?()
                existing_tmp << v[:name]
              end
              existing_tmp.sort!
              if values_tmp==existing_tmp then
                no_indirect_update=true
              elsif !keep_all_data
                ontology.drop_object_property(name, property.ontology_name,ns,property.ns)
                existing=[]
              end
            elsif @properties[attr].type==:datatype_property then
              #FIXME will report only properties in default ns
              existing=ontology.get_datatype_property(name, property.ontology_name,ns,property.ns)
              existing=[existing] unless existing.is_a?(Array)
              existing=[] if existing.nil?()
              tmp=Array.new()
              values.each do |v|
                next if v.nil?() || v[:direct].nil?()
                tmp << v[:direct]
              end
              tmp.sort!
              existing.sort!
              if tmp==existing then
                no_data_update=true
              elsif !keep_all_data
                ontology.drop_datatype_property(name, property.ontology_name,ns,property.ns)
                existing=[]
              end
            else
              warn "Storing properties of type #{property.type} not supported yet"
              next
            end if updating

            values.each do |v|
              next if v.nil?() || (v[:direct].nil?() && v[:indirect].nil?())
              if property.type==:object_property && property.options[:tree_leaving_property] then
                if existing.include?({:name => v[:indirect], :ns => v[:ns], :unknown => false}) then
                  existing.delete({:name => v[:indirect], :ns => v[:ns], :unknown => false})
                else
                  ontology.create_object_property(changed_thing_names,name, property.ontology_name, v[:indirect], ns, property.ns, v[:ns]) unless no_indirect_update
                end
              elsif property.type==:object_property && !property.options[:tree_leaving_property] then
                if existing.empty?() then
                  stored=v[:direct].store(changed_thing_names,ontology, name, v[:ns], nil, false, [], keep_all_data)
                  raise StandardError, "Storing for #{v[:direct].name} failed" if stored.nil?()
                  ontology.create_object_property(changed_thing_names,name, property.ontology_name, stored[:name], ns, property.ns, stored[:ns])
                elsif existing.size==1 && values.size==1 && property.range.size==1 && !existing[0].nil?() && (values[0][:direct].class.options[:naming_policy]==:unique || values[0][:direct].class.options[:naming_policy]==:parent_and_unique) then
                  #prevent deletion and recreation of e.g. ACL or ForwardSets
                  stored=v[:direct].store(changed_thing_names,ontology, name, existing[0][:ns], existing[0][:name], true, [], keep_all_data)
                  raise StandardError, "Storing for #{v[:direct].name} failed" if stored.nil?()
                  existing=[]
                elsif v[:direct].class.options[:naming_policy].to_s.match(/^parent/) && existing.include?({:name => "#{name}___#{v[:direct].name}", :ns => v[:ns], :unknown => false}) then
                  stored=v[:direct].store(changed_thing_names,ontology, name, v[:ns], "#{name}___#{v[:direct].name}", true, [], keep_all_data)
                  raise StandardError, "Storing for #{name}___#{v[:direct].name} failed" if stored.nil?()
                  existing.delete({:name => "#{name}___#{v[:direct].name}", :ns => v[:ns], :unknown => false})
                elsif !v[:direct].class.options[:naming_policy].to_s.match(/^parent/) && existing.include?({:name => v[:direct].name, :ns => v[:ns], :unknown => false}) then
                  stored=v[:direct].store(changed_thing_names,ontology, name, v[:ns], v[:direct].name, true, [], keep_all_data)
                  raise StandardError, "Storing for #{v[:direct].name} failed" if stored.nil?()
                  existing.delete({:name => v[:direct].name, :ns => v[:ns], :unknown => false})
                else
                  stored=v[:direct].store(changed_thing_names,ontology, name, v[:ns], nil, false, [], keep_all_data)
                  raise StandardError, "Storing for #{v[:direct].name} failed" if stored.nil?()
                  ontology.create_object_property(changed_thing_names,name, property.ontology_name, stored[:name], ns, property.ns, stored[:ns])
                end
              elsif property.type==:datatype_property then
                if existing.include?({:name => v[:direct], :ns => v[:ns], :unknown => false}) then
                  existing.delete({:name => v[:direct], :ns => v[:ns], :unknown => false})
                else
                  ontology.create_datatype_property(name, property.ontology_name, v[:direct].to_s, ns, property.ns) unless no_data_update
                end
              else
                warn "Storing properties of type #{property.type} not supported yet"
                next
              end
            end unless value.nil?() || no_indirect_update || no_data_update
            existing.each do |ev|
              next if ev.nil?()
              Iof::Data::BaseClass.delete(ontology,ev[:name],ev[:ns])
            end if updating && !keep_all_data && !property.options[:tree_leaving_property] && property.type==:object_property
          rescue Exception => e
            if value.is_a?(Iof::Data::BaseClass) then
              error "Failed to store individual '#{value.name}' of class '#{value.class}' for property '#{property.name}' of '#{name}'", e
            elsif value.is_a?(Array) then
              error "Failed to store #{value.size} individuals for property '#{property.name} of '#{name}''", e
            elsif value.is_a?(Hash) && value.include?(:direct) && value[:direct].is_a?(Iof::Data::BaseClass)
              error "Failed to store individual '#{value[:direct].name}' of class '#{value[:direct].class}' for property '#{property.name}' of '#{name}'", e
            else
              error "Failed to store value '#{value}' for property '#{property.name}' of '#{name}'", e
            end
          end
        end
        return {:name => name, :ns => ns}
      end
    end

    class BaseProperty < Base
      class << self
        def init(type=:property,name=nil)
          super(type,name)
          @domain=Array.new()
          @range=Array.new()
        end
        attr_accessor :domain, :range

        def is_in_data_range?(value,garbage=nil)
          return true unless Iof::Data::CHECK_CONSISTENCY
          #TODO extend data type support
          return value.is_a?(String) || value.is_a?(Symbol) || value.is_a?(Integer) || value.is_a?(TrueClass) || value.is_a?(FalseClass)
        end

        def is_in_object_range?(klass,ns)
          return true unless Iof::Data::CHECK_CONSISTENCY
          result=true
          result&&=!klass.nil?()
          result&&=klass.is_a?(Iof::Data::BaseClass) || klass.is_a?(String)
          @range.each do |cls|
            klass.is_class?(cls,ns)
          end unless result
          return result
        end

        def is_in_domain?(klass,ns)
          return true unless Iof::Data::CHECK_CONSISTENCY
          result=true
          result&&=!klass.nil?()
          result&&=klass.is_a?(Iof::Data::BaseClass) || klass.is_a?(String)
          @domain.each do |cls|
            klass.is_class?(cls,ns)
          end unless result
          return result
        end
      end
      self.init(:property,"BaseProperty")

      def initialize()
        super()
        if self.class.domain.nil?() || self.class.nil?()  then
          error "Trying to instantiate property '#{self.class.to_s}' with invalid domain or range definition"
          raise Iof::Data::ConstraintViolationError, "Trying to instantiate property '#{self.class.to_s}' with invalid domain or range definition"
        end
        @subject=nil
        @object=nil
        @annotation=nil
        @timestamp=nil
      end
      attr_reader :subject, :object, :annotation, :timestamp

      #FIXME load default ns
      def set_subject(subject,ns=:no)
        raise ArgumentError, "set_subject only takes a reference (name) to an individual" if subject.is_a?(Iof::Data::Base)
        @subject={:name => subject, :ns => ns}
      end

      #FIXME load default ns
      def set_object(object,ns=:no)
        raise ArgumentError, "set_object only takes a reference (name) to an individual or a value" if object.is_a?(Iof::Data::Base)
        if self.class.type==:object_property then
          @object={:name => object, :ns => ns}
        else
          if is_in_data_range?(object) then
            @object=object
          else
            raise ConstraintViolationError, "#{object} does not fit into range of #{self.class.to_s}"
          end
        end
      end

      def subject=(subject)
        set_subject(subject)
      end

      def object=(object)
        set_object(object)
      end

      def annotation=(annotation)
        if annotation.is_a?(String) || annotation.nil?()
          @annotation=annotation
        else
          @annotation=annotation.to_s()
        end if self.class.options[:allows_annotation]
      end

      def timestamp=(timestamp)
        if timestamp.is_a?(String) || timestamp.nil?()
          @timestamp=timestamp
        else
          @timestamp=timestamp.to_s()
        end if self.class.options[:allows_annotation]
      end

      def is_property?(property)
        c_property=nil
        if property.is_a?(BaseProperty) then
          c_property=property.class.to_s.intern
        elsif property.is_a?(Symbol) || property.is_a?(String) then
          c_property=property.intern
        elsif property.is_a?(Class)
          c_property=property.to_s.intern
        else
          raise ArgumentError, "#{property.class} is an unexpected type"
        end
        result=false
        result||=self.class.is_sub_of?(c_property)
      end

      def is_in_data_range?(value,ns)
        self.class.is_in_data_range?(value,ns)
      end

      def is_in_object_range?(klass,ns)
        self.class.is_in_object_range?(klass,ns)
      end

      def is_in_domain?(value,ns)
        self.class.is_in_domain?(value,ns)
      end

      def store(changed_thing_names,ontology,ns)
        if @object.nil?() || @subject.nil?() then
          error "subject or object are missing; cannnot store property '#{self.class.to_s}'"
        else
          label=nil
          timestamp=nil
          if self.class.options[:allows_annotation]
            label=self.annotation() || ""
            timestamp=self.timestamp() || ""
          end
          if self.class.type==:object_property then
            ontology.create_object_property(changed_thing_names,@subject[:name],self.class.ontology_name,@object[:name],@subject[:ns],ns,@object[:ns],label,timestamp)
          elsif self.class.type==:datatype_property then
            ontology.create_datatyp_property(@subject[:name],self.class.ontology_name,@object,@subject[:ns],ns,label,timestamp)
          elsif self.class.type==:annotation_property then
            ontology.create_annotation_property(@subject[:name],self.class.ontology_name,@object,@subject[:ns],ns,label,timestamp)
          else
            error "Unsupported property type; failed to store"
          end
        end
      rescue Exception => e
        error "Failed to store property #{self.class.to_s}", e
      rescue => e
        error "Failed to store property #{self.class.to_s}", e
      end
    end

    class BaseObjectProperty < BaseProperty
      class << self
        def init(type=:object_property,name=nil)
          super(type,name)
        end
        alias :"is_in_range?" :"is_in_object_range?"
      end
      self.init(:object_property,"BaseObjectProperty")

      def initialize()
        super()
      end

      def store(changed_thing_names,ontology,ns)
        super(changed_thing_names,ontology,ns)
      end

      alias :"is_in_range?" :"is_in_object_range?"
    end

    class BaseDatatypeProperty < BaseProperty
      class << self
        def init(type=:datatype_property,name=nil)
          super(type,name)
        end
        alias :"is_in_range?" :"is_in_data_range?"
      end
      self.init(:datatype_property,"BaseDatatypeProperty")

      def initialize()
        super()
      end

      def store(changed_thing_names,ontology,ns)
        super(ontology,ns)
      end

      alias :"is_in_range?" :"is_in_data_range?"
    end

    class BaseAnnotationProperty < BaseProperty
      class << self
        def init(type=:annotation_property,name=nil)
          super(type,name)
        end
        alias :"is_in_range?" :"is_in_data_range?"
      end
      self.init(:annotation_property,"BaseAnnotationProperty")

      def initialize()
        super()
      end

      def store(changed_thing_names,ontology,ns)
        super(ontology,ns)
      end

      alias :"is_in_range?" :"is_in_data_range?"
    end
  end
end
