# encoding: utf-8
# license: gpl3p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST
module Iof
  module Data
    class Handler < Ghun::Base::Module
      #TODP add support for config chosen ontologies, maybe in runner/collector/writer
      def initialize(ontology_name=:writer)
        super(Ghun::Log::Source::ONTOLOGY)
        @component_mutex=Monitor.new()
        @connection_mutex=Monitor.new()
        @ontology_mutex=Monitor.new()
        @components=Array.new()
        #component leaving properties (CLP) are properties that leave the blob build from individuals and properties describing one component
        #Exceptions are properties to trivial external individuals like networks or addresses and isConnectedTo properties as they get a special handling
        #An example for a CLP is the isPoePoweredBy property
        @components[0]=Hash.new() #components needed by other components and without CLP, e.g. core network components
        @components[1]=Hash.new() #components maybe needed by other components and with CLP to category 0, e.g. VoIP-Phones or access points or CDP detected devices
        @components[2]=Hash.new() #components maybe needed by other components and with CLP to category 0 or 1, e.g guessed switches
        @components[3]=Hash.new() #components not needed by other components and with CLP to category 0-2, e.g. guessed endpoints

        @neighbors=Array.new()
        @connections=Array.new()
        @ontology=Ghun::Base::Blackboard.ontology.init_ontology(ontology_name)
        #TODO extend on interfaces
        @changed_names=Hash.new()
        @changed_interface_names=Hash.new()
        @changed_thing_names=Hash.new()
        @known_connections=Hash.new()
        @static=Hash.new()
        @static[:vlan]=Hash.new
        @static[:mac]=Hash.new
        @static[:ipv4_address]=Hash.new
        @static[:ipv6_address]=Hash.new
        @static[:ipv4_network]=Hash.new
        @static[:ipv6_network]=Hash.new
        @static[:icmpv4]=Hash.new
        @static[:icmpv6]=Hash.new
        @static[:udp]=Hash.new
        @static[:tcp]=Hash.new
        @static_mutex=Mutex.new()
        @stuff=Array.new()
        @stuff_mutex=Mutex.new()
        @group=Hash.new()
        @group[:vsa]=Hash.new
        @group_mutex=Mutex.new()
        @member=Hash.new()
        @member[:vsa]=Hash.new()
        @member_mutex=Mutex.new()
        @location_mutex=Monitor.new()
        @locations=Hash.new()
        @location_tree=Hash.new()
        @location_name_to_label=Hash.new()
        @ad_mutex=Monitor.new()
        @ads=Hash.new()
        @ad_name_to_label=Hash.new()
        @root_ads=Array.new()
      end
      attr_reader :ontology
      attr_reader :changed_names, :changed_interface_names, :changed_thing_names

      def write_changes(restart_ontology=true)
        store_static()
        store_administrative_domains()
        store_locations()
        store_components()
        store_connections()
        store_groups()
        store_stuff()
        @ontology_mutex.synchronize {
          @ontology.disconnect() if @ontology.connected?()
          @ontology.connect() if restart_ontology && !@ontology.connected?()
        }
      end

      def declare_component_name_change(old,new)
        debug "Declared component name change from #{old} to #{new}"
        @changed_names[old.intern]=new unless @changed_names.include?(old.intern)
        @changed_thing_names[old.intern]=new unless @changed_thing_names.include?(old.intern)

      end

      def declare_interface_name_change(old,new)
        debug "Declared interface name change from #{old} to #{new}"
        @changed_interface_names[old.intern]=new unless @changed_interface_names.include?(old.intern)
        @changed_thing_names[old.intern]=new unless @changed_thing_names.include?(old.intern)
      end


      def get_static(type,*args)
        id=nil
        name=nil
        klass=nil
        values=Hash.new()
        case type
        when :mac then
          klass=Iof::Data::NO::MacAddress
          id=args[0].to_s.intern
          name="__mac_#{args[0].to_s}__"
          values[:no_mac_address_value]=args[0].to_s
        when :vlan then
          klass=Iof::Data::NO::EthernetVlan
          id=args[0].to_s.intern
          name="__vlan_#{args[0].to_s}__"
          values[:no_vlan_id]=args[0].to_s
        when :icmpv4 then
          klass=Iof::Data::NO::ICMPv4Type
          id=args[0].to_s.intern
          name="__icmpv4_#{args[0].to_s}__"
          values[:no_icmpv4_type_value]=args[0].to_s
        when :icmpv6 then
          klass=Iof::Data::NO::ICMPv6Type
          id=args[0].to_s.intern
          name="__icmpv6_#{args[0].to_s}__"
          values[:no_icmpv6_type_value]=args[0].to_s
        when :udp then
          klass=Iof::Data::NO::UDPPort
          id=args[0].to_s.intern
          name="__udp_#{args[0].to_s}__"
          values[:no_udp_port_value]=args[0].to_s
        when :tcp then
          klass=Iof::Data::NO::TCPPort
          id=args[0].to_s.intern
          name="__tcp_#{args[0].to_s}__"
          values[:no_tcp_port_value]=args[0].to_s
        when :ipv4_address then
          klass=Iof::Data::NO::IPv4Address
          values[:no_ipv4_address_value]=args[0].to_s
          if args.size==1 then
            id=args[0].to_s.intern
            name="__ipv4_address_#{args[0].to_s}__"
          else
            netmask=Iof::Data.ipv4_parse_netmask(args[1])
            length=Iof::Data.ipv4_netmask_to_length(netmask)
            id="#{args[0].to_s}/#{length}".intern
            name="__ipv4_address_#{args[0].to_s}/#{length}__"
            values[:no_ipv4_netmask_value]=netmask
            values[:no_ipv4_netmask_length]=length
          end
        when :ipv4_network then
          klass=Iof::Data::NO::IPv4Network
          values[:no_ipv4_address_value]=args[0].to_s
          if args.size==1 then
            id=args[0].to_s.intern
            name="__ipv4_network_#{args[0].to_s}__"
          else
            netmask=Iof::Data.ipv4_parse_netmask(args[1])
            length=Iof::Data.ipv4_netmask_to_length(netmask)
            id="#{args[0].to_s}/#{length}".intern
            name="__ipv4_network_#{args[0].to_s}/#{length}__"
            values[:no_ipv4_netmask_value]=netmask
            values[:no_ipv4_netmask_length]=length
          end
        when :ipv6_address then
          klass=Iof::Data::NO::IPv6Address
          values[:no_ipv6_address_value]=args[0].to_s
          if args.size==1 then
            id=args[0].to_s.intern
            name="__ipv6_address_#{args[0].to_s}__"
          else
            id="#{args[0].to_s}/#{args[1].to_s}".intern
            name="__ipv6_address_#{args[0].to_s}/#{args[1]}__"
            values[:no_ipv6_netmask_length]=args[1].to_s
          end
        when :ipv6_network then
          klass=Iof::Data::NO::IPv6Network
          values[:no_ipv6_address_value]=args[0].to_s
          if args.size==1 then
            id=args[0].to_s.intern
            name="__ipv6_network_#{args[0].to_s}__"
          else
            id="#{args[0].to_s}/#{args[1].to_s}".intern
            name="__ipv6_network_#{args[0].to_s}/#{args[1]}__"
            values[:no_ipv6_netmask_length]=args[1].to_s
          end
        end
        static=nil
        @static_mutex.synchronize {
          if @static[type].include?(id) then
            static=@static[type][id]
          else
            static=klass.new()
            static.name=name
            values.each do |key,value|
              static[key]=value
            end
            @static[type][id]=static
          end
        }
        return static
      rescue => e
        error "Failed to get individual for type #{type.to_s} and arguments #{args.inspect}", e
        return nil
      end

      def get_static_name(type,*args)
        static=get_static(type,*args)
        if static then
          return static.name
        else
          return nil
        end
      end

      def get_mac_address_name(mac)
        get_static_name(:mac,Iof::Data.mac_plain(mac))
      end

      def get_vlan_name(vlan_id)
        get_static_name(:vlan,vlan_id.to_s)
      end

      def get_udp_port_name(port)
        get_static_name(:udp,port.to_s)
      end

      def get_tcp_port_name(port)
        get_static_name(:tcp,port.to_s)
      end

      def get_icmpv4_type_name(type)
        get_static_name(:icmpv4,type.to_s)
      end

      def get_icmpv6_type_name(type)
        get_static_name(:icmpv6,type.to_s)
      end

      def get_ipv4_address_name(address,netmask=nil)
        if netmask.nil?() then
          get_static_name(:ipv4_address,address)
        else
          get_static_name(:ipv4_address,address,netmask)
        end
      end

      def get_ipv4_network_name(address,netmask)
        get_static_name(:ipv4_network,address,netmask)
      end

      def get_ipv6_address_name(address,netmask=nil)
        if netmask.nil?() then
          get_static_name(:ipv6_address,Iof::Data.ipv6_address_uncompressed(address))
        else
          get_static_name(:ipv6_address,Iof::Data.ipv6_address_uncompressed(address),netmask)
        end
      end

      def get_ipv6_network_name(address,netmask)
        get_static_name(:ipv6_network,Iof::Data.ipv6_address_uncompressed(address),netmask)
      end

      def get_group_name(type,*args)
        group=get_group(type,*args)
        if group then
          return group.name
        else
          return nil
        end
      end

      def get_group(type,*args)
        id=nil
        name=nil
        klass=nil
        values=Hash.new()
        case type
        when :vsa then
          klass=Iof::Data::NO::VSA
          id=args[0].to_s.intern
          name="__vsa_#{args[0].to_s}__"
        end
        group=nil
        @group_mutex.synchronize {
          if @group[type].include?(id) then
            group=@group[type][id]
          else
            group=klass.new()
            group.name=name
            values.each do |key,value|
              group[key]=value
            end
            @group[type][id]=group
            @member_mutex.synchronize {
              @member[type][id]=Hash.new()
            }
          end
        }
        return group
      rescue => e
        error "Failed to get individual for type #{type.to_s} and arguments #{args.inspect}", e
        return nil
      end

      def set_group_member(type,group_name,member_name)
        @member_mutex.synchronize {
          unless @member[type][group_name.to_s.intern].include?(member_name.intern) then
            membership=Iof::Data::NO::HasVSAMember.new()
            membership.subject=get_group_name(type, group_name)
            membership.object=member_name
            @member[type][group_name.to_s.intern][member_name.intern]=membership
          end
        }
      end

      def get_location_name(location)
        @location_mutex.synchronize {
          types=[]
          types << :building unless location[:building].nil?()
          types << :floor unless location[:floor].nil?()
          types << :door unless location[:door].nil?()
          types << :id unless location[:id].nil?()
          super_location=nil
          super_location_label=nil
          super_location_tree=@location_tree
          new_name=nil
          types.each do |type|
            if super_location_tree.include?(location["#{type}_label".intern].intern)
              super_location_label=super_location_tree[location["#{type}_label".intern].intern][:__self__]
              super_location=@locations[super_location_label]
              super_location_tree=super_location_tree[location["#{type}_label".intern].intern]
            elsif !location[type].nil?()
              super_location_tree[location["#{type}_label".intern].intern]=Hash.new()
              new_location=nil
              property=nil
              case type
              when :building
                new_location=Iof::Data::NO::Building.new()
                new_location[:no_building_short_name]=location[:building_short_name]
                new_location[:no_building_long_name]=location[:building_long_name]
                new_name="__location_#{location[type].to_s}"
              when :floor
                new_location=Iof::Data::NO::Floor.new()
                new_location[:no_floor_number]=location[:floor_number]
                new_name="#{super_location.name.to_s}___#{location[type].to_s}"
                property=:no_floor
              when :door
                new_location=Iof::Data::NO::Door.new()
                new_location[:no_door_number]=location[:door_number]
                new_name="#{super_location.name.to_s}___#{location[type].to_s}"
                property=:no_door
              when :id
                new_location=Iof::Data::NO::LocationId.new()
                new_location[:no_location_id_value]=location[:id_value]
                new_location[:no_location_id_comment]=location[:id_comment]
                new_name="#{super_location.name.to_s}___#{location[type].to_s}"
                property=:no_location_id
              end
              location[:administrative_domain_name].each do |ad|
                new_location[:no_assigned_administrative_domain]=ad
              end if location.include?(:administrative_domain_name)
              if type==types[-1]
                new_location[:no_location_internal_label]=location[:label]
              else
                new_location[:no_location_internal_label]=location["#{type}_label".intern]
              end
              new_location.name=new_name.clone()
              @location_name_to_label[new_location.name.intern]=new_location[:no_location_internal_label].intern
              super_location_tree[location["#{type}_label".intern].intern][:__self__]=new_location[:no_location_internal_label].intern
              @locations[super_location_tree[location["#{type}_label".intern].intern][:__self__]]=new_location
              unless super_location.nil?()
                super_location[property]=new_location.name
              end
              super_location_label=super_location_tree[location["#{type}_label".intern].intern][:__self__]
              super_location=@locations[super_location_label]
              super_location_tree=super_location_tree[location["#{type}_label".intern].intern]
            end
          end
          return super_location.name.to_s
        }
      end



      def get_location(location)
        @location_mutex.synchronize {
          if location.is_a?(Hash)
            name=get_location_name(location)
            return @locations[@location_name_to_label[name.intern]]
          elsif location.is_a?(String) || location.is_a?(Symbol)
            if @locations.include?(location.intern)
              return @ads[location.intern]
            elsif @location_name_to_label.include?(location.intern) && @locations.include?(@location_name_to_label[location.intern])
             return @locations[@location_name_to_label[location.intern]][:class]
            end
          end
          return nil
        }
      end

      def get_administrative_domain_name(ad)
        @ad_mutex.synchronize {
          if @ads.include?(ad[:label])
            return @ads[ad[:label]][:class].name
          else
            new_ad=Iof::Data::NO::AdministrativeDomain.new()
            new_ad.name="__ad_#{ad[:label]}"
            new_ad[:no_administrative_domain_short_name]=ad[:short_name]
            new_ad[:no_administrative_domain_long_name]=ad[:long_name]
            new_ad[:no_administrative_domain_internal_label]=ad[:label]
            @ads[ad[:label]]={:subs => Array.new(), :class => new_ad}
            @ad_name_to_label[new_ad.name.intern]=ad[:label]
            if ad[:super_label].nil?() || ( !ad[:super_label].nil?() && !@ads.include?(ad[:super_label]))
              warning "Super AD #{ad[:super_label]} is unknown, creating #{ad[:label]} as root AD" if ( !ad[:super_label].nil?() && !@ads.include?(ad[:super_label]))
              @root_ads << ad[:label]
            elsif @ads.include?(ad[:super_label])
              @ads[ad[:super_label]][:subs] << ad[:label]
              @ads[ad[:super_label]][:class][:no_sub_administrative_domain]=new_ad.name
            end
            return new_ad.name
          end
        }
      end

      def get_administrative_domain(ad)
        @ad_mutex.synchronize {
          if ad.is_a?(Hash)
            name=get_administrative_domain_name(ad)
            return @ads[@ad_name_to_label[name.intern]][:class]
          elsif ad.is_a?(String) || ad.is_a?(Symbol)
            if @ads.include?(ad.intern)
              return @ads[ad.intern][:class]
            elsif @ad_name_to_label.include?(ad.intern) && @ads.include?(@ad_name_to_label[ad.intern])
             return @ads[@ad_name_to_label[ad.intern]][:class]
            end
          end
          return nil
        }
      end


      def set_component(component,category)
        @component_mutex.synchronize {
          if @components[category][component.name.intern].nil? then
            @components[category][component.name.intern]=component
          else
            warn "Overwriting unsafed state for component '#{component.name}' in category #{category}"
            @components[category][component.name.intern]=component
          end
        }
      end

      def set_connection(connection)
        @connection_mutex.synchronize {
          @connections << connection
        }
      end

      def set_stuff(stuff)
        @stuff_mutex.synchronize {
          @stuff << stuff
        }
      end

      private

      def store_locations()
        @location_mutex.synchronize {
          @location_tree.each do |key,value|
            store_locations_helper(value)
          end
        }
      end

      def store_locations_helper(subtree)
        subtree.each do |key,value|
          next if key==:__self__
          store_locations_helper(value)
        end
        location=@locations[subtree[:__self__]]
        location.store(@changed_thing_names,@ontology,nil,:no,nil,nil,[:no_location_internal_label])
      end

      def store_administrative_domains()
        @ad_mutex.synchronize {
          @root_ads.each do |ad|
            store_administrative_domains_helper(@ads[ad])
          end
        }
      end

      def store_administrative_domains_helper(subtree)
        subtree[:subs].each do |ad|
          store_administrative_domains_helper(@ads[ad])
        end
        ad=subtree[:class]
        ad.store(@changed_thing_names,@ontology,nil,:no)
      end


      def store_static()
        @static_mutex.synchronize {
          @ontology_mutex.synchronize {
            @static.each do |type,data|
              debug "Storing #{type.to_s}"
              data.each do |id,d|
                next if d.nil?()
                d.store(@changed_thing_names,@ontology,nil,:no)
              end
            end
          }
        }
      end

      def store_stuff()
        @stuff_mutex.synchronize {
          @ontology_mutex.synchronize {
            @stuff.each do |s|
              if s.is_a?(Iof::Data::BaseClass)
                debug "Storing instance of #{s.class.name}"
                s.store(@changed_thing_names,@ontology,nil,:no)
              elsif s.is_a?(Iof::Data::BaseProperty)
                debug "Storing instance of #{s.class.name}"
                s.store(@changed_thing_names,@ontology,:no)
              else
                debug "Ontology storage is not supported for class #{s.class}"
              end
            end
          }
        }
      end

      def store_components
        debug "Storing components"
        components=nil
        @component_mutex.synchronize {
          components=@components
          @components=[{},{},{},{}]
        }
        components.each_with_index do |c,category|
          debug "Storing components for category #{category}" unless c.empty?()
          info "Got no components for category #{category}" if c.empty?()
          c.each do |name,e|
            if e.nil?() then
              error "Got empty component"
              next
            end
            @ontology_mutex.synchronize {
              name=e.name
              stored=e.store(@changed_thing_names,@ontology,nil,:no,nil,nil,[:no_identifier],(category==3))
              unless name==stored[:name] then
                @changed_names[name.intern]=stored
                @changed_thing_names[name.intern]=stored
                debug "Name for component '#{name}' changed to '#{stored[:name]}'"
              end
              #drop all connections, store_connections will restore still existing ones
              drop_connections(stored,:no)
            }
          end
        end
      end

      def drop_connections(component,ns)
        url=Iof::Data.namespace_to_url(ns)
        result=@ontology.query("SELECT ?l1 WHERE { <#{url}#{component[:name]}> no:hasLayer1Interface ?l1 }")
        result.each do |line|
          @ontology.drop_object_property(line[:l1], "isConnectedTo", line[:__ns_l1], :no)
        end
      end

      def store_connections()
        debug "Storing connections"
        connections=nil
        @connection_mutex.synchronize {
          connections=@connections
          @connections=Array.new()
        }
        @ontology_mutex.synchronize {
          connections.each do |c|
            begin
              if c.is_a?(Iof::Data::NO::IsConnectedTo) then
                debug "Storing property #{c.class.to_s} with subject '#{c.subject}' and object '#{c.object}'"
                subject_component=c.subject[:name].sub(/___.*/,'')
                object_component=c.object[:name].sub(/___.*/,'')
                if @changed_interface_names.include?(c.subject[:name].intern)
                  debug "Changed subject from '#{c.subject[:name]}' to '#{@changed_interface_names[c.subject[:name].intern]}'"
                  c.subject=@changed_interface_names[c.subject[:name].intern]
                elsif @changed_thing_names.include?(c.subject[:name].intern)
                  debug "Changed subject from '#{c.subject[:name]}' to '#{@changed_thing_names[c.subject[:name].intern]}'"
                  c.subject=@changed_thing_names[c.subject[:name].intern]
                elsif @changed_names.include?(subject_component.intern)
                  new_subject=c.subject[:name].sub(/^#{subject_component}___/,"#{@changed_names[subject_component.intern]}___")
                  debug "Changed subject from '#{c.subject[:name]}' to '#{new_subject}'"
                  c.subject=new_subject
                end
                if @changed_interface_names.include?(c.object[:name].intern)
                  debug "Changed obect from '#{c.object[:name]}' to '#{@changed_interface_names[c.object[:name].intern]}'"
                  c.object=@changed_interface_names[c.object[:name].intern]
                elsif @changed_thing_names.include?(c.object[:name].intern)
                  debug "Changed subject from '#{c.object[:name]}' to '#{@changed_thing_names[c.object[:name].intern]}'"
                  c.object=@changed_thing_names[c.object[:name].intern]
                elsif @changed_names.include?(object_component.intern)
                  new_object=c.object[:name].sub(/^#{object_component}___/,"#{@changed_names[object_component.intern]}___")
                  debug "Changed object from '#{c.object[:name]}' to '#{new_object}'"
                  c.object=new_object
                end
                if (@known_connections.include?(c.subject[:name].intern) && @known_connections[c.subject[:name].intern].include?(c.object[:name].intern)) || (@known_connections.include?(c.object[:name].intern) && @known_connections[c.object[:name].intern].include?(c.subject[:name].intern)) then
                  debug "Connection already stored"
                else
                  c.store(@changed_thing_names,@ontology,:no)
                  @known_connections[c.subject[:name].intern]=Array.new() unless @known_connections.include?(c.subject[:name].intern)
                  @known_connections[c.subject[:name].intern] << c.object[:name].intern
                  @known_connections[c.object[:name].intern]=Array.new() unless @known_connections.include?(c.object[:name].intern)
                  @known_connections[c.object[:name].intern] << c.subject[:name].intern
                end
              else
                debug "Not storing non connection property #{c.class.to_s} with subject '#{c.subject}' and object '#{c.object}'"
              end
            rescue => e
              error "Failed to store property #{c.class.to_s} with subject '#{c.subject}' and object '#{c.object}'", e
            end
          end
        }
      end

      def store_groups()
        debug "Storing groups"
        @group_mutex.synchronize {
          @ontology_mutex.synchronize {
            @group.each do |type,data|
              debug "Storing #{type.to_s}"
              data.each do |id,d|
                next if d.nil?()
                d.store(@changed_thing_names,@ontology,nil,:no)
              end
            end
          }
        }

        debug "Storing group members"
        @member_mutex.synchronize {
          @ontology_mutex.synchronize {
            @member.each do |type,data|
              debug "Storing #{type.to_s} members"
              data.each do |id,d|
                d.each do |m_id,m|
                  next if m.nil?()
                  m.store(@changed_thing_names,@ontology,:no)
                end unless d.nil?()
              end
            end
          }
        }
      end
    end
  end
end
