# encoding: utf-8
# license: gpl3p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST

module Iof
  module Ontology
    #API calls are always in synchronous mode
    class SparUlBase < Iof::Ontology::Base
      def initialize(name,type)
        super(name,type)
      end

      def query(query,show_datatype_properties=true,show_anonymous_nodes=false,list_namespaces=[@default_ns])
        return execute_request({:query => query, :show_datatype_properties => show_datatype_properties, :show_anonymous_nodes=>show_anonymous_nodes,:list_namespaces=>list_namespaces,:mode=>:sync})
      end

      def enqueue_query(id,query,show_datatype_properties=true,show_anonymous_nodes=false,list_namespaces=[@default_ns])
        enqueue_request(id,{:query => query, :show_datatype_properties => show_datatype_properties, :show_anonymous_nodes=>show_anonymous_nodes,:list_namespaces=>list_namespaces,:mode=>:async})
      end


      def create_individual(klass,individual,klass_ns=@default_ns,individual_ns=@default_ns)
        q="INSERT DATA { #{individual_ns}:#{individual} a #{klass_ns}:#{klass} }"
        query(q)
        return { :name => individual, :ns => individual_ns.intern, :unknown => false }
      rescue => e
        error "Failed to create individual '#{individual_ns.to_s}:#{individual}' with class '#{klass_ns.to_s}:#{klass}'", e
        return nil
      end

      def exists_individual?(individual,individual_ns=@default_ns)
        q="SELECT DISTINCT * WHERE { #{individual_ns.to_s}:#{individual} rdf:type ?t}"
        test=query(q)
        return test.size()>1
      rescue => e
        error "Failed to check individual '#{individual_ns.to_s}:#{individual}' existence", e
        return nil
      end

      def drop_individual(individual,individual_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        i.remove()
        return true
      rescue => e
        error "Could not remove individual '#{individual_ns.to_s}:#{individual}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def has_datatype_property?(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        p=get_jena_datatype_property(property,property_ns.intern)
        if p.nil?() then
          error "Datatype property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        return i.has_property(p)
      rescue => e
        error "Could not determine existence of datatype property '#{property_ns.to_s}:#{property}' of individual '#{individual_ns.to_s}:#{individual}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def has_object_property?(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        p=get_jena_object_property(property,property_ns.intern)
        if p.nil?() then
          error "Object property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        return i.has_property?(p)
      rescue => e
        error "Could not determine existence of object property '#{property_ns.to_s}:#{property}' of individual '#{individual_ns.to_s}:#{individual}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def get_object_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns,list_namespaces=[@default_ns],show_unknown_nodes=false)
        list_namespaces=@prefixes.keys if list_namespaces.nil?() || list_namespaces.empty?()
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        result=Array.new()
        p=get_jena_object_property(property,property_ns.intern)
        if p.nil?() then
          error "Object property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        values=i.list_property_values(p)
        values.each do |value|
          found=false
          @prefixes.each do |prefix,url|
            if value.to_string.match(/^#{url}/) then
              v=value.to_string
              found=true
              if list_namespaces.include?(prefix) then
                v.gsub!(/^#{url}/,'')
                r=Hash.new()
                r[:name]=v
                r[:ns]=prefix
                r[:unknown]=false
                result << r
              else
                debug "Value #{value.to_string} matched by namespace filter"
              end
            end
          end
          if show_unknown_nodes then
            debug "Adding result '#{value.to_string}' with unknown namespace to result"
            r=Hash.new()
            r[:name]=value.to_string
            r[:unknown]=true
            result << r
          else
            debug "Result '#{value.to_string}' contains unknown namespace"
          end unless found
        end unless values.nil?
        return result
      rescue => e
        error "Could not determine value(s) of property '#{property_ns.to_s}:#{property}' of individual '#{individual_ns.to_s}:#{individual}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def get_datatype_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        result=Array.new()
        p=get_jena_datatype_property(property.intern,property_ns)
        if p.nil?() then
          error "Datatype property '#{property_ns.to_s}#{property}' is unknown"
          return nil
        end
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        values=i.list_property_values(p)
        values.each do |v|
          result << v.get_string
        end unless values.nil?
        return result
      rescue => e
        error "Could not determine value(s) of property '#{property_ns.to_s}:#{property}' of individual '#{individual_ns.to_s}:#{individual}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def drop_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        prefix=nil
        if property_ns.is_a?(Symbol) then
          prefix=property_ns.intern
        else
          prefix=url_to_prefix(property_ns)
        end
        p=nil
        p=get_jena_datatype_property(property,property_ns.intern) if @datatype_properties.include?(prefix) && @datatype_properties[prefix].include?(property)
        p=get_jena_object_property(property,property_ns.intern) if @object_properties.include?(prefix) && @object_properties[prefix].include?(property)
        if p.nil?() then
          error "Property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i.remove_all(p)
        return true
      rescue => e
        error "Could not remove property '#{property_ns.to_s}:#{property}' for individual '#{individual_ns.to_s}:#{individual}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def drop_object_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        p=get_jena_object_property(property,property_ns.intern)
        if p.nil?() then
          error "Object property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i.remove_all(p)
        return true
      rescue => e
        error "Could not remove property '#{property_ns.to_s}:#{property}' for individual '#{individual_ns.to_s}:#{individual}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def drop_datatype_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        p=get_jena_datatype_property(property,property_ns.intern)
        if p.nil?() then
          error "Datatype property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i.remove_all(p)
        return true
      rescue => e
        error "Could not remove property '#{property_ns.to_s}:#{property}' for individual '#{individual_ns.to_s}:#{individual}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def create_object_property(individual,property,target,individual_ns=@default_ns,property_ns=@default_ns,target_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        p=get_jena_object_property(property,property_ns.intern)
        t=get_jena_individual(target,target_ns.intern)
        error "Failed to retrieve jena object for individual '#{individual_ns.to_s}:#{individual}'" if i.nil?()
        error "Failed to retrieve jena object for property '#{property_ns.to_s}:#{property}'" if p.nil?()
        error "Failed to retrieve jena object for individual '#{target_ns.to_s}:#{target}'" if t.nil?()
        if i.nil?() || p.nil?() || t.nil?()
          return false
        else
          s=@model.createStatement(i,p,t)
          @model.add(s)
          return true
        end
      rescue => e
        error "Failed to create object property '#{property_ns.to_s}:#{property}' between '#{individual_ns.to_s}:#{individual}' and '#{target_ns.to_s}:#{target}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def create_datatype_property(individual,property,value,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        p=get_jena_datatype_property(property,property_ns.intern)
        error "Failed to retrieve jena object for individual '#{individual_ns.to_s}:#{individual}'" if i.nil?()
        error "Failed to retrieve jena object for property '#{property_ns.to_s}:#{property}'" if p.nil?()
        if i.nil?() || p.nil?()
          return false
        else
          s=@model.createStatement(i,p,value.to_s)
          @model.add(s)
          return true
        end
      rescue => e
        error "Failed to create datatype property '#{property_ns.to_s}:#{property}' for '#{individual_ns.to_s}:#{individual}' with value '#{value}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

    private

      def execute_helper(args)
        list_namespaces=args[:list_namespaces]
        list_namespaces=@prefixes.keys if list_namespaces.nil?() || list_namespaces.empty?()
        query=build_query(args[:query])
        if query.match(/^[ \t]*INSERT/i) || query.match(/^[ \t]*DELETE/i)
          #update query
          raise "Blocked SPARQL/Update query #{query} in read_only mode" if @read_only
          debug "Executing SPARQL/Update query '#{query}', with namespace filter '#{list_namespaces.inspect}'"
          result=execute_sparul_helper(query,args[:mode])
          debug "Query finished"
          return result
        elsif query.match(/^[ \t]*SELECT/i)
          debug "Executing SPARQL query '#{query}', with namespace filter '#{list_namespaces.inspect}'"
          result=execute_sparql_helper(query,args[:mode],args[:show_anonymous_nodes],args[:show_datatype_properties],list_namespaces)
          debug "Query finished"
          return result
        else
          raise "Detected blocked query type: #{query}"
        end
      rescue => e
        error "Failed to execute query: '#{query}'", e
        raise
      end

      def execute_sparql_helper(query,mode,show_datatype_properties,show_anonymous_nodes,list_namespaces)
        raise AbstractClassError, "Implement execute_sparql_helper in #{self.class}"
      end

      def execute_sparul_helper(query,mode)
        raise AbstractClassError, "Implement execute_sparul_helper in #{self.class}"
      end
    end
  end
end
