# encoding: utf-8
# license: gpl3p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST

module Iof
  module Ontology
    #This jena interface is a rework of the original IO Jena interface
    #Jena API calls will work only in sync mode
    #a SPARQL/SPARUL based implementation will be implemented in the future
    class SparUlJena < Iof::Ontology::Base
      Iof::Ontology::Handler::ENGINE_SELECTOR.register_member(self,:sparul_jena)
      def initialize(name)
        super(name,:jena)
        # should not contain any data and is used to retrieve the ontology structure
        # this file is also used to initialize TDB storages or be read in the case of missing data file
        config.declare_dynamic_key(:"ontology.#{@name.to_s}.structure_file",:string,File.join(Ghun::Base::Blackboard.application.data_path,'ontologies','network.owl'))
        # file or tdb
        config.declare_dynamic_key(:"ontology.#{@name.to_s}.storage_backend",:string,"file")
        #for backend 'file'
        config.declare_dynamic_key(:"ontology.#{@name.to_s}.storage_file",:string,File.join(Ghun::Base::Blackboard.local.data_path,'ontologies','network.owl'),false)
        #for backend 'tdb'
        config.declare_dynamic_key(:"ontology.#{@name.to_s}.storage_directory",:string,File.join(Ghun::Base::Blackboard.local.data_path,'ontologies','network'),false)
        config.declare_dynamic_key(:"ontology.#{@name.to_s}.version_after_write",:bool,true)
        config.declare_dynamic_key(:"ontology.#{@name.to_s}.start_with_empty_ontology",:bool,false)
        #startup types
        # full - the ontology populated with data will be loaded and reasoned
        # mapping - only the structure file will be read and reasoned
        # fastwrite - the ontology populated with data will be loaded the reasoner is not active
        # it is possible to change from full to fastwrite during operation
        config.declare_dynamic_key(:"ontology.#{@name.to_s}.startup_type",:string,"full")

        @structure_file=config[:"ontology.#{@name.to_s}.structure_file"]
        @storage_backend=config[:"ontology.#{@name.to_s}.storage_backend"].intern
        @storage_file=config[:"ontology.#{@name.to_s}.storage_file"]
        @storage_directory=config[:"ontology.#{@name.to_s}.storage_directory"]
        @version_after_write=config[:"ontology.#{@name.to_s}.version_after_write"]
        @start_with_empty_ontology=config[:"ontology.#{@name.to_s}.start_with_empty_ontology"]
        @startup_type=config[:"ontology.#{@name.to_s}.startup_type"].intern

        if @start_with_empty_ontology && @readonly
          warn "start_with_empty_ontology and read_only is active for ontology #{@name}", "start_with_empty_ontology tries to delete existing data, read_only tries to protect themwill prevent this", "Prefering read_only"
          @start_with_empty_ontology=false
        end

        raise "Unknown startup type #{@startup_type}" if @startup_type!=:mapper && @startup_type!=:full && @startup_type!=:fastwrite
        unless File.exists?(@structure_file)
          raise "Ontology structure file #{@structure_file} does not exist"
        end

        #file handling is easier in read only case
        if @readonly && @storage_backend==:tdb
          @storage_backend=:file
          @storage_file=@structure_file
        end

        if @startup_type==:mapper
          @readonly=true
          @storage_file=@structure_file
          @storage_backend=:file
        else
          if @storage_backend==:file && (@storage_file.nil?() || @storage_file.empty?())
            raise "Storage backend 'file' requested but no file name given"
          elsif @storage_backend==:tdb  && (@storage_directory.nil?() || @storage_directory.empty?())
            raise "Storage backend 'tdb' requested but no directory name given"
          end
          raise "Unknown backend #{@storage_backend}" if @storage_backend!=:file && @storage_backend!=:tdb
        end

        options=Java::Util::Properties.new()
        options.set_property("USE_UNIQUE_NAME_ASSUMPTION","true")
        options.set_property("REALIZE_INDIVIDUAL_AT_A_TIME","true")
        options.set_property("USE_ANNOTATION_SUPPORT","true")
        options.set_property("HIDE_TOP_PROPERTY_VALUES","false")
        Pellet::PelletOptions.set_options(options)

        @model=nil
        @datatype_properties=nil
        @object_properties=nil
        @classes=nil

        @exec_mutex=Monitor.new()
        @sync_query_exec=nil
        @async_query_exec=nil
        @async_query_id=nil
        self.allow_parallel_execution=true
      end
      attr_reader :datatype_properties, :object_properties, :classes, :model

      def query(query,show_datatype_properties=true,show_anonymous_nodes=false,list_namespaces=[@default_ns])
        return execute_request({:query => query, :show_datatype_properties => show_datatype_properties, :show_anonymous_nodes=>show_anonymous_nodes,:list_namespaces=>list_namespaces,:mode=>:sync})
      end

      def enqueue_query(id,query,show_datatype_properties=true,show_anonymous_nodes=false,list_namespaces=[@default_ns])
        enqueue_request(id,{:query => query, :show_datatype_properties => show_datatype_properties, :show_anonymous_nodes=>show_anonymous_nodes,:list_namespaces=>list_namespaces,:mode=>:async})
      end

      def abort_running_request(request_id)
        @exec_mutex.synchronize {
          if request_id.nil?()
            unless @sync_query_exec.nil?()
              warn "Trying to abort query running in sync mode"
              begin
                @sync_query_exec.abort()
              rescue => e
                error "Failed query abort", e
              end
            end
          elsif !request_id.nil?() && @async_query_id==request_id
            unless @async_query_exec.nil?()
              warn "Trying to abort query running async mode"
              begin
                @async_query_exec.abort()
                @state_mutex.synchronize {
                  return @request_state[request_id.intern]=:aborted if @request_state.include?(request_id.intern)
                }
              rescue => e
                error "Failed query abort", e
              end
            end
          end
        }
      end
      alias :abort_query :abort_request

      def create_individual(klass,individual,klass_ns=@default_ns,individual_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        c=get_jena_class(klass,klass_ns.intern)
        @model.create_individual("#{@prefixes[individual_ns.intern]}#{individual}",c)
        return { :name => individual, :ns => individual_ns.intern, :unknown => false }
      rescue => e
        error "Failed to create individual '#{individual_ns.to_s}:#{individual}' with class '#{klass_ns.to_s}:#{klass}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def exists_individual?(individual,individual_ns=@default_ns)
        !get_jena_individual(individual,individual_ns.intern).nil?()
      end

      def drop_individual(individual,individual_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        i.remove()
        return true
      rescue => e
        error "Could not remove individual '#{individual_ns.to_s}:#{individual}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def has_datatype_property?(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        p=get_jena_datatype_property(property,property_ns.intern)
        if p.nil?() then
          error "Datatype property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        return i.has_property(p)
      rescue => e
        error "Could not determine existence of datatype property '#{property_ns.to_s}:#{property}' of individual '#{individual_ns.to_s}:#{individual}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def has_object_property?(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        p=get_jena_object_property(property,property_ns.intern)
        if p.nil?() then
          error "Object property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        return i.has_property?(p)
      rescue => e
        error "Could not determine existence of object property '#{property_ns.to_s}:#{property}' of individual '#{individual_ns.to_s}:#{individual}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def get_all_datatype_properties(individual,blacklist=[],individual_ns=@default_ns,list_namespaces=[@default_ns],show_unknown_nodes=false)
        list_namespaces=@prefixes.keys if list_namespaces.nil?() || list_namespaces.empty?()
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        properties=Hash.new()
        statements=i.list_properties.to_a
        statements.each do |s|
          p=s.get_predicate.to_s
          next if p.nil?()
          found=false
          type_match=false
          property_prefix=nil
          @prefixes.each do |prefix,url|
            if p.match(/^#{url}/) then
              found=true
              property_prefix=prefix
              if list_namespaces.include?(prefix) && @data_properties[prefix].include?(p.intern) && !(blacklist.include?(prefix) && blacklist[prefix].include?(p.intern)) then
                type_match=true
                p.gsub!(/^#{url}/,'')
                unless properties.include?("#{prefix}:#{p}".intern) then
                  properties["#{prefix}:#{p}".intern]=Hash.new
                  properties["#{prefix}:#{p}".intern][:name]=p
                  properties["#{prefix}:#{p}".intern][:ns]=prefix
                  properties["#{prefix}:#{p}".intern][:unknown]=false
                  properties["#{prefix}:#{p}".intern][:values]=Array.new()
                end
              else
                debug "Property '#{prefix}:#{p}' matched by namespace filter" unless list_namespaces.include?(prefix)
                debug "Property '#{prefix}:#{p}' is not an data property" unless @data_properties[prefix].include?(p.intern)
                debug "Property '#{p}' for individual '#{individual_ns.to_s}:#{individual}' is blacklisted by caller" if blacklist.include?(prefix) && blacklist[prefix].include?(p.intern)
              end
            end
          end
          if show_unknown_nodes then
            debug "Adding property '#{p}' with unknown namespace to result"
            properties["#{p}".intern]=Hash.new
            properties["#{p}".intern][:property]=p
            properties["#{p}".intern][:unknown]=true
            properties["#{p}".intern][:values]=Array.new()
          else
            debug "Property '#{p}' contains unknown namespace"
          end unless found
          if type_match then
            value=s.get_literal.to_string
            properties["#{property_prefix}:#{p}".intern][:values] << { :value => value }
          end
        end unless statements.nil?()
        return properties
      rescue => e
        error "Could not retrieve list of all object properties for individual '#{individual_ns.to_s}:#{individual}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def get_all_object_properties(individual,blacklist=[],individual_ns=@default_ns,list_namespaces=[@default_ns],show_unknown_nodes=false)
        list_namespaces=@prefixes.keys if list_namespaces.nil?() || list_namespaces.empty?()
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        properties=Hash.new()
        statements=i.list_properties.to_a
        statements.each do |s|
          p=s.get_predicate.to_s
          next if p.nil?()
          found=false
          type_match=false
          property_prefix=nil
          @prefixes.each do |prefix,url|
            if p.match(/^#{url}/) then
              found=true
              property_prefix=prefix
              if list_namespaces.include?(prefix) && @object_properties[prefix].include?(p.intern) && !(blacklist.include?(prefix) && blacklist[prefix].include?(p.intern)) then
                type_match=true
                p.gsub!(/^#{url}/,'')
                unless properties.include?("#{prefix}:#{p}".intern) then
                  properties["#{prefix}:#{p}".intern]=Hash.new
                  properties["#{prefix}:#{p}".intern][:property]=p
                  properties["#{prefix}:#{p}".intern][:ns]=prefix
                  properties["#{prefix}:#{p}".intern][:unknown]=false
                  properties["#{prefix}:#{p}".intern][:values]=Array.new()
                end
              else
                debug "Property '#{prefix}:#{p}' matched by namespace filter" unless list_namespaces.include?(prefix)
                debug "Property '#{prefix}:#{p}' is not an object property" unless @object_properties[prefix].include?(p.intern)
                debug "Property '#{p}' for individual '#{individual_ns.to_s}:#{individual}' is blacklisted by caller" unless blacklist.include?(prefix) && blacklist[prefix].include?(p.intern)
              end
            end
          end
          if show_unknown_nodes then
            debug "Adding property '#{p}' with unknown namespace to result"
            properties["#{p}".intern]=Hash.new
            properties["#{p}".intern][:property]=p
            properties["#{p}".intern][:unknown]=true
            properties["#{p}".intern][:values]=Array.new()
          else
            debug "Property '#{p}' contains unknown namespace"
          end unless found
          if type_match then
            found=false
            value=s.get_resource.to_string
            @prefixes.each do |prefix,url|
              if value.match(/^#{url}/) then
                found=true
                if list_namespaces.include?(prefix) then
                  value.gsub!(/^#{url}/,'')
                  properties["#{property_prefix}:#{p}".intern][:values] << { :name => value, :ns => prefix, :unknown => false }
                else
                  debug "Value '#{prefix}:#{value}' matched by namespace filter"
                end
              end
            end
            if show_unknown_nodes then
              debug "Adding result '#{value}' with unknown namespace to result"
              if property_prefix.nil?() then
                properties[p.intern] << value
              else
                properties["#{property_prefix}:#{p}".intern][:values] << { :name => value, :unknown => true }
              end
            else
              debug "Result '#{value}' contains unknown namespace"
            end unless found
          end
        end unless statements.nil?()
        return properties
      rescue => e
        error "Could not retrieve list of all object properties for individual '#{individual_ns.to_s}:#{individual}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def drop_all_properties(individual,individual_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        i.clear_properties
        return true
      rescue => e
        error "Could not remove all properties for individual '#{individual_ns.to_s}:#{individual}'",e
        return false
      ensure
        @model.leave_critical_section()
      end

      def get_object_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns,list_namespaces=[@default_ns],show_unknown_nodes=false)
        list_namespaces=@prefixes.keys if list_namespaces.nil?() || list_namespaces.empty?()
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        result=Array.new()
        p=get_jena_object_property(property,property_ns.intern)
        if p.nil?() then
          error "Object property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        values=i.list_property_values(p)
        values.each do |value|
          found=false
          @prefixes.each do |prefix,url|
            if value.to_string.match(/^#{url}/) then
              v=value.to_string
              found=true
              if list_namespaces.include?(prefix) then
                v.gsub!(/^#{url}/,'')
                r=Hash.new()
                r[:name]=v
                r[:ns]=prefix
                r[:unknown]=false
                result << r
              else
                debug "Value #{value.to_string} matched by namespace filter"
              end
            end
          end
          if show_unknown_nodes then
            debug "Adding result '#{value.to_string}' with unknown namespace to result"
            r=Hash.new()
            r[:name]=value.to_string
            r[:unknown]=true
            result << r
          else
            debug "Result '#{value.to_string}' contains unknown namespace"
          end unless found
        end unless values.nil?
        return result
      rescue => e
        error "Could not determine value(s) of property '#{property_ns.to_s}:#{property}' of individual '#{individual_ns.to_s}:#{individual}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def get_datatype_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        result=Array.new()
        p=get_jena_datatype_property(property.intern,property_ns)
        if p.nil?() then
          error "Datatype property '#{property_ns.to_s}#{property}' is unknown"
          return nil
        end
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        values=i.list_property_values(p)
        values.each do |v|
          result << v.get_string
        end unless values.nil?
        return result
      rescue => e
        error "Could not determine value(s) of property '#{property_ns.to_s}:#{property}' of individual '#{individual_ns.to_s}:#{individual}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def drop_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        prefix=nil
        if property_ns.is_a?(Symbol) then
          prefix=property_ns.intern
        else
          prefix=url_to_prefix(property_ns)
        end
        p=nil
        p=get_jena_datatype_property(property,property_ns.intern) if @datatype_properties.include?(prefix) && @datatype_properties[prefix].include?(property)
        p=get_jena_object_property(property,property_ns.intern) if @object_properties.include?(prefix) && @object_properties[prefix].include?(property)
        if p.nil?() then
          error "Property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i.remove_all(p)
        return true
      rescue => e
        error "Could not remove property '#{property_ns.to_s}:#{property}' for individual '#{individual_ns.to_s}:#{individual}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def drop_object_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        p=get_jena_object_property(property,property_ns.intern)
        if p.nil?() then
          error "Object property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i.remove_all(p)
        return true
      rescue => e
        error "Could not remove property '#{property_ns.to_s}:#{property}' for individual '#{individual_ns.to_s}:#{individual}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def drop_datatype_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        p=get_jena_datatype_property(property,property_ns.intern)
        if p.nil?() then
          error "Datatype property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i.remove_all(p)
        return true
      rescue => e
        error "Could not remove property '#{property_ns.to_s}:#{property}' for individual '#{individual_ns.to_s}:#{individual}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def create_object_property(individual,property,target,individual_ns=@default_ns,property_ns=@default_ns,target_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        p=get_jena_object_property(property,property_ns.intern)
        t=get_jena_individual(target,target_ns.intern)
        error "Failed to retrieve jena object for individual '#{individual_ns.to_s}:#{individual}'" if i.nil?()
        error "Failed to retrieve jena object for property '#{property_ns.to_s}:#{property}'" if p.nil?()
        error "Failed to retrieve jena object for individual '#{target_ns.to_s}:#{target}'" if t.nil?()
        if i.nil?() || p.nil?() || t.nil?()
          return false
        else
          s=@model.createStatement(i,p,t)
          @model.add(s)
          return true
        end
      rescue => e
        error "Failed to create object property '#{property_ns.to_s}:#{property}' between '#{individual_ns.to_s}:#{individual}' and '#{target_ns.to_s}:#{target}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def create_datatype_property(individual,property,value,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        p=get_jena_datatype_property(property,property_ns.intern)
        error "Failed to retrieve jena object for individual '#{individual_ns.to_s}:#{individual}'" if i.nil?()
        error "Failed to retrieve jena object for property '#{property_ns.to_s}:#{property}'" if p.nil?()
        if i.nil?() || p.nil?()
          return false
        else
          s=@model.createStatement(i,p,value.to_s)
          @model.add(s)
          return true
        end
      rescue => e
        error "Failed to create datatype property '#{property_ns.to_s}:#{property}' for '#{individual_ns.to_s}:#{individual}' with value '#{value}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

    private

      def connect_helper()
        if @storage_backend==:file
          prepare_ontology_file() unless @startup_type==:mapper
          load_ontology_file()
        elsif @storage_backend==:tdb
          prepare_ontology_tdb()
          load_ontology_tdb()
        end
        init_classes(@model)
        init_properties(@model)
        @connected=true
      end

      def disconnect_helper()
        if @storage_backend==:file
          write_ontology_file() unless @readonly
        elsif @storage_backend==:tdb
          write_ontology_tdb() unless @readonly
        end
        cleanup()
        @connected=false
      end

      def execute_helper(args)
        #FIXME detect SPARUL and block while @readonly
        list_namespaces=args[:list_namespaces]
        list_namespaces=@prefixes.keys if list_namespaces.nil?() || list_namespaces.empty?()
        query=build_query(args[:query])
        debug "Executing query '#{query}', with namespace filter '#{list_namespaces.inspect}'"
        result=jena_query(query,args[:mode],args[:show_anonymous_nodes],args[:show_datatype_properties],list_namespaces)
        debug "Query finished"
        return result
      rescue => e
        error "Failed to execute query: '#{query}'", e
        raise
      end

      def prepare_ontology_file()
        unless @readonly
          if @start_with_empty_ontology && File.exists?(@storage_file)
            info "Removing existing ontology #{@storage_file} as requested"
            File.delete(@storage_file)
          end
          unless Dir.exists?(File.dirname(@storage_file))
            debug "Creating directory #{File.dirname(@storage_file)}"
            FileUtils.mkdir_p(File.dirname(@storage_file))
          end
          unless File.writable?(File.dirname(@storage_file))
            raise "Directory #{File.dirname(@storage_file)} is not writable"
          end
          FileUtils.cp(@structure_file, @storage_file) unless File.exists?(@storage_file)
        end
      end

      def load_ontology_file()
        file=@structure_file
        file=@storage_file if File.exists?(@storage_file)

        if @startup_type==:mapper || @startup_type==:full
          @model=Jena::Rdf::Model::ModelFactory.createOntologyModel(Pellet::Jena::PelletReasonerFactory.THE_SPEC)
          @model.read("file://#{file}")
          debug "Preparing ontology"
          @model.prepare
          graph=@model.get_graph().get_kb()
          debug "Classifying ontology"
          graph.classify
          debug "Realizing ontology"
          graph.realize
          if @startup_type==:mapper
            #disconnect reasoner from ontology
            old_model=@model
            @model=Jena::Rdf::Model::ModelFactory.createOntologyModel(Jena::Ontology::OntModelSpec::OWL_MEM)
            @model.add(old_model)
          end
        else
          @model=Jena::Rdf::Model::ModelFactory.createOntologyModel(Jena::Ontology::OntModelSpec::OWL_MEM)
          @model.read("file://#{file}")
        end
      end

      def write_ontology_file()
        io=Java::IO::FileOutputStream.new(@storage_file)
        @model.write(io)
        io.close()
        @model.close
        #TODO handle versioning
        #FileUtils.cp(@storage_file,versioned_file_name)
      end

      def prepare_ontology_tdb()
        unless @readonly
          if @start_with_empty_ontology && Dir.exists?(@storage_directory)
            info "Removing existing ontology #{@storage_directory} as requested"
            FileUtils.rm_r(@storage_directory)
          end
          unless Dir.exists?(@storage_directory)
            debug "Creating directory #{@storage_directory}"
            FileUtils.mkdir_p(@storage_directory)
          end
          unless File.writable?(@storage_directory)
            raise "Directory #{@storage_directory} is not writable"
          end
          unless File.exists?(File.join(@storage_directory,'nodes.dat'))
            info "Creating empty TDB storage"
            tdb_dataset=Jena::Tdb::TdbFactory.create_dataset(@storage_directory)
            tdb_graph=tdb_dataset.get_default_model().get_graph()
            Jena::Tdb::TdbLoader.load(tdb_graph,[@structure_file],false)
            tdb_graph.close()
            tdb_dataset.close()
          end
        end
      end

      def load_ontology_tdb()
        #read_only and running with in mapper mode is handled by file mode methods only
        tdb_model=Jena::Tdb::TDBFactory.createModel(@storage_directory)
        if @startup_type==:full
          @model = Jena::Rdf::Model::ModelFactory.createOntologyModel(Pellet::Jena::PelletReasonerFactory.THE_SPEC, tdb_model);
          debug "Preparing ontology"
          @model.prepare
          graph=@model.get_graph().get_kb()
          debug "Classifying ontology"
          graph.classify
          debug "Realizing ontology"
          graph.realize
        else
          @model = Jena::Rdf::Model::ModelFactory.createOntologyModel(Jena::Ontology::OntModelSpec::OWL_MEM, tdb_model)
        end
      end

      def write_ontology_tdb()
        @model.commit()
        @model.close()
      end

      def cleanup()
        @model=nil
        @datatype_properties=nil
        @object_properties=nil
        @classes=nil
      end

      def jena_query(query,mode,show_datatype_properties,show_anonymous_nodes,list_namespaces)
        list_namespaces=@prefixes.keys if list_namespaces.nil?() || list_namespaces.empty?()
        results=Array.new()
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        if mode==:sync
          @sync_query_exec=Jena::Query::QueryExecutionFactory.create(Jena::Query::QueryFactory.create(query), @model)
          exec=@sync_query_exec
        else
          @async_query_exec=Jena::Query::QueryExecutionFactory.create(Jena::Query::QueryFactory.create(query), @model)
          exec=@async_query_exec
        end
        begin
          exec.execSelect.each do |raw|
            result=Hash.new()
            vars=raw.varNames.to_a
            vars.each do |v|
              r=raw.get(v)
              if r.nil?() then
                result[v.intern] = r
              else
                result[v.intern] = r.to_string
              end
            end
            results << result
          end
        rescue => e
          error "Failed to execute query: '#{query}'", e
          results=nil
          raise
        ensure
          exec.close
          if mode==:sync
            @sync_query_exec=nil
          else
            @async_query_exec=nil
            @async_query_id=nil
          end
        end
        return format_result(results,list_namespaces,show_datatype_properties,show_anonymous_nodes)
      rescue => e
        error "Failed to execute query: '#{query}'", e
        results=nil
        raise
      ensure
        @model.leave_critical_section()
      end


      def get_jena_individual(name,ns)
        prefix=nil
        if ns.is_a?(Symbol) then
          prefix=ns
        else
          prefix=url_to_prefix(ns)
        end
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        @model.get_individual("#{@prefixes[prefix]}#{name}")
      rescue => e
        error "Failed to retrieve individual", e
      ensure
        @model.leave_critical_section()
      end

      def get_jena_datatype_property(name,ns)
        prefix=nil
        if ns.is_a?(Symbol) then
          prefix=ns
        else
          prefix=url_to_prefix(ns)
        end
        if @datatype_properties.include?(prefix) && @datatype_properties[prefix].include?(name.intern) then
          return @datatype_properties[prefix][name.intern]
        else
          error "Datatype property '#{name}' with namespace #{ns} unknown in ontology"
          raise PropertyUnknownError
        end
      end

      def get_jena_object_property(name,ns)
        prefix=nil
        if ns.is_a?(Symbol) then
          prefix=ns
        else
          prefix=url_to_prefix(ns)
        end
        if @object_properties.include?(prefix) && @object_properties[prefix].include?(name.intern) then
          return @object_properties[prefix][name.intern]
        else
          error "Object property '#{name}' with namespace #{ns} unknown in ontology"
          raise PropertyUnknownError
        end
      end

      def get_jena_class(name,ns)
        prefix=nil
        if ns.is_a?(Symbol) then
          prefix=ns
        else
          prefix=url_to_prefix(ns)
        end
        if @classes.include?(prefix) && @classes[prefix].include?(name.intern) then
          return @classes[prefix][name.intern]
        else
          error "Class '#{name}' with namespace #{ns} unknown in ontology"
          raise ClassUnknownError, "Class '#{name}' with namespace #{ns} unknown in ontology"
        end
      end

      def init_classes(model)
        @classes=Hash.new()
        @prefixes.keys.each do |prefix|
          @classes[prefix]=Hash.new()
        end
        model.list_classes.each do |c|
          @prefixes.each do |prefix,url|
            if c.to_string.match(/^#{url}/) then
              debug "Found class '#{c.to_string}' with namespace '#{prefix.to_s}' in ontology"
              @classes[prefix][c.to_string.sub(/^#{url}/,"").intern]=c
              break
            end
          end
        end unless model.nil?
      end

      def init_properties(model)
        @datatype_properties=Hash.new()
        @object_properties=Hash.new
        @prefixes.keys.each do |prefix|
          @datatype_properties[prefix]=Hash.new()
          @object_properties[prefix]=Hash.new
        end
        model.list_object_properties.each do |op|
          @prefixes.each do |prefix,url|
            if op.to_string.match(/^#{url}/) then
              debug "Found object property '#{op.to_string}' with namespace '#{prefix.to_s}' in ontology"
              @object_properties[prefix][op.to_string.sub(/^#{url}/,"").intern]=op
              break
            end
          end
        end unless model.nil?
        model.list_datatype_properties.each do |dp|
          @prefixes.each do |prefix,url|
            if dp.to_string.match(/^#{url}/) then
              debug "Found datatype property '#{dp.to_string}' with namespace '#{prefix.to_s}' in ontology"
              @datatype_properties[prefix][dp.to_string.sub(/^#{url}/,"").intern]=dp
              break
            end
          end
        end unless model.nil?
      end
    end
  end
end
