# encoding: utf-8
# license: gpl3p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST

module Iof
  module Ontology
    #This jena interface is a rework of the original IO Jena interface
    #Jena API calls will work only in sync mode
    #a SPARQL/SPARUL based implementation will be implemented in the future
    class SparUlJena < Iof::Ontology::Base
      Iof::Ontology::Handler::ENGINE_SELECTOR.register_member(self,:sparul_jena_tdb)
      def initialize(name)
        super(name,:jena)
        # should not contain any data and is used to retrieve the ontology structure
        # this file is also used to initialize TDB storages or be read in the case of missing data file
        config.declare_dynamic_key(:"ontology.#{@name.to_s}.structure_file",:string,File.join(Ghun::Base::Blackboard.application.data_path,'ontologies','network.owl'))
        config.declare_dynamic_key(:"ontology.#{@name.to_s}.storage_directory",:string,File.join(Ghun::Base::Blackboard.local.data_path,'ontologies','network'),false)
        config.declare_dynamic_key(:"ontology.#{@name.to_s}.version_after_write",:bool,true)
        config.declare_dynamic_key(:"ontology.#{@name.to_s}.start_with_empty_ontology",:bool,false)
        #startup types
        # full - the ontology populated with data will be loaded and reasoned
        # fastwrite - the ontology populated with data will be loaded the reasoner is not active
        # it is possible to change from full to fastwrite during operation (TODO implement this)
        config.declare_dynamic_key(:"ontology.#{@name.to_s}.startup_type",:string,"full")

        @structure_file=config[:"ontology.#{@name.to_s}.structure_file"]
        @storage_backend=config[:"ontology.#{@name.to_s}.storage_backend"].intern
        @storage_file=config[:"ontology.#{@name.to_s}.storage_file"]
        @storage_directory=config[:"ontology.#{@name.to_s}.storage_directory"]
        @version_after_write=config[:"ontology.#{@name.to_s}.version_after_write"]
        @start_with_empty_ontology=config[:"ontology.#{@name.to_s}.start_with_empty_ontology"]
        @startup_type=config[:"ontology.#{@name.to_s}.startup_type"].intern

        if @start_with_empty_ontology && @readonly
          warn "start_with_empty_ontology and read_only is active for ontology #{@name}", "start_with_empty_ontology tries to delete existing data, read_only tries to protect themwill prevent this", "Prefering read_only"
          @start_with_empty_ontology=false
        end

        raise "Unknown startup type #{@startup_type}" if @startup_type!=:full && @startup_type!=:fastwrite
        unless File.exists?(@structure_file)
          raise "Ontology structure file #{@structure_file} does not exist"
        end

        raise "No storage directory name for tdb storage given" if @storage_directory.nil?() || @storage_directory.empty?()

        options=Java::Util::Properties.new()
        options.set_property("USE_UNIQUE_NAME_ASSUMPTION","true")
        options.set_property("REALIZE_INDIVIDUAL_AT_A_TIME","true")
        options.set_property("USE_ANNOTATION_SUPPORT","true")
        options.set_property("HIDE_TOP_PROPERTY_VALUES","false")
        Pellet::PelletOptions.set_options(options)

        @model=nil
        @dataset=nil
        @graphstore=nil
        @datatype_properties=nil
        @object_properties=nil
        @classes=nil

        @exec_mutex=Monitor.new()
        @sync_query_exec=nil
        @async_query_exec=nil
        @async_query_id=nil
        self.allow_parallel_execution=true
      end
      attr_reader :datatype_properties, :object_properties, :classes, :model

      def query(query,show_datatype_properties=true,show_anonymous_nodes=false,list_namespaces=[@default_ns])
        return execute_request({:query => query, :show_datatype_properties => show_datatype_properties, :show_anonymous_nodes=>show_anonymous_nodes,:list_namespaces=>list_namespaces,:mode=>:sync})
      end

      def enqueue_query(id,query,show_datatype_properties=true,show_anonymous_nodes=false,list_namespaces=[@default_ns])
        enqueue_request(id,{:query => query, :show_datatype_properties => show_datatype_properties, :show_anonymous_nodes=>show_anonymous_nodes,:list_namespaces=>list_namespaces,:mode=>:async})
      end

      def abort_running_request(request_id)
        @exec_mutex.synchronize {
          if request_id.nil?()
            unless @sync_query_exec.nil?()
              warn "Trying to abort query running in sync mode"
              begin
                @sync_query_exec.abort()
              rescue => e
                error "Failed query abort", e
              end
            else
              debug "Either no request or an update request is running"
            end
          elsif !request_id.nil?() && @async_query_id==request_id
            unless @async_query_exec.nil?()
              warn "Trying to abort query running async mode"
              begin
                @async_query_exec.abort()
                @state_mutex.synchronize {
                  return @request_state[request_id.intern]=:aborted if @request_state.include?(request_id.intern)
                }
              rescue => e
                error "Failed query abort", e
              end
            else
              debug "Either no request or an update request is running"
            end
          end
        }
      end
      alias :abort_query :abort_request

      def create_individual(klass,individual,klass_ns=@default_ns,individual_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        c=get_jena_class(klass,klass_ns.intern)
        @model.create_individual("#{@prefixes[individual_ns.intern]}#{individual}",c)
        return { :name => individual, :ns => individual_ns.intern, :unknown => false }
      rescue => e
        error "Failed to create individual '#{individual_ns.to_s}:#{individual}' with class '#{klass_ns.to_s}:#{klass}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def exists_individual?(individual,individual_ns=@default_ns)
        !get_jena_individual(individual,individual_ns.intern).nil?()
      end

      def drop_individual(individual,individual_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        i.remove()
        return true
      rescue => e
        error "Could not remove individual '#{individual_ns.to_s}:#{individual}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def has_datatype_property?(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        p=get_jena_datatype_property(property,property_ns.intern)
        if p.nil?() then
          error "Datatype property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        return i.has_property(p)
      rescue => e
        error "Could not determine existence of datatype property '#{property_ns.to_s}:#{property}' of individual '#{individual_ns.to_s}:#{individual}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def has_object_property?(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        p=get_jena_object_property(property,property_ns.intern)
        if p.nil?() then
          error "Object property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        return i.has_property?(p)
      rescue => e
        error "Could not determine existence of object property '#{property_ns.to_s}:#{property}' of individual '#{individual_ns.to_s}:#{individual}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def get_object_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns,list_namespaces=[@default_ns],show_unknown_nodes=false)
        list_namespaces=@prefixes.keys if list_namespaces.nil?() || list_namespaces.empty?()
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        result=Array.new()
        p=get_jena_object_property(property,property_ns.intern)
        if p.nil?() then
          error "Object property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        values=i.list_property_values(p)
        values.each do |value|
          found=false
          @prefixes.each do |prefix,url|
            if value.to_string.match(/^#{url}/) then
              v=value.to_string
              found=true
              if list_namespaces.include?(prefix) then
                v.gsub!(/^#{url}/,'')
                r=Hash.new()
                r[:name]=v
                r[:ns]=prefix
                r[:unknown]=false
                result << r
              else
                debug "Value #{value.to_string} matched by namespace filter"
              end
            end
          end
          if show_unknown_nodes then
            debug "Adding result '#{value.to_string}' with unknown namespace to result"
            r=Hash.new()
            r[:name]=value.to_string
            r[:unknown]=true
            result << r
          else
            debug "Result '#{value.to_string}' contains unknown namespace"
          end unless found
        end unless values.nil?
        return result
      rescue => e
        error "Could not determine value(s) of property '#{property_ns.to_s}:#{property}' of individual '#{individual_ns.to_s}:#{individual}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def get_datatype_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        result=Array.new()
        p=get_jena_datatype_property(property.intern,property_ns)
        if p.nil?() then
          error "Datatype property '#{property_ns.to_s}#{property}' is unknown"
          return nil
        end
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        values=i.list_property_values(p)
        values.each do |v|
          result << v.get_string
        end unless values.nil?
        return result
      rescue => e
        error "Could not determine value(s) of property '#{property_ns.to_s}:#{property}' of individual '#{individual_ns.to_s}:#{individual}'", e
        return nil
      ensure
        @model.leave_critical_section()
      end

      def drop_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        prefix=nil
        if property_ns.is_a?(Symbol) then
          prefix=property_ns.intern
        else
          prefix=url_to_prefix(property_ns)
        end
        p=nil
        p=get_jena_datatype_property(property,property_ns.intern) if @datatype_properties.include?(prefix) && @datatype_properties[prefix].include?(property)
        p=get_jena_object_property(property,property_ns.intern) if @object_properties.include?(prefix) && @object_properties[prefix].include?(property)
        if p.nil?() then
          error "Property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i.remove_all(p)
        return true
      rescue => e
        error "Could not remove property '#{property_ns.to_s}:#{property}' for individual '#{individual_ns.to_s}:#{individual}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def drop_object_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        p=get_jena_object_property(property,property_ns.intern)
        if p.nil?() then
          error "Object property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i.remove_all(p)
        return true
      rescue => e
        error "Could not remove property '#{property_ns.to_s}:#{property}' for individual '#{individual_ns.to_s}:#{individual}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def drop_datatype_property(individual,property,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        if i.nil?() then
          error "Individual '#{individual_ns.to_s}:#{individual}' does not exist"
          return nil
        end
        p=get_jena_datatype_property(property,property_ns.intern)
        if p.nil?() then
          error "Datatype property '#{property_ns.to_s}:#{property}' is unknown"
          return nil
        end
        i.remove_all(p)
        return true
      rescue => e
        error "Could not remove property '#{property_ns.to_s}:#{property}' for individual '#{individual_ns.to_s}:#{individual}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def create_object_property(individual,property,target,individual_ns=@default_ns,property_ns=@default_ns,target_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        p=get_jena_object_property(property,property_ns.intern)
        t=get_jena_individual(target,target_ns.intern)
        error "Failed to retrieve jena object for individual '#{individual_ns.to_s}:#{individual}'" if i.nil?()
        error "Failed to retrieve jena object for property '#{property_ns.to_s}:#{property}'" if p.nil?()
        error "Failed to retrieve jena object for individual '#{target_ns.to_s}:#{target}'" if t.nil?()
        if i.nil?() || p.nil?() || t.nil?()
          return false
        else
          s=@model.createStatement(i,p,t)
          @model.add(s)
          return true
        end
      rescue => e
        error "Failed to create object property '#{property_ns.to_s}:#{property}' between '#{individual_ns.to_s}:#{individual}' and '#{target_ns.to_s}:#{target}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

      def create_datatype_property(individual,property,value,individual_ns=@default_ns,property_ns=@default_ns)
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        i=get_jena_individual(individual,individual_ns.intern)
        p=get_jena_datatype_property(property,property_ns.intern)
        error "Failed to retrieve jena object for individual '#{individual_ns.to_s}:#{individual}'" if i.nil?()
        error "Failed to retrieve jena object for property '#{property_ns.to_s}:#{property}'" if p.nil?()
        if i.nil?() || p.nil?()
          return false
        else
          s=@model.createStatement(i,p,value.to_s)
          @model.add(s)
          return true
        end
      rescue => e
        error "Failed to create datatype property '#{property_ns.to_s}:#{property}' for '#{individual_ns.to_s}:#{individual}' with value '#{value}'", e
        return false
      ensure
        @model.leave_critical_section()
      end

    private

      def connect_helper()
        prepare_ontology()
        load_ontology()
        init_classes(@model)
        init_properties(@model)
        @connected=true
      end

      def disconnect_helper()
        write_ontology() unless @readonly
        cleanup()
        @connected=false
      end

      def prepare_ontology()
        unless @readonly
          if @start_with_empty_ontology && Dir.exists?(@storage_directory)
            info "Removing existing ontology #{@storage_directory} as requested"
            FileUtils.rm_r(@storage_directory)
          end
          unless Dir.exists?(@storage_directory)
            debug "Creating directory #{@storage_directory}"
            FileUtils.mkdir_p(@storage_directory)
          end
          unless File.writable?(@storage_directory)
            raise "Directory #{@storage_directory} is not writable"
          end
          unless File.exists?(File.join(@storage_directory,'nodes.dat'))
            info "Creating empty TDB storage"
            tdb_dataset=Jena::Tdb::TdbFactory.create_dataset(@storage_directory)
            tdb_graph=tdb_dataset.get_default_model().get_graph()
            Jena::Tdb::TdbLoader.load(tdb_graph,[@structure_file],false)
            tdb_graph.close()
            tdb_dataset.close()
          end
        end
      end

      def load_ontology()
        @dataset=Jena::Tdb::TDBFactory.create_dataset(@storage_directory)
        @graphstore=Jena::Update::GraphStoreFactory.create(@dataset)
        if @startup_type==:full
          @model = Jena::Rdf::Model::ModelFactory.createOntologyModel(Pellet::Jena::PelletReasonerFactory.THE_SPEC, @dataset.default_model)
          debug "Preparing ontology"
          @model.prepare
          graph=@model.get_graph().get_kb()
          debug "Classifying ontology"
          graph.classify
          debug "Realizing ontology"
          graph.realize
        else
          @model = Jena::Rdf::Model::ModelFactory.createOntologyModel(Jena::Ontology::OntModelSpec::OWL_MEM, @dataset.default_model())
        end
      end

      def write_ontology()
        @model.commit()
        @dataset.close()
        @model.close unless @model.closed?()
        @graphstore.close unless @graphstore.closed?()
      end

      def cleanup()
        @model=nil
        @graphstore=nil
        @dataset=nil
        @sync_query_exec=nil
        @async_query_exec=nil

        @datatype_properties=nil
        @object_properties=nil
        @classes=nil
      end

      def execute_sparql_helper(query,mode,show_datatype_properties,show_anonymous_nodes,list_namespaces)
        list_namespaces=@prefixes.keys if list_namespaces.nil?() || list_namespaces.empty?()
        results=Array.new()
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        if mode==:sync
          @sync_query_exec=Jena::Query::QueryExecutionFactory.create(Jena::Query::QueryFactory.create(query), @model)
          exec=@sync_query_exec
        else
          @async_query_exec=Jena::Query::QueryExecutionFactory.create(Jena::Query::QueryFactory.create(query), @model)
          exec=@async_query_exec
        end
        begin
          exec.execSelect.each do |raw|
            result=Hash.new()
            vars=raw.varNames.to_a
            vars.each do |v|
              r=raw.get(v)
              if r.nil?() then
                result[v.intern] = r
              else
                result[v.intern] = r.to_string
              end
            end
            results << result
          end
        rescue => e
          error "Failed to execute query: '#{query}'", e
          results=nil
          raise
        ensure
          exec.close
          if mode==:sync
            @sync_query_exec=nil
          else
            @async_query_exec=nil
            @async_query_id=nil
          end
        end
        return format_result(results,list_namespaces,show_datatype_properties,show_anonymous_nodes)
      rescue => e
        error "Failed to execute query: '#{query}'", e
        results=nil
        raise
      ensure
        @model.leave_critical_section()
      end

      def execute_sparul_helper(query,mode)
        list_namespaces=@prefixes.keys if list_namespaces.nil?() || list_namespaces.empty?()
        @model.enter_critical_section(Jena::Shared::Lock.WRITE)
        if mode==:sync
          @sync_query_exec=nil
        else
          @async_query_exec=nil
        end
        exec=Jena::Update::UpdateExecutionFactory.create(Jena::Update::UpdateFactory.create(query), @graph_store)
        begin
          exec.execute
        rescue => e
          error "Failed to execute query: '#{query}'", e
          raise
        ensure
          exec.close
          if mode==:sync
            @sync_query_exec=nil
          else
            @async_query_exec=nil
            @async_query_id=nil
          end
        end
        return true
      rescue => e
        error "Failed to execute query: '#{query}'", e
        raise
      ensure
        @model.leave_critical_section()
      end

      def get_jena_individual(name,ns)
        prefix=nil
        if ns.is_a?(Symbol) then
          prefix=ns
        else
          prefix=url_to_prefix(ns)
        end
        @model.enter_critical_section(Jena::Shared::Lock.READ)
        @model.get_individual("#{@prefixes[prefix]}#{name}")
      rescue => e
        error "Failed to retrieve individual", e
      ensure
        @model.leave_critical_section()
      end

      def get_jena_datatype_property(name,ns)
        prefix=nil
        if ns.is_a?(Symbol) then
          prefix=ns
        else
          prefix=url_to_prefix(ns)
        end
        if @datatype_properties.include?(prefix) && @datatype_properties[prefix].include?(name.intern) then
          return @datatype_properties[prefix][name.intern]
        else
          error "Datatype property '#{name}' with namespace #{ns} unknown in ontology"
          raise PropertyUnknownError
        end
      end

      def get_jena_object_property(name,ns)
        prefix=nil
        if ns.is_a?(Symbol) then
          prefix=ns
        else
          prefix=url_to_prefix(ns)
        end
        if @object_properties.include?(prefix) && @object_properties[prefix].include?(name.intern) then
          return @object_properties[prefix][name.intern]
        else
          error "Object property '#{name}' with namespace #{ns} unknown in ontology"
          raise PropertyUnknownError
        end
      end

      def get_jena_class(name,ns)
        prefix=nil
        if ns.is_a?(Symbol) then
          prefix=ns
        else
          prefix=url_to_prefix(ns)
        end
        if @classes.include?(prefix) && @classes[prefix].include?(name.intern) then
          return @classes[prefix][name.intern]
        else
          error "Class '#{name}' with namespace #{ns} unknown in ontology"
          raise ClassUnknownError, "Class '#{name}' with namespace #{ns} unknown in ontology"
        end
      end

      def init_classes(model)
        @classes=Hash.new()
        @prefixes.keys.each do |prefix|
          @classes[prefix]=Hash.new()
        end
        model.list_classes.each do |c|
          @prefixes.each do |prefix,url|
            if c.to_string.match(/^#{url}/) then
              debug "Found class '#{c.to_string}' with namespace '#{prefix.to_s}' in ontology"
              @classes[prefix][c.to_string.sub(/^#{url}/,"").intern]=c
              break
            end
          end
        end unless model.nil?
      end

      def init_properties(model)
        @datatype_properties=Hash.new()
        @object_properties=Hash.new
        @prefixes.keys.each do |prefix|
          @datatype_properties[prefix]=Hash.new()
          @object_properties[prefix]=Hash.new
        end
        model.list_object_properties.each do |op|
          @prefixes.each do |prefix,url|
            if op.to_string.match(/^#{url}/) then
              debug "Found object property '#{op.to_string}' with namespace '#{prefix.to_s}' in ontology"
              @object_properties[prefix][op.to_string.sub(/^#{url}/,"").intern]=op
              break
            end
          end
        end unless model.nil?
        model.list_datatype_properties.each do |dp|
          @prefixes.each do |prefix,url|
            if dp.to_string.match(/^#{url}/) then
              debug "Found datatype property '#{dp.to_string}' with namespace '#{prefix.to_s}' in ontology"
              @datatype_properties[prefix][dp.to_string.sub(/^#{url}/,"").intern]=dp
              break
            end
          end
        end unless model.nil?
      end
    end
  end
end
