# encoding: utf-8
# license: gpl2p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST
module Iof
  module Parser
    class SSHCisco < Iof::Parser::CLIBase
      Iof::Parser::SELECTOR.register_member(self,[:cisco,nil,:ssh,nil])
      def initialize(target,target_handler)
        super(target,target_handler)

        @interfaces=Hash.new()
        @acls_list=Hash.new()
        @acls_yaml=Hash.new()
        @ipv4_forwards=nil
        @ipv6_forwards=nil

        @components=nil
        @neighbors=nil
        @connections=nil

        @temporary_storage=Hash.new()
        @mac_addresses=Hash.new()
        @mac_by_interface=Hash.new()
        @mac_by_interface_parsed=Hash.new()
        @interface_by_mac=Hash.new()
        @modules=Hash.new()

        @vtp_vlan_by_interface=Hash.new()
        @vtp_vlans=Array.new()

        @power_outgoing=Hash.new()
        @powered_devices=Hash.new()
        @cdp_neighbors_by_interface=Hash.new()

        @port_channel_to_interface=Hash.new()
        @interface_to_port_channel=Hash.new()

        @preparse_commands=[:hosts,:version,:ipv4,:ipv6,:arp,:mac_addresses,:arp,:vlan,:cdp_neighbors]
        @preparse_additional=[]
        @parse_commands=[:power,:etherchannel,:cdp_interfaces,:configuration,:cdp_neighbors,:mac_addresses,:interfaces_trunk,:switch,:redundancy,:vtp_status,:vtp_devices,:module]
        @parse_additional=[:module_all]
        @postparse_commands=[:mac_addresses]
        @postparse_additional=[]
        @preparse_clean=(@preparse_commands+@preparse_additional)-(@parse_commands+@parse_additional+@postparse_commands+@postparse_additional)
        @parse_clean=(@parse_commands+@parse_additional)-(@postparse_commands+@postparse_additional)
        @postparse_clean=(@postparse_commands+@postparse_additional)

      end

      def cleanup(commands)
        commands.each do |c|
          if @target.results.include?(c)
            @target.results[c][:stdout]=nil
            @target.results[c][:stderr]=nil
          else
            debug "Nothing to clean up for command #{c} for target #{@target.description()}"
          end
        end unless commands.nil?()
      end

      def preparse()
        unless super()
          @target.status_preparse=false
          return false
        end
        [:hosts,:version,:ipv4,:ipv6,:arp,:mac_addresses,:arp,:vlan,:cdp_neighbors].each do |c|
          parse_command(c, :preparse)
        end
        cleanup(@preparse_clean)
        return true
      rescue => e
        error "Failed to preparse target #{@target.description()}", e
        @target.status_preparse=false
        return false
      end

      def parse(data_handler)
        unless super(data_handler)
          @target.status_parse=false
          return false
        end
#        [:power,:etherchannel,:cdp_interfaces,:configuration,:cdp_neighbors,:mac_addresses,:interfaces_trunk,:switch,:redundancy,:vtp_status,:vtp_devices].each do |c|
        [:power,:etherchannel,:cdp_interfaces,:configuration,:cdp_neighbors,:mac_addresses,:interfaces_trunk,:switch,:redundancy,:vtp_status,:vtp_devices,:module].each do |c|
          c=:module_all if c==:module && @temporary_storage[:long_interface_names] && !@temporary_storage[:switch_ids].nil?() && !@temporary_storage[:switch_ids].empty?() && @target.commands.include?(:module_all) && !(@target.results[:module_all][:stdout].nil?() || @target.results[:module_all][:stdout]=='' || @target.results[:module_all][:stdout].match("Line has invalid autocommand"))
          parse_command(c, :parse)
        end
        generate_output()
        cleanup(@parse_clean)
        return true
      rescue => e
        error "Failed to parse target #{@target.description()}", e
        @target.status_parse=false
        return false
      end

      def postparse()
        unless super()
          @target.status_postparse=false
          return false
        end
        [:mac_addresses].each do |c|
          parse_command(c, :postparse)
        end
        process_incoming()
        postprocess_parsed_data()
        send_results()
        cleanup(@preparse_clean)
        return true
      rescue => e
        error "Failed to postparse target #{@target.description()}", e
        @target.status_postparse=false
        return false
      end

     private

      def parse_hosts(label,command,result,step)
        result[:stdout].split("\n").each do |line|
          next if line.nil?()
          line.strip!
          next if line==''
          if line.match(/Default domain is/)
            @temporary_storage[:default_domain]=line.sub(/.*domain is /,'').strip()
            break
          end
        end
      end

      def parse_version(label,command,result,step)
        hostname=nil
        processor_board_serial=nil
        system_serial=nil
        result[:stdout].split("\n").each do |line|
          next if line.nil?()
          line.strip!
          next if line==""
          if line.match(/uptime is/)
            hostname=line.sub!(/\s*uptime is.*$/,'').strip()
            hostname+=".#{@temporary_storage[:default_domain]}" if @temporary_storage.include?(:default_domain) && !@temporary_storage[:default_domain].nil?() && @temporary_storage[:default_domain]!=''
            @temporary_storage.delete(:default_domain)
            debug "Target #{@target.description()} has hostname #{hostname}"
            @target.add_name(@target_handler, hostname, :own)
          elsif line.match(/^Processor board ID/) then
            processor_board_serial=line.sub!(/^Processor board ID\s*/,'')
            debug "Target #{@target.description()} has processor board id #{processor_board_serial}"
          elsif line.match(/^System serial number/) then
            system_serial=line.sub!(/System serial number\s*:\s*/,'')
            debug "Target #{@target.description()} has system serial #{system_serial}"
          #else
          #  debug "Ignoring not recognized line: #{line}"
          end
        end
        @target.special_mutex.synchronize {
          if system_serial
            @target.special[:serial]=system_serial
            debug "Ignoring processor board id '#{processor_board_serial}'; target #{@target.description()} has system serial '#{system_serial}'"
          else
            @target.special[:serial]=processor_board_serial
          end
          @target.identifier << @target.special[:serial]
        }
      end

      def parse_ipv4(label,command,result,step)
        header=false
        result[:stdout].split("\n").each do |line|
          next if line.nil?()
          line.strip!
          next if line==''
          unless header
            header=true
            next
          end
          sp_line=line.split(' ')
          unless sp_line[1]=='unassigned' || sp_line[1].match(/unnumbered/) || sp_line[4]=='down' || sp_line[5]=='down' || sp_line[6]=='down' || sp_line[0].match(/^Loopback/) || sp_line[0].match(/^Lo/)
            debug "IPv4 address #{sp_line[1]} at Interface #{sp_line[0]} on #{@target.description()}"
            @target.add_ipv4(@target_handler, sp_line[1])
          end
        end
      end

      def parse_ipv6(label,command,result,step)
        return if result[:stdout].nil?()
        interface_line=true
        interface=""
        down=false
        result[:stdout].split("\n").each do |line|
          next if line.nil?()
          next if line.strip()==''
          interface_line=line.match(/^ +/).nil?()
          line.strip!()
          if interface_line
            interface=line.split(' ')[0].strip()
            down=line.match(/down/)
            next
          else
            l=line.strip()
            unless l=='unassigned' || l.match(/unnumbered/) || down || interface.match(/^Loopback/) || interface.match(/^Lo/)
              debug "IPv6 address #{l} at Interface #{interface} on #{@target.description()}"
              @target.add_ipv6(@target_handler,l)
            end
          end
        end
      end

      def parse_module(label,command,result,step)
        if result[:stdout].nil?() || result[:stdout]=='' || result[:stdout].match("Line has invalid autocommand")
          debug "#{@target.description()} has no switch output"
        else
          if @target.device_type.match(/^6/) then
            @modules=parse_module_c6500(label,command,result,step)
          elsif @target.device_type.match(/^4/) then
            @modules=parse_module_c4500(label,command,result,step)
          else
            warn "No parser method for #{@target.description} (#{@target.device_vendor}:#{@target.device_type})"
          end
        end
      end
      alias :parse_module_all :parse_module

      def parse_module_c6500(label,command,result,step)
        modules=Hash.new()
        sub_id=nil
        modules[sub_id]=Hash.new()
        first_header=true
        first_content=false
        mac_header=false
        mac_content=false
        sub_header=false
        sub_content=false
        diag_header=false
        diag_content=false
        result[:stdout].split("\n").each do |line|
          if line.strip.match(/-+ -+ -+ -+ -+/) || line.strip.match(/^---- -+$/)
            if first_header
              first_header=false
              first_content=true
              next
            elsif mac_header
              mac_header=false
              mac_content=true
              next
            elsif sub_header
              sub_header=false
              sub_content=true
              next
            elsif diag_header
              diag_header=false
              diag_content=true
              next
            end
          elsif line.strip.empty?()
            if first_content
              first_content=false
              mac_header=true
              next
            elsif mac_content
              mac_content=false
              sub_header=true
              next
            elsif sub_content
              sub_content=false
              diag_header=true
              next
            elsif diag_content
              #C6500 with VSS active print complete module for each participating switch with module_all
              diag_content=false
              first_header=true
              next
            end
          end
          next if line.strip.empty?()
          if line.match(/Switch Number/)
            id=line.sub(/.*Switch Number:/,'').sub(/Role.*/,'').strip
            if id.match(/^[0-9]+$/)
              sub_id=id.intern
              modules[sub_id]=Hash.new()
            end
          end
          if first_content
            sp_line=line.split(' ')
            id=sp_line[0].strip().intern
            modules[sub_id][id]=Hash.new()
            modules[sub_id][id][:id]=sp_line[0].strip()
            modules[sub_id][id][:name]="module_#{sp_line[0].strip()}"
            modules[sub_id][id][:number_of_ports]=sp_line[1].strip()
            modules[sub_id][id][:serial]=sp_line[-1].strip()
            modules[sub_id][id][:model]=sp_line[-2].strip()
            modules[sub_id][id][:card_type]=sp_line[2..-3].join(' ').strip
            modules[sub_id][id][:interfaces]=Array.new()
            if modules[sub_id][id][:card_type].match(/Supervisor/)
              modules[sub_id][id][:type]=:supervisor
            elsif modules[sub_id][id][:card_type].match(/WiSM|WLAN/)
              modules[sub_id][id][:type]=:wireless
            elsif modules[sub_id][id][:card_type].match(/Firewall/)
              modules[sub_id][id][:type]=:firewall
            elsif modules[sub_id][id][:card_type].match(/10GE|10+BaseF+[XT]|GBIC|10+[mM][bB]|SFP|RJ45|/)
              modules[sub_id][id][:type]=:interface
            else
              modules[sub_id][id][:type]=:unknown
              warn "Could not determine module type from card type #{modules[sub_id][id][:card_type]}"
            end
          elsif mac_content
            #TODO parse hardware, firmware and software versions from module output
            #TODO parse mac addresses for interfaces on modular devices from module output
            next
          elsif sub_content
            next
          elsif diag_content
            next
          end
        end
        return modules
      end

      def parse_module_c4500(label,command,result,step)
        modules=Hash.new()
        sub_id=nil
        modules[sub_id]=Hash.new()
        first_header=true
        first_content=false
        mac_header=false
        mac_content=false
        result[:stdout].split("\n").each do |line|
          if line.match(/-+\+-+\+-+\+-+\+-+/)
            if first_header
              first_header=false
              first_content=true
              next
            elsif mac_header
              mac_header=false
              mac_content=true
              next
            end
          elsif line.strip.empty?()
            if first_content
              first_content=false
              mac_header=true
              next
            elsif mac_content
              #C4500 with VSS active print complete module for each participating switch
              mac_content=false
              first_header=true
              next
            end
          end
          next if line.strip.empty?()
          if line.match(/Switch Number/)
            id=line.sub(/.*Switch Number:/,'').sub(/Role.*/,'').strip
            if id.match(/^[0-9]+$/)
              sub_id=id.intern
              modules[sub_id]=Hash.new()
            end
          end
          if first_content
            sp_line=line.split(' ')
            id=sp_line[0].strip().intern
            modules[sub_id][id]=Hash.new()
            modules[sub_id][id][:id]=sp_line[0].strip()
            modules[sub_id][id][:name]="module_#{sp_line[0].strip()}"
            modules[sub_id][id][:number_of_ports]=sp_line[1].strip()
            modules[sub_id][id][:serial]=sp_line[-1].strip()
            modules[sub_id][id][:model]=sp_line[-2].strip()
            modules[sub_id][id][:card_type]=sp_line[2..-3].join(' ').strip
            modules[sub_id][id][:interfaces]=Array.new()
            if modules[sub_id][id][:card_type].match(/Supervisor/)
              modules[sub_id][id][:type]=:supervisor
            elsif modules[sub_id][id][:card_type].match(/WiSM|WLAN/)
              modules[sub_id][id][:type]=:wireless
            elsif modules[sub_id][id][:card_type].match(/Firewall/)
              modules[sub_id][id][:type]=:firewall
            elsif modules[sub_id][id][:card_type].match(/10GE|10+BaseF+[XT]|GBIC|10+[mM][bB]|SFP|RJ45|/)
              modules[sub_id][id][:type]=:interface
            else
              modules[sub_id][id][:type]=:unknown
              warn "Could not determine module type from card type #{modules[sub_id][id][:card_type]}"
            end
          elsif mac_content
            #TODO parse hardware, firmware and software versions from module output
            #TODO parse mac addresses for interfaces on modular devices from module output
            next
          end
        end
        return modules
      end



      def parse_mac_addresses(label,command,result,step)
        case step
        when :preparse
          mac_result=nil
          if @target.device_type.match(/^[6]/) then
            mac_result=parse_mac_addresses_c6000(label,command,result,step)
          elsif @target.device_type.match(/^4/) then
            mac_result=parse_mac_addresses_c4000(label,command,result,step)
          else
            mac_result=parse_mac_addresses_default(label,command,result,step)
          end
          mac_result.each do |mac|
            m=mac[:mac_address].intern
            @mac_addresses[m]={:vlans => Array.new(), :interfaces => Array.new() } unless @mac_addresses.include?(m)
            @mac_addresses[m][:vlans] << mac[:vlan] unless @mac_addresses[m][:vlans].include?(mac[:vlan])
            @mac_addresses[m][:interfaces] << mac[:interface] unless @mac_addresses[m][:interfaces].include?(mac[:interface])
            @interface_by_mac[m]=Array.new() unless @interface_by_mac.include?(m)
            @interface_by_mac[m] << mac[:interface] unless @interface_by_mac[m].include?(mac[:interface])
            @mac_by_interface[mac[:interface].intern]=Array.new() unless @mac_by_interface.include?(mac[:interface].intern)
            @mac_by_interface[mac[:interface].intern] << mac[:mac_address] unless @mac_by_interface[mac[:interface].intern].include?(mac[:mac_address])
          end
          @mac_addresses.each do |m,value|
            debug "Multiple seen mac address #{m.to_s} on interfaces #{value[:interfaces].to_s} for #{@target.description()}" if value[:interfaces].size > 1
            value[:interfaces].size().times { @target_handler.update_mac(m.to_s,:seen) }
          end
        when :parse
          #TODO try to fix not_implemented case
          #FIXME: handle port-channel
          @mac_by_interface.each do |interface,macs|
            if macs.size() <= 0
              #no mac neighbors; do nothing
              next
            elsif macs.size() == 1
              #only one mac neighbor; try to join with cdp neighbor without consideration of device class
              mac=macs[0]
              neighbors=@cdp_neighbors_by_interface[interface]
              if neighbors.nil?() || neighbors.size() <= 0
                #found one mac address and no cdp neighbors
                #assuming single host on this port
                @mac_by_interface_parsed[interface]=macs.clone() unless interface.to_s.match(/^Port-channel/)
              elsif neighbors.size() == 1
                #found one mac address and one cdp neighbor
                n=neighbors[0]
                if n.include?(:mac_addresses) && n[:mac_addresses].include?(mac)
                  #mapping from mac address to cdp neighbor and ip address already done
                elsif n.include?(:mac_addresses) && n[:mac_addresses].size>0 && !n[:mac_addresses].include?(mac)
                  warn Ghun::Log::Type::REPORT, "Seen mac address #{mac} on interface #{interface} for device #{@target.description()} is not the one for the cdp neighbor #{n.inspect}"
                  warn "Seen mac address #{mac} on interface #{interface} for device #{@target.description()} is not the one for the cdp neighbor #{n.inspect}"
                else
                  #joining
                  target=nil
                  target=nil
                  target=@target_handler.get_target_by_name(n[:name])
                  n[:mac_addresses].each do |a|
                    t=@target_handler.get_target_by_mac(a)
                    unless t.nil?()
                      target=t
                      break
                    end
                  end if target.nil?()
                  n[:ipv4_addresses].each do |a|
                    t=@target_handler.get_target_by_ipv4(a)
                    unless t.nil?()
                      target=t
                      break
                    end
                  end if target.nil?()
                  if target.nil?()
                    warn "Found no target for neighbor; this may be caused by active runner filters; #{n.inspect()}"
                  else
                    n[:mac_addresses]=Array.new() unless n.include?(:mac_addresses)
                    target.add_mac(@target_handler, mac)
                    n[:mac_addresses] << mac unless n[:mac_addresses].include?(mac)
                    target.ipv4_addresses.each do |a|
                      @target_handler.assign_mac_ipv4(mac, a)
                    end
                  end
                end
              else
                #found one mac address and multiple cdp neighbors
                warn Ghun::Log::Type::REPORT, "Found one mac address #{mac} on interface #{interface} for device #{@target.description()} but multiple cdp neighbors"
                warn "Found one mac address #{mac} on interface #{interface} for device #{@target.description()} but multiple cdp neighbors"
              end
            else
              #multiple mac neighbors; try to join with cdp neighbors with consideration of device class and try to hand over mac list
              neighbors=@cdp_neighbors_by_interface[interface]
              if neighbors.nil?() || neighbors.size() <= 0
                #found multiple mac addresses and no cdp neighbors
                #assuming multiple hosts connected to unmanaged switch connected to this interface
                @mac_by_interface_parsed[interface]=macs.clone() unless interface.to_s.match(/^Port-channel/)
              elsif neighbors.size() == 1
                #found multiple mac addresses and one cdp neighbor
                # checking neighbor type
                n=neighbors[0]
                if n[:device_class]==:switch
                  #drop seen mac_addresses
                  macs.each do |m|
                    @target_handler.update_mac(m.to_s,:switch_filter)
                  end
                elsif n[:device_class]==:wireless_component || n[:device_class]==:voice_component
                  target=nil
                  target=@target_handler.get_target_by_name(n[:name])
                  n[:mac_addresses].each do |a|
                    t=@target_handler.get_target_by_mac(a)
                    unless t.nil?()
                      target=t
                      break
                    end
                  end if target.nil?()
                  n[:ipv4_addresses].each do |a|
                    t=@target_handler.get_target_by_ipv4(a)
                    unless t.nil?()
                      target=t
                      break
                    end
                  end if target.nil?()
                  if target.nil?()
                    warn "Found no target for neighbor; this may be caused by active runner filters; #{n.inspect()}"
                  else
                    target.special_mutex.synchronize() {
                      target.special[:mac_addresses]=Array.new() unless target.special.include?(:mac_addresses)
                      macs.each do |m|
                        target.special[:mac_addresses] << mac unless (n.include?(:mac_addresses) && n[:mac_addresses].include?(m)) || target.mac_addresses.include?(m)
                      end
                    }
                  end
                else
                  warn "CDP neighbor has unknown device_class; dropping seen mac addresses"
                  macs.each  { |m| @target_handler.update_mac(m.to_s,:error) }
                end
              else
                #found multiple mac addresses and multiple cdp neighbors
                warn Ghun::Log::Type::REPORT, "Found multiple mac addresses #{macs} on interface #{interface} for device #{@target.description()} for multiple cdp neighbors"
                warn "Found multiple mac addresses #{macs} on interface #{interface} for device #{@target.description()} for multiple cdp neighbors"
                macs.each  { |m| @target_handler.update_mac(m.to_s,:not_implemented) }
              end
            end unless macs.nil?()
          end
        when :postparse
          #TODO evaluate incoming seen mac addresses
          #TODO warn for seen mac addresses on both sides of a link
        end
      end

      def parse_mac_addresses_default(label,command,result,step)
        mac_result=Array.new()
        header=true
        result[:stdout].split("\n").each do |line|
          next if line.nil?()
          line.strip!
          next if line==''
          if header then
            debug "Skipping header line: #{line}"
            header=false if line.match(/^-+\s+-+\s+-+\s+-+$/)
          elsif line.match(/^Total/)
            debug "Skipping footer line: #{line}"
          else
            #debug "Parsing line #{line}"
            cells=line.split(' ')
            mac_entry=Hash.new
            mac_entry[:vlan]=cells[0].strip()
            mac_entry[:mac_address]=Iof::Data.mac_plain(cells[1].strip())
            if mac_entry[:mac_address].nil?() || mac_entry[:mac_address]==''
              warn "Failed to parse MAC address in line '#{line}'"
              next
            end
            mac_entry[:interface]=cells[3].strip()
            next if cells[3]=="flood"
            mac_entry[:interface]=expand_interface_shortname(mac_entry[:interface])
            debug "Found neighbour via mac address table with mac address #{mac_entry[:mac_address]} for vlan #{mac_entry[:vlan]} on interface #{mac_entry[:interface]}"
            mac_result << mac_entry
          end
        end
        return mac_result
      end
      def parse_mac_addresses_c6000(label,command,result,step)
        mac_result=Array.new()
        header=true
        result[:stdout].split("\n").each do |line|
          next if line.nil?()
          line.strip!
          next if line==''
          next if line.match(/No entries present/)
          if header then
            debug "Skipping header line: #{line}"
            header=false if line.match(/^-+\++-+\++-+\++-+\++-+\++-+$/)
            header=false if line.match(/^-+\++-+\++-+\++-+\++-+$/)
          elsif line.match(/^Total/)
            debug "Skipping footer line: #{line}"
          else
            #debug "Parsing line #{line}"
            line.sub!(/^\*/,' ')
            cells=line.split(' ')
            mac_entry=Hash.new
            mac_entry[:vlan]=cells[0].strip()
            mac_entry[:mac_address]=Iof::Data.mac_plain(cells[1].strip())
            if mac_entry[:mac_address].nil?() || mac_entry[:mac_address]==''
              warn "Failed to parse MAC address in line '#{line}'"
              next
            end
            mac_entry[:interface]=cells[5].strip()
            next if cells[5]=="flood"
            mac_entry[:interface]=expand_interface_shortname(mac_entry[:interface])
            debug "Found neighbour via mac address table with mac address #{mac_entry[:mac_address]} for vlan #{mac_entry[:vlan]} on interface #{mac_entry[:interface]}"
            mac_result << mac_entry
          end
        end
        return mac_result
      end

      def parse_mac_addresses_c4000(label,command,result,step)
        mac_result=Array.new()
        header=true
        result[:stdout].split("\n").each do |line|
          next if line.nil?()
          line.strip!
          next if line==''
          if header then
            debug "Skipping header line: #{line}"
            header=false if line.match(/^-+\++-+\++-+\++-+\++-+$/)
            header=false if line.match(/^-+\++-+\++-+\++-+$/)
          elsif line.match(/^Total/)
            debug "Skipping footer line: #{line}"
          else
            #debug "Parsing line #{line}"
            line.sub!(/^\*/,' ')
            cells=line.split(' ')
            mac_entry=Hash.new
            mac_entry[:vlan]=cells[0].strip()
            mac_entry[:mac_address]=Iof::Data.mac_plain(cells[1].strip())
            if mac_entry[:mac_address].nil?() || mac_entry[:mac_address]==''
              warn "Failed to parse MAC address in line '#{line}'"
              next
            end
            mac_entry[:interface]=cells[4].strip()
            next if cells[4]=="flood"
            mac_entry[:interface]=expand_interface_shortname(mac_entry[:interface])
            debug "Found neighbour via mac address table with mac address #{mac_entry[:mac_address]} for vlan #{mac_entry[:vlan]} on interface #{mac_entry[:interface]}"
            mac_result << mac_entry
          end
        end
        return mac_result
      end

      def parse_arp(label,command,result,step)
        first=true
        result[:stdout].split("\n").each do |line|
          next if line.nil?()
          line.strip!
          next if line==''
          if first then
            debug "Skipping first line: #{line}"
            first=false
          else
            cells=line.split(' ')
            mac=cells[3].strip()
            ipv4=cells[1].strip()
            unless mac.nil?() || ipv4.nil?() || ipv4.downcase()=="incomplete"
              mac=@target_handler.add_mac(mac)
              ipv4=@target_handler.add_ipv4(ipv4)
              @target_handler.assign_mac_ipv4(mac,ipv4) unless mac.nil?() || ipv4.nil?()
              debug "Found ipv4 address mapping for #{ipv4} to mac address #{mac}"
            else
              debug "Found no ipv4 address for mac address #{mac}"
            end
          end
        end
      end

      def parse_vlan(label,command,result,step)
        header=true
        last_vlan=nil
        result[:stdout].split("\n").each do |line|
          next if line.nil?()
          line.strip!
          next if line==''
          if header
            header=false if line.match(/^-+ -+ -+ -+$/)
            next
          else
            sp_line=line.split(' ')
            interfaces=nil
            if sp_line[0].to_i==0 then
              interfaces=line.split(' ').join('').split(',')
            else
              last_vlan=sp_line[0].to_i
              @vtp_vlans << last_vlan unless @vtp_vlans.include?(last_vlan)
              debug "Found vlan #{last_vlan} with name #{sp_line[1].strip()}"
              @target_handler.name_vlan(last_vlan, sp_line[1].strip(),@target.own_name) unless sp_line[1].strip().downcase.match(/vlan0*#{last_vlan}/)
              interfaces=sp_line[3..-1].join('').split(',')
            end
            interfaces.each do |interface|
              i=expand_interface_shortname(interface).intern
              @vtp_vlan_by_interface[i]=Array.new() unless @vtp_vlan_by_interface.include?(i)
              unless @vtp_vlan_by_interface[i].include?(last_vlan)
                @vtp_vlan_by_interface[i] << last_vlan
                @target_handler.update_vlan(last_vlan, :vtp)
                @target_handler.update_vlan(last_vlan, :seen)
              end
            end
          end
        end
      end

      def parse_cdp_neighbors(label,command,result,step)
        case step
        when :preparse
          entry=nil
          parsing_addresses=false
          result[:stdout].split("\n").each do |line|
            next if line.nil?()
            line.strip!
            next if line==''
            if line=="-------------------------" then
              if entry.nil?() then
                debug "Starting first entry"
                entry=Iof::Parser.empty_cdp_neighbor()
                entry[:seen_by]=@target.name.clone()
                next
              else
                @cdp_neighbors_by_interface[entry[:seen_at_interface].intern]=Array.new() unless @cdp_neighbors_by_interface.include?(entry[:seen_at_interface].intern)
                @cdp_neighbors_by_interface[entry[:seen_at_interface].intern] << entry
                entry[:ipv4_addresses]=entry[:ipv4_addresses].uniq.sort.compact
                entry[:mac_addresses]=entry[:mac_addresses].uniq.sort.compact
                entry[:ipv4_addresses].each do |i|
                  entry[:mac_addresses].each do |m|
                    @target_handler.assign_mac_ipv4(m, i)
                  end unless entry[:mac_addresses].nil?()
                end unless entry[:ipv4_addresses].nil?()
                debug "New entry started"
                entry=Iof::Parser.empty_cdp_neighbor()
                entry[:seen_by]=@target.name.clone()
              end
            else
              if line.match(/^Device ID/) then
                parsing_addresses=false
                entry[:name]=line.sub(/^Device ID:\s*/,'').strip
                if entry[:name].downcase.match(/^ata/) || entry[:name].downcase.match(/^sep/)
                  mac=Iof::Data.mac_plain(entry[:name].downcase.sub(/^sep/,'').sub(/^ata/,''))
                  unless entry[:mac_addresses].include?(mac) || mac.nil?() || mac==''
                    entry[:mac_addresses] << mac
                    @target_handler.add_mac(mac)
                  end
                end
                entry[:name].sub!(/.+\((.+)\)/,"\\1") if entry[:name].match(/.+\((.+)\)/)
              elsif line.match(/^Entry address\(es\)/) then
                parsing_addresses=true
              elsif (line.match(/^IP address:/)) && parsing_addresses then
                address=line.sub(/^IP address:\s*/,'').strip()
                entry[:ipv4_addresses] << address
              elsif (line.match(/^IP address:/)) && !parsing_addresses then
                address=line.sub(/^IP address:\s*/,'').strip()
                entry[:management_ipv4_addresses] << address
              elsif line.match(/^Platform:/) then
                platform,capabilities=line.split(',')
                entry[:platform]=platform.strip().sub(/^Platform:\s*/,'').strip()
                p=entry[:platform].downcase
                if p.match(/^cisco/) || p.match(/^ap340/) || p.match(/^air/) || p.match(/^n5k-c/) || p.match(/^n7k-c/) || p.match(/^ws-c/) || p.match(/s[fg]-[1-9]00/)
                  #modern devices || wireless components || nexus5000 devices || nexus7000 devices ||  old catalyst switches
                  entry[:vendor]="cisco"
                  p.sub!(/^cisco/,'')
                  p.strip!
                elsif p.match(/^ms/)
                  #seems to be a microsens device
                  entry[:vendor]="microsens"
                elsif p.match(/^mikrotik/)
                  entry[:device_class]=:switch
#                  entry[:model]=:cdp_switch
                  entry[:vendor]="mikrotik"
                end
                if p.match(/^air-wlc/) || p.match(/^air-ap/) || p.match(/^air-lap/)
                  entry[:device_class]=:wireless_component
                  entry[:model]=p.sub(/air-/,'').sub(/-.*/,'')
                elsif p.match(/^ap340/)
                  entry[:device_class]=:wireless_component
                  entry[:model]=p.sub(/ap/,'')
                elsif p.match(/cigesm/)
                  entry[:device_class]=:switch
                  entry[:model]="cigesm"
                elsif p.match(/^ata/) || p.match(/^ip/) || p.match(/^vg/)
                  entry[:device_class]=:voice_component
                  entry[:model]=p
                elsif p.match(/^n[57]k-c/)
                  entry[:device_class]=:switch
                  entry[:model]=p.sub(/n[57]-c/,'n').sub(/-.*/,'')
                elsif p.match(/s[fg][1-9]00/)
                  entry[:device_class]=:switch
                  entry[:model]=p.sub(/-.*/,'')
                elsif p.match(/^ws-c/)
                  entry[:device_class]=:switch
                  entry[:model]=p.sub(/ws-c/,'').sub(/-.*/,'')
                elsif p.match(/^1900/)
                  entry[:device_class]=:switch
                  entry[:model]=p.sub(/-.*/,'')
                else
                  entry[:device_class]=:unkown
                  entry[:model]=p
                end if entry[:vendor]=="cisco"
                entry[:capabilities]=parse_capabilities(capabilities.strip().sub(/^Capabilities:\s*/,'').strip())
                parsing_addresses=false
              elsif line.match(/^Interface:/) then
                own,foreign=line.split(',')
                own.strip!
                own.sub!(/^Interface:\s*/,'')
                own.strip!
                entry[:seen_at_interface]=own
                foreign.strip!
                foreign.sub!(/^Port ID \(outgoing port\):\s*/,'')
                foreign.strip!
                entry[:interface]=foreign
                parsing_addresses=false
              elsif line.match(/^Second Port Status/) then
                #IP Phones with internal switch only
                sec=line.sub!(/^Second Port Status:\s*/,'')
                if sec.downcase()=="up" then
                  entry[:second_interface_up]=true
                else
                  entry[:second_interface_up]=false
                end
              elsif line.match(/Power drawn/)
                drawn=line.sub(/^Power drawn: /,'').sub(/ Watts.*$/,'')
                entry[:reports_power]=(drawn.to_f!=0.0 && !drawn.index(drawn.to_f().to_s()).nil?())
              elsif line.match(/Power request levels/)
                levels=line.gsub(/^Power request levels are:/,'')
                levels.split().each do |l|
                  begin
                    l_i=l.to_i
                    entry[:powered]||=(l_i>0)
                  rescue => e
                    error "Failed to request level '#{l}' in line '#{line}'", e
                  end
                end
              #else
                #debug "Ignoring line #{line}"
              end
            end
          end
          if entry.nil?() then
            debug "No neighbours found"
          else
            @cdp_neighbors_by_interface[entry[:seen_at_interface].intern]=Array.new() unless @cdp_neighbors_by_interface.include?(entry[:seen_at_interface].intern)
            @cdp_neighbors_by_interface[entry[:seen_at_interface].intern] << entry
            entry[:ipv4_addresses]=entry[:ipv4_addresses].uniq.sort.compact
            entry[:mac_addresses]=entry[:mac_addresses].uniq.sort.compact
            entry[:ipv4_addresses].each do |i|
              entry[:mac_addresses].each do |m|
                @target_handler.assign_mac_ipv4(m, i)
              end unless entry[:mac_addresses].nil?()
            end unless entry[:ipv4_addresses].nil?()
            entry=nil
          end

          @cdp_neighbors_by_interface.each do |interface,neighbors|
            neighbors.each do |neighbor|
              target=nil

              #searching target
              neighbor[:ipv4_addresses].each do |a|
                t=@target_handler.get_target_by_ipv4(a)
                if t.nil?() && !target.nil?()
                  debug "Found new address #{a} for existing target #{target.description()}"
                  target=target.add_ipv4(@target_handler, a)
                elsif !t.nil?() && target.nil?()
                  target=t
                elsif target.__id__!=t.__id__
                  warn "Got conflicting targets for reported IPv4 addresses; triggering merge of #{target.description()} and #{t.description()}"
                  target=target.add_ipv4(@target_handler, a)
                end
              end
              neighbor[:mac_addresses].each do |a|
                t=@target_handler.get_target_by_mac(a)
                if t.nil?() && !target.nil?()
                  debug "Found new address #{a} for existing target #{target.description()}"
                  target=target.add_mac(@target_handler, a)
                elsif !t.nil?() && target.nil?()
                  target=t
                elsif target.__id__!=t.__id__
                  warn "Got conflicting targets for reported addresses; triggering merge of #{target.description()} and #{t.description()}"
                  target=target.add_mac(@target_handler, a)
                end
              end
              t=@target_handler.get_target_by_name(neighbor[:name])
              if t.nil?() && !target.nil?()
                debug "Found new name #{neighbor[:name]} for existing target #{target.description()}"
                target=target.add_name(@target_handler, neighbor[:name])
              elsif !t.nil?() && target.nil?()
                target=t
              elsif target.__id__!=t.__id__
                warn "Got conflicting targets for reported addresses and name; triggering merge of #{target.description()} and #{t.description()}"
                target=target.add_name(@target_handler, neighbor[:name])
                target=target.add_name(@target_handler, neighbor[:name],:own)
              end

              if target.nil?()
                target=Iof::Target::Item.new()
                neighbor[:ipv4_addresses].each do |a|
                  target=target.add_ipv4(@target_handler, a)
                end
                neighbor[:mac_addresses].each do |a|
                  target=target.add_mac(@target_handler, a)
                end
                target=target.add_name(@target_handler, neighbor[:name])
              end

              if target.names.empty?()
                target=target.add_name(@target_handler,SecureRandom.uuid())
                warn "Target #{target.description} had no name; generated UUID #{target.names[0]}"
              end

              target.special_mutex.synchronize {
                target.special[:cdp_sightings]=Hash.new() unless target.special.include?(:cdp_sightings)
                target.special[:cdp_sightings][@target.name.intern]=Array.new() unless target.special[:cdp_sightings].include?(@target.name.intern)
                target.special[:cdp_sightings][@target.name.intern]<<neighbor

                if neighbor[:device_class]==:wireless_component
                  target.device_vendor=neighbor[:vendor] unless (neighbor[:vendor].nil?() || neighbor[:vendor]=='')
                  target.device_type=:cdp_ap
                elsif neighbor[:device_class]==:voice_component
                  target.device_vendor=neighbor[:vendor] unless (neighbor[:vendor].nil?() || neighbor[:vendor]=='')
                  target.device_type=:cdp_phone
                elsif neighbor[:device_class]==:switch
                  target.device_vendor=neighbor[:vendor] unless (neighbor[:vendor].nil?() || neighbor[:vendor]=='')
                  target.device_type=neighbor[:model] unless (target.device_type==:cdp_switch || neighbor[:model].nil?() || neighbor[:model]=='')
                  if target.status_preparse()==false && target.device_type!=:cdp_switch
                    warn "Preparse for #{target.description()} failed; switching to cdp mode"
                    #preparse or earlier steps were not successfull; forcing handling via CDP parser
                    target.device_type=:cdp_switch
                    #TODO enable identify if implemented
                    #target.identify(nil)
                    target.status_identify=true
                    target.status_prepare=nil
                    target.status_collect=nil
                    target.status_preparse=nil
                    target.status_parse=nil
                    target.status_postparse=nil
                    target.status_cleanup=nil
                    target.prepare(nil)
                    target.collect(nil, nil)
                    target.preparse(@target_handler)
                  end
                end
                if target.filtered?() || target.special[:filtered_by_runner]
                  warn "Filtered target #{target.description()} was seen as cdp neighbor; forcing cdp mode"
                  target.filter_protected_by_cdp()
                  target.device_type=:cdp_switch if neighbor[:device_class]==:switch
                  #TODO enable identify if implemented
                  target.status_identify=true
                  target.status_prepare=nil
                  target.status_collect=nil
                  target.status_preparse=nil
                  target.status_parse=nil
                  target.status_postparse=nil
                  target.status_cleanup=nil
                  target.prepare(nil)
                  target.collect(nil, nil)
                  target.preparse(@target_handler)
                end
                target.filter_protected_by_cdp()
                if @mac_by_interface.include?(neighbor[:seen_at_interface].intern)
                  target.special[:mac_sightings]=Hash.new() unless target.special.include?(:mac_sightings)
                  target.special[:mac_sightings][@target.name.intern]=Array.new() unless target.special[:mac_sightings].include?([@target.name.intern])
                  target.special[:mac_sightings][@target.name.intern]+=@mac_by_interface[neighbor[:seen_at_interface].intern]
                end
              }
            end unless neighbors.nil?()
          end
        when :parse
          @cdp_neighbors_by_interface.each do |interface,neighbors|
            neighbors.each do |neighbor|
              next unless neighbor[:device_class]==:switch
              target=nil
              target=@target_handler.get_target_by_name(neighbor[:name])
              neighbor[:mac_addresses].each do |a|
                t=@target_handler.get_target_by_mac(a)
                unless t.nil?()
                  target=t
                  break
                end
              end if target.nil?()
              neighbor[:ipv4_addresses].each do |a|
                t=@target_handler.get_target_by_ipv4(a)
                unless t.nil?()
                  target=t
                  break
                end
              end if target.nil?()
              if target.nil?()
                warn "Found no target for neighbor; this may be caused by active runner filters; #{neighbor.inspect()}"
              else
                if !target.status_preparse() && target.device_type!=:cdp_switch
                  warn "Preparse for #{target.description()} failed; switching to cdp mode"
                  #preparse or earlier steps were not successfull; forcing handling via CDP parser
                  target.device_type=:cdp_switch
                  #TODO enable identify if implemented
                  #target.identify(nil)
                  target.status_identify=true
                  target.status_prepare=nil
                  target.status_collect=nil
                  target.status_preparse=nil
                  target.status_parse=nil
                  target.status_postparse=nil
                  target.status_cleanup=nil
                  target.prepare(nil)
                  target.collect(nil, nil)
                  target.preparse(@target_handler)
                end
              end
            end unless neighbors.nil?()
          end
        when :postparse
          #nothing to do
        end
      end

      def parse_etherchannel(label,command,result,step)
        host_name=@target.name
        #TODO handle link aggregation
        return if result[:stdout].nil?()
        channel_line=true
        header=true
        channel=nil
        result[:stdout].split("\n").each do |line|
          next if line.nil?()
          next if line.strip()==''
          if header
            header=false if line.match(/-+\+-+\+-+\+-+/)
            next
          end
          channel_line=line.match(/^ +/).nil?()
          line.strip!()
          if channel_line
            Ghun::Output::DISTRIBUTED_HANDLER[:etherchannels].append_data(channel) if Ghun::Output::DISTRIBUTED_HANDLER[:etherchannels] && !channel.nil?()
            sp_line=line.split(' ')
            interfaces=sp_line[3..-1]
            channel=Hash.new()
            channel[:switch]=host_name
            channel[:channel]=expand_interface_shortname(sp_line[1].sub(/\(.*\)/,''))
            @interfaces[channel[:channel].intern]=Hash.new() unless @interfaces.include?(channel[:channel].intern)
            channel[:channel_flags]=sp_line[1].sub(/.*\((.*)\)/,'\1')
            if channel[:channel_flags].match(/S/)
              channel[:channel_layer]='2'
              @interfaces[channel[:channel].intern][:layer]=2 unless @interfaces[channel[:channel].intern].include?(:layer)
            elsif channel[:channel_flags].match(/R/)
              channel[:channel_layer]='3'
              @interfaces[channel[:channel].intern][:layer]=3 unless @interfaces[channel[:channel].intern].include?(:layer)
            else
              channel[:channel_layer]='unknown'
            end
            channel[:channel_protocol]=sp_line[2]
            channel[:channel_protocol]="unknown" if sp_line[2]=='-'
            channel[:number_active]=0
            channel[:number_inactive]=0
            channel[:active]=""
            channel[:inactive]=""
            @port_channel_to_interface[channel[:channel].intern]=Array.new()
            interfaces.each do |i|
              long_i=expand_interface_shortname(i.sub(/\(.*\)/,''))
              @port_channel_to_interface[channel[:channel].intern] << long_i
              if @interface_to_port_channel.include?(long_i.intern)
                warn "Interface #{long_i} is used in two port-channels: #{@interface_to_port_channel[long_i.intern]} and #{channel[:channel]}"
              else
                @interface_to_port_channel[long_i.intern]=channel[:channel]
              end
              flags=i.sub(/.*\((.*)\)/,'\1')
              if flags.match(/P/)
                channel[:number_active]+=1
                channel[:active]+=" #{long_i}(#{flags})"
              else
                channel[:number_inactive]+=1
                channel[:inactive]+=" #{long_i}(#{flags})"
              end
            end
            channel[:active].strip!()
            channel[:inactive].strip!()
          else
            interfaces=line.split(' ')
            interfaces.each do |i|
              long_i=expand_interface_shortname(i.sub(/\(.*\)/,''))
              flags=i.sub(/.*\((.*)\)/,'\1')
              if flags.match(/P/)
                channel[:number_active]+=1
                channel[:active]+=" #{long_i}(#{flags})"
              else
                channel[:number_inactive]+=1
                channel[:inactive]+=" #{long_i}(#{flags})"
              end
            end
            channel[:active].strip!()
            channel[:inactive].strip!()
          end
        end
        Ghun::Output::DISTRIBUTED_HANDLER[:etherchannels].append_data(channel) if Ghun::Output::DISTRIBUTED_HANDLER[:etherchannels] && !channel.nil?()
      end

      def parse_cdp_interfaces(label,command,result,step)
        @temporary_storage[:cdp_interfaces]=Hash.new()
        result[:stdout].split("\n").each do |line|
          next if line.nil?()
          next if line==""
          next if line.match(/^\s+/)
          line.strip!()
          sp_line=line.split(' ')
          entry=Hash.new
          name=sp_line[0].intern
          if @temporary_storage[:cdp_interfaces].include?(name)
            warn "Duplicate entry for interface #{name.to_s} in output of 'show cdp interfaces' on #{@target.description()}"
            next
          else
            @temporary_storage[:cdp_interfaces][name]=Hash.new()
            @temporary_storage[:cdp_interfaces][name][:up]=true
            @temporary_storage[:cdp_interfaces][name][:admin_up]=true
            @temporary_storage[:cdp_interfaces][name][:monitor]=false
          end
          down=!(line.match(/down/).nil?)
          up=!(line.match(/up/).nil?)
          admin_down=!(line.match(/administratively down/).nil?)
          if up && down then
            if line.match(/\(monitoring\)/) then
              info Ghun::Log::Type::REPORT, "Interface '#{name.to_s}' on device #{@target.description()} is in monitoring mode"
              info "Interface '#{name.to_s}' on device #{@target.description()} is in monitoring mode"
              @temporary_storage[:cdp_interfaces][name][:monitor]=true
              @temporary_storage[:cdp_interfaces][name][:up]=false
            else
              warn "Got conflicting state info for interface '#{name.to_s}' on device #{@target.description()}; got '#{line}'; assuming monitoring mode"
              error Ghun::Log::Type::REPORT, "Got conflicting state info for interface '#{name.to_s}' on device #{@target.description()}; got '#{line}'; assuming monitoring mode"
              @temporary_storage[:cdp_interfaces][name][:monitor]=true
              @temporary_storage[:cdp_interfaces][name][:up]=true
              @temporary_storage[:cdp_interfaces][name][:up]=false
            end
          elsif up then
            @temporary_storage[:cdp_interfaces][name][:up]=true
          elsif down then
            @temporary_storage[:cdp_interfaces][name][:up]=false
          else
            error "This should never happen", "Interface #{entry.to_s} is neither up nor down", "Assuming garbage in command output"
            @temporary_storage[:cdp_interfaces].delete(name)
            next
          end
          if admin_down then
            @temporary_storage[:cdp_interfaces][name][:admin_up]=false
          else
            @temporary_storage[:cdp_interfaces][name][:admin_up]=true
          end
          debug "Found cdp enabled interface #{name} on device #{@target.description()}: #{@temporary_storage[:cdp_interfaces][name]}"
        end
      end


      def parse_power(label,command,result,step)
        if result[:stdout].nil?()
          @power_outgoing=nil
          return
        end
        #TODO report power consumption to corresponding target
        @power_outgoing[:interfaces]=Hash.new()
        header=false
        interface_header1=nil
        interface_header2=nil
        interface_section=false
        footer=false
        interface_table=Array.new()
        result[:stdout].split("\n").each do |line|
          next if line.nil?()
          next if line.match(/^\s*$/)
          header=true unless header || interface_section || interface_header1 || interface_header2
          if line.match(/^Interface/) && header && !interface_section then
            interface_header1=line.dup()
            header=false
          elsif !interface_section && !interface_header1.nil?() && interface_header2.nil?()
            interface_header2=line.dup()
          elsif !interface_section && !interface_header1.nil?() && !interface_header2.nil?()
            simple_header=!(interface_header2.match(/^ +\(Watts\) *$/).nil?())
            cols=line.split()
            names=interface_header1.split()
            names_offset=0
            decrease_offset=false
            pos=0
            column_number=0
            cols.each do |c|
              col=Hash.new()
              col[:start]=pos
              col[:stop]=pos+c.length()-1
              if decrease_offset then
                names_offset+=1
                decrease_offset=false
                col[:name]=interface_header2[(col[:start]..col[:stop])].strip
                col[:simple]=false
              else
                if interface_header1[(col[:start]..col[:stop])].match(/Power/) && !simple_header
                  col[:name]=interface_header2[(col[:start]..col[:stop])].strip
                  decrease_offset=true
                  col[:simple]=false
                else
                  col[:name]=interface_header1[(col[:start]..col[:stop])].strip
                  col[:simple]=true
                end
              end
              pos+=c.length()+1
              interface_table[column_number]=col
              column_number+=1
            end
            #workaround for misformatted table header on some software version of 6500 series
            interface_table.each do |i|
              next unless i[:simple]
              next unless i[:name].match(/ +/)
              start=i[:start]
              stop=i[:stop]
              pos=0
              name=i[:name].dup()
              names=name.split().compact()
              (0..(names.size()-1)).each do |n|
                if n==0 then
                  i[:name]=names[n].strip()
                  i[:stop]=start+name.index(names[n+1])-2
                  pos=i[:stop]+2
                  #at least two columns; no check for last column necessary
                elsif n==(names.size()-1)
                  i=Hash.new()
                  #last column
                  i[:name]=names[n]
                  i[:start]=pos
                  i[:stop]=stop
                  interface_table << i
                else
                  i=Hash.new()
                  i[:name]=names[n]
                  i[:start]=pos
                  i[:stop]=start+name.index(names[n+1])-2
                  pos=i[:stop]+2
                  interface_table << i
                end
              end
            end
            interface_table.each do |i|
              i[:field]=case i[:name]
              when /^To /
                :power_to_device
              when /^From/
                :power_from_switch
              when /Act(ual)*Consumption/
                :consumption
              when "Admin", "Power", "Interface", "Priority", "Device", "Class", "Police", "Max"
                i[:name].downcase.intern
              when "Oper"
                :operational
              end
            end
            interface_section=true
            interface_header1=nil
            interface_header2=nil
          elsif line.match(/^Available/) && header then
            available=line.strip.sub(/^Available:/,'').sub(/\(w\).*/,'')
            used=line.strip.sub(/.*Used:/,'').sub(/\(w\).*/,'')
            remaining=line.strip.sub(/.*Remaining:/,'').sub(/\(w\).*/,'')
            begin
              if (available.to_f - used.to_f - remaining.to_f).abs() < 1 && available.to_f().to_s().match(/#{available}/) && used.to_f().to_s().match(/#{used}/) && remaining.to_f().to_s().match(/#{remaining}/)
                debug "Got global power consumption values (#{available},#{used},#{remaining}) for #{@target.description()}"
                @power_outgoing[:available]=available
                @power_outgoing[:used]=used
                @power_outgoing[:remaining]=remaining
              else
                warn "Could not parse line '#{line}', that should contain global power consumption data"
              end
            rescue => e
              error "Failed to check global power consumption values ignoring", e
            end
          elsif interface_section
            i=Hash.new()
            footer=true if line.match(/^-+/)
            unless footer
              interface_table.each do |c|
                next if c[:field].nil?() || c[:start].nil?() || c[:stop].nil?()
                break if line[(c[:start]..c[:stop])].nil?()
                i[c[:field]]=line[(c[:start]..c[:stop])].strip
              end
              unless i[:interface].nil?() || i[:interface]==""
                i[:interface]=expand_interface_shortname(i[:interface])
                error "Multiple entries for interface #{i[:interface]} on #{@target.description()}; ignoring #{i.inspect}" if @power_outgoing[:interfaces].include?(i[:interface].intern)
                @power_outgoing[:interfaces][i[:interface].intern] = i unless @power_outgoing[:interfaces].include?(i[:interface].intern)
              end

            else
              interface_section=false
            end
          elsif footer
            debug "Ignoring footer line '#{line}'"
          end
        end
        @power_outgoing[:interfaces].each do |interface,poe_status|
          unless @interfaces.include?(interface)
            @interfaces[interface]=Hash.new()
            @interfaces[interface][:name]=interface.to_s
          end
          if poe_status.nil?()
            #interface seems to be not PoE capable; do nothing
            #no need for checking neighbors
            next
          elsif poe_status[:power_from_switch]=="0.0" || poe_status[:power_from_switch]=="0"
            #interface seems to be PoE capable, but does not power any device
            @interfaces[interface][:poe_capable]=true
            @interfaces[interface][:powers_device]=false
            #no need for checking neighbors
            next
          elsif poe_status.include?(:power_from_switch)
            #interface seems to be PoE capable and seems to power a connected device
            @interfaces[interface][:poe_capable]=true
            @interfaces[interface][:powers_device]=true
          end

          cdp_neighbors=@cdp_neighbors_by_interface[interface]
          if cdp_neighbors.nil?() || cdp_neighbors.size()==0
            # no neighbors to handle
            next
          else
            powered_neighbors=Array.new()
            cdp_neighbors.each do |cdp_neighbor|
              powered_neighbors << cdp_neighbor if cdp_neighbor[:powered]
            end
            if powered_neighbors.empty?()
              #no powered devices
              next
            elsif powered_neighbors.size()>1
              #more than one powered device
              warn "Found more than one powered device at interface #{interface.to_s} on device #{@target.description()}"
              next
            else
              neighbor=@target_handler.get_target_by_name(powered_neighbors[0][:name]) || @target_handler.get_target_by_ipv4(powered_neighbors[0][:ipv4_addresses][0])
              neighbor.special_mutex.synchronize {
                neighbor.special[:poe_powered]=true
                neighbor.special[:poe_capable]=true
                neighbor.special[:poe_power_consumption]=poe_status[:power_to_device]
                neighbor.special[:poe_powered_by]="#{@target.name}___#{interface}"
              } unless neighbor.nil?() || neighbor==''
            end
          end
        end unless @power_outgoing.nil?()
      end

      def parse_configuration(label,command,result,step)
        #TODO add support for mac access-list
        #TODO add support for ipv6 access-list
        #TODO send send vlans to cdp neighbors
        section=nil
        section_name=""
        section_item=nil
        interfaces=Array.new()
        acls=Hash.new()
        acl_refs=Hash.new()
        ipv4_forwards=Array.new()
        ipv6_forwards=Array.new()

        result[:stdout].split("\n").each do |line|
          next if line.nil?()
          if line.match(/^\s*$/) then
            debug "Ignoring empty line"
          elsif line=='!' || !(line.match(/^\s+/) || section.nil?()) then
            if section.nil? then
              debug "Ignoring line #{line}"
            else
              case section
              when :interface
                debug "End of interface section '#{section_name}'"
                interfaces << section_item
                section_item=nil
              when :extended_acl
                 debug "End of extended acl section '#{section_name}'"
                 warn "Duplicate acl entry; using last one" unless acls[section_item[:name].intern].nil?
                 acls[section_item[:name].intern]=section_item[:rules]
                 acl_refs[section_item[:name].intern]=0 unless acl_refs.include?(section_item[:name].intern)
                 debug "Completed acl with name '#{section_item[:name]}'"
                 section_item=nil
              when :ignored
                debug "End of ignored section of type '#{section_name}'"
              end
              section=nil
              section_name=nil
            end
          elsif line[0]=='!' then
            debug "Ignoring comment line #{line}"
          elsif line[0]==' ' then
            if section.nil? then
              debug "Found indented line without open section: #{line}"
            else
              sp_line=line.strip.split(' ')
              inverse=false

              if sp_line[0]=='no' then
                inverse=true
                sp_line.shift
              end

              sp_line.each { |s| s.strip!() }
              case section
              when :interface
                if sp_line[0]=="shutdown" then
                  section_item[:shutdown]=!inverse
                elsif sp_line[0]=="description" then
                  section_item[:description]=sp_line[1..(sp_line.size-1)].join(' ')
                elsif sp_line[0]=="ip" || sp_line[0]=="ipv6" then
                  next if sp_line[1]=='dhcp' && sp_line[2]=='snooping'
                  if sp_line[0]=="ipv6"
                    prefix="ipv6"
                  else
                    prefix="ipv4"
                  end
                  sp_line.shift
                  if sp_line[0]=="address" && inverse then
                    section_item.delete("#{prefix}_addresses".intern)
                  elsif sp_line[0]=="address" && !inverse then
                    section_item["#{prefix}_addresses".intern]=Array.new if section_item["#{prefix}_addresses".intern].nil?
                    section_item["#{prefix}_addresses".intern] << {:address => sp_line[1], :netmask => sp_line[2]}
                  elsif sp_line[0]=="access-group" || sp_line[0]=="traffic-filter" then
                    #IPv4 uses access-group; IPv6 uses traffic-filter instead
                    section_item[:acl]=Array.new() if section_item[:acl].nil?
                    section_item[:acl] << {:id => sp_line[1].intern, :direction => sp_line[2]}
                    acl_refs[sp_line[1].intern]=0 unless acl_refs.include?(sp_line[1].intern)
                    acl_refs[sp_line[1].intern]+=1
                    debug "Found #{sp_line[0]} for interface '#{section_item[:name]}': '#{section_item[:acl].inspect}'"
                  elsif sp_line[0]=='enable' then
                    #IPv6 only
                    section_item["#{prefix}_enable".intern]=!inverse
                  elsif sp_line[0]=='nd' && sp_line[1]=='prefix'
                    #IPv6 only
                    address,mask,garbage=sp_line[2].split('/')
                    section_item[:ipv6_prefix] = {:address => address, :netmask => mask}
                  else
                    debug "Ignoring unknown line '#{line}' for section '#{section_name}'"
                  end
                elsif sp_line[0]=="switchport" then
                  sp_line.shift
                  if sp_line[0]=="" then
                    section_item[:virtual]=false unless inverse
                  elsif sp_line[0]=="voice" && sp_line[1]=="vlan" then
                    section_item[:voice_configuration_items]=true
                    section_item[:voice_vlan]=sp_line[2]
                  elsif sp_line[0]=="voice" then
                    section_item[:voice_configuration_items]=true
                    debug "Ignoring unknown 'switchport voice' statement: '#{line}'"
                  elsif sp_line[0]=="access" && sp_line[1]=="vlan" then
                    section_item[:access_configuration_items]=true
                    section_item[:access_vlan]=sp_line[2]
                  elsif sp_line[0]=="access" then
                    debug "Ignoring unknown 'switchport access' statement: '#{line}'"
                    section_item[:access_configuration_items]=true
                  elsif sp_line[0]=="trunk" && sp_line[1]=="native" && sp_line[2]=="vlan" then
                    section_item[:trunk_configuration_items]=true
                    section_item[:native_vlan]=sp_line[3]
                  elsif sp_line[0]=="trunk" && sp_line[1]=="allowed" && sp_line[2]=="vlan" && sp_line[3]=='add' then
                    section_item[:trunk_configuration_items]=true
                    section_item[:allowed_vlans]<<(sp_line[4..(sp_line.size-1)].join(' '))
                  elsif sp_line[0]=="trunk" && sp_line[1]=="allowed" && sp_line[2]=="vlan" then
                    section_item[:trunk_configuration_items]=true
                    section_item[:allowed_vlans]=Array.new()
                    section_item[:allowed_vlans]<<(sp_line[3..(sp_line.size-1)].join(' '))
                  elsif sp_line[0]=="trunk" then
                    debug "Ignoring unknown 'switchport trunk' statement: '#{line}'"
                    section_item[:trunk_configuration_items]=true
                  elsif sp_line[0]=="mode" then
                    section_item[:mode]=sp_line[1].intern
                    section_item[:mode]=:trunk if section_item[:mode]==:dynamic
                    if section_item[:mode]==:access then
                      section_item[:access_configuration_items]=true
                    elsif section_item[:mode]==:trunk then
                      section_item[:trunk_configuration_items]=true
                    end
                  else
                    debug "Ignoring unknown line '#{line}' for section '#{section_name}'"
                  end
                elsif sp_line[0]=="channel-group" then
                  section_item[:is_in_channel_group]=sp_line[1]
                  debug "Interface #{section_item[:name]} is port of channel group '#{section_item[:is_in_channel_group]}'"
                end
              when :extended_acl
                tmp=parse_acl(sp_line)
                section_item[:rules] << tmp unless tmp.nil?
              when :ignored
              else
                debug "Ignoring line in unknown section of type '#{section.to_s}': '#{line}'"
              end
            end
          elsif line[0]!=' ' && !section.nil?()
            sp_line=line.strip.split(' ')
            warn "Found non indented line in not ignored open section '#{section_name}': #{line}" unless section==:ignored
          else
            sp_line=line.strip.split(' ')
            case sp_line[0]
            when 'interface'
              debug "Started interface section"
              section=:interface
              section_name=line
              section_item=Hash.new
              section_item[:name]=sp_line[1]
              @temporary_storage[:long_interface_names]=@temporary_storage[:long_interface_names] || !section_item[:name].match(/[0-9]+\/[0-9]+\/[0-9]+$/).nil?()
              section_item[:virtual]=!(section_item[:name].match(/^Vlan/).nil?())
              section_item[:virtual_id]=section_item[:name].sub(/^Vlan/,'') if section_item[:virtual]
              section_item[:is_channel_group]=section_item[:name].sub(/^Port-channel/,'') if section_item[:name][/Port-channel/]
              section_item[:mode]=:trunk if section_item[:is_channel_group]
              debug "Interface #{section_item[:name]} is Port channel for channel group #{section_item[:is_channel_group]}" if section_item.include?(:is_channel_group)
              debug "Interface #{section_item[:name]} is Virtual interface for vlan #{section_item[:virtual_id]}" if section_item[:virtual]
              section_item[:ipv6_enable]=false
            when 'access-list'
              debug "Found acl line"
              name=sp_line[1].intern
              if acls[name].nil?()
                acl_refs[name]=0
                acls[name]=Array.new
              end
              tmp=parse_acl(sp_line[2..(sp_line.size-1)])
              acls[name]<<tmp unless tmp.nil?
            when 'ip'
              if sp_line[1]=='access-list' then
                debug "Started extended acl section"
                section=:extended_acl
                section_name=sp_line[3..(sp_line.size-1)].join('_').intern
                section_item=Hash.new
                section_item[:name]=sp_line[3..(sp_line.size-1)].join('_')
                section_item[:type]=sp_line[2]
                section_item[:rules]=Array.new
              elsif sp_line[1]=='route' then
                unless sp_line[2]=='profile' then
                  if sp_line[2]=='vrf'
                    warn "Support for VRF instances is still missing; ignoring line #{line}"
                  else
                    forward=Iof::Data::NO::Layer3Forward.new()
                    if sp_line[3]=="255.255.255.255" then
                      forward[:no_destination_ipv4_address]=@data_handler.get_ipv4_address_name(sp_line[2])
                    else
                      forward[:no_destination_ipv4_network]=@data_handler.get_ipv4_network_name(sp_line[2],Iof::Data.ipv4_to_netmask(sp_line[3]))
                    end
                    forward[:no_next_hop_ipv4_address]=@data_handler.get_ipv4_address_name(sp_line[4])
                    ipv4_forwards << forward
                    forward=nil
                  end
                end
              else
                debug "Ignoring unknown ip line"
              end
            when 'ipv6'
              if sp_line[1]=='route' then
                unless sp_line[2]=='profile' then
                  if sp_line[2]=='vrf'
                    warn "Support for VRF instances is still missing; ignoring line #{line}"
                  else
                    forward=Iof::Data::NO::Layer3Forward.new()
                    address,netmask,garbage=sp_line[2].split('/')
                    if netmask=="128" then
                      forward[:no_destination_ipv6_address]=@data_handler.get_ipv6_address_name(address)
                    else
                      forward[:no_destination_ipv6_network]=@data_handler.get_ipv6_network_name(address,netmask)
                    end
                    forward[:no_next_hop_ipv6_address]=@data_handler.get_ipv6_address_name(sp_line[3])
                    ipv6_forwards << forward
                    forward=nil
                  end
                end
              else
                debug "Ignoring unknown ipv6 line"
              end
            when 'line'
              debug "Started line section; ignoring"
              section=:ignored
              section_name="line"
            when 'banner'
              debug "Found banner section; ignoring"
              section=:ignored
              section_name="banner"
            when 'vlan'
              debug "Started vlan section; ignoring"
              section=:ignored
              section_name="banner"
            when 'firewall'
              debug "Found firewall line"
            end
          end
        end

        acls.each do |id,acl|
          if acl_refs[id].nil?() || acl_refs[id]<=0
            info "ACL #{id.to_s} is not used anywhere; skipping"
          else
            deny_all=Iof::Data::NO::Layer4ACLEntry.new()
            deny_all[:no_acl_action]=:deny
            acl << deny_all
            @acls_list[id]=acl_array_to_list(acl)
            @acls_yaml[id]=@acls_list[id].to_iof_yaml
          end
        end
        acls=nil

        if ipv4_forwards.nil?() || ipv4_forwards.empty?()
          @ipv4_forwards=nil
        else
          @ipv4_forwards=forwards_array_to_list(ipv4_forwards,:ipv4)
          @ipv4_forwards[:no_yaml_representation]=@ipv4_forwards.to_iof_yaml()
        end

        if ipv6_forwards.nil?() || ipv6_forwards.empty?()
          @ipv6_forwards=nil
        else
          @ipv6_forwards=forwards_array_to_list(ipv6_forwards,:ipv6)
          @ipv6_forwards[:no_yaml_representation]=@ipv6_forwards.to_iof_yaml()
        end

        interfaces.each do |interface|
          if interface[:ipv6_enable] then
            interface.delete(:ipv6_enable)
          else
            interface.delete(:ipv6_addresses)
            interface.delete(:ipv6_enable)
          end
          if interface[:mode].nil?() && (interface[:access_configuration_items] || interface[:trunk_configuration_items] || interface[:voice_configuration_items]) then
            error Ghun::Log::Type::REPORT, "Interface #{interface[:name]} on device #{@target.description()} has switchport configuration without explicit mode setting"
            warn "Interface #{interface[:name]} on device #{@target.description()} has switchport configuration without explicit mode setting"
            if interface[:access_configuration_items] && !interface[:trunk_configuration_items]
              interface[:mode]=:access
            elsif !interface[:access_configuration_items] && interface[:trunk_configuration_items]
              interface[:mode]=:trunk
            elsif !interface[:access_configuration_items] && !interface[:trunk_configuration_items]
              interface[:mode]=:trunk
            elsif interface[:access_configuration_items] && interface[:trunk_configuration_items]
              interface[:mode]=:trunk
            end
            if !interface[:shutdown] && !(@temporary_storage[:cdp_interfaces].include?(interface[:name].intern) && (!@temporary_storage[:cdp_interfaces][interface[:name].intern][:up] || !@temporary_storage[:cdp_interfaces][interface[:name].intern][:admin_up]))
              info Ghun::Log::Type::REPORT, "Assuming mode '#{interface[:mode]}' for interface #{interface[:name]} on device #{@target.description()}"
              info Ghun::Log::Type::REPORT, "Interface #{interface[:name]} on device #{@target.description()} has#{(interface[:access_configuration_items])?" ":" no "}access mode configuration items"
              info Ghun::Log::Type::REPORT, "Interface #{interface[:name]} on device #{@target.description()} has#{(interface[:trunk_configuration_items])?" ":" no "}trunk mode configuration items"
              info Ghun::Log::Type::REPORT, "Interface #{interface[:name]} on device #{@target.description()} has#{(interface[:voice_configuration_items])?" ":" no "}voice configuration items" if interface[:voice_configuration_items]
              debug "Assuming mode '#{interface[:mode]}' for interface #{interface[:name]} on device #{@target.description()}"
              debug "Interface #{interface[:name]} on device #{@target.description()} has#{(interface[:access_configuration_items])?" ":" no "}access mode configuration items" if interface[:access_configuration_items]
              debug "Interface #{interface[:name]} on device #{@target.description()} has#{(interface[:trunk_configuration_items])?" ":" no "}trunk mode configuration items"
              debug "Interface #{interface[:name]} on device #{@target.description()} has#{(interface[:voice_configuration_items])?" ":" no "}voice configuration items"
            end
          end
          if interface[:mode]==:access && interface[:access_configuration_items].nil?() && interface[:voice_configuration_items].nil?()
            info Ghun::Log::Type::REPORT, "Interface #{interface[:name]} on device #{@target.description()} has mode 'access' without access mode configuration items"
            debug "Interface #{interface[:name]} on device #{@target.description()} has mode 'access' without access mode configuration items"
          elsif interface[:mode]==:access && interface[:trunk_configuration_items]
            info Ghun::Log::Type::REPORT, "Interface #{interface[:name]} on device #{@target.description()} has mode 'access' and trunk mode configuration items"
            debug "Interface #{interface[:name]} on device #{@target.description()} has mode 'access' and trunk mode configuration items"
          elsif interface[:mode]==:trunk && interface[:trunk_configuration_items].nil?() && interface[:voice_configuration_items].nil?()
            debug "Interface #{interface[:name]} on device #{@target.description()} has mode 'trunk' without trunk mode configuration items"
          elsif interface[:mode]==:trunk && interface[:access_configuration_items]
            info Ghun::Log::Type::REPORT, "Interface #{interface[:name]} on device #{@target.description()} has mode 'trunk' and access mode configuration items"
            debug "Interface #{interface[:name]} on device #{@target.description()} has mode 'trunk' and access mode configuration items"
          end
          layer2=false
          layer2||=interface.include?(:mode)
          layer2||=interface.include?(:allowed_vlans)
          layer2||=interface.include?(:voice_vlan)
          layer2||=interface.include?(:native_vlan)
          layer2||=interface.include?(:access_vlan)
          layer3=false
          layer3||=interface.include?(:ipv4_addresses)
          layer3||=interface.include?(:ipv6_addresses)
          layer3||=interface.include?(:virtual_id)
          layer3||=interface[:virtual]
          if layer2 && layer3 && interface[:shutdown]
            interface[:layer]=1
          elsif layer2 && layer3 && !interface[:shutdown]
            error "Interface #{interface[:name]} on device #{@target.description()} has layer 2 and layer 3 configuration items"
            error Ghun::Log::Type::REPORT, "Interface #{interface[:name]} on device #{@target.description()} has layer 2 and layer 3 configuration items"
            interface[:layer]=1
          elsif layer2 then
            interface[:layer]=2
          elsif layer3 then
            interface[:layer]=3
          else
            warn "Interface #{interface[:name]} on device #{@target.description()} has neither layer 2 nor layer 3 configuration items" if !interface[:shutdown] && !(@temporary_storage[:cdp_interfaces].include?(interface[:name].intern) && (!@temporary_storage[:cdp_interfaces][interface[:name].intern][:up] || !@temporary_storage[:cdp_interfaces][interface[:name].intern][:admin_up]))
            interface[:layer]=1
          end
          if interface.include?(:native_vlan) && !interface[:native_vlan].nil?() && interface[:native_vlan]!=''
            @target_handler.update_vlan(interface[:native_vlan], :native)
            @target_handler.update_vlan(interface[:native_vlan], :seen)
          end
          if interface.include?(:voice_vlan) && !interface[:voice_vlan].nil?() && interface[:voice_vlan]!=''
            @target_handler.update_vlan(interface[:voice_vlan], :voice)
            @target_handler.update_vlan(interface[:voice_vlan], :seen)
          end
          if interface.include?(:allowed_vlans) && !interface[:allowed_vlans].nil?()
            interface[:allowed_vlans]=parse_allowed_vlans(interface[:allowed_vlans])
            interface[:allowed_vlans].each do |vlan|
              @target_handler.update_vlan(vlan, :tagged)
              @target_handler.update_vlan(vlan, :seen)
            end
          end
          interface[:vtp_vlans]=@vtp_vlan_by_interface[interface[:name].intern] if @vtp_vlan_by_interface.include?(interface[:name].intern) && !@vtp_vlan_by_interface[interface[:name].intern].nil?()
          if interface[:mode]==:trunk
            interface.delete(:access_vlan)
            interface[:allowed_vlans]=parse_allowed_vlans(["1-4094"]) unless interface[:allowed_vlans]
          elsif interface[:mode]==:access
            interface[:native_vlan]=interface[:access_vlan]
            interface.delete(:access_vlan)
            interface.delete(:allowed_vlans)
          end
          interface[:native_vlan]="1" unless interface[:native_vlan]
          @interfaces[interface[:name].intern]=interface
        end
      end

      def parse_interfaces_trunk(label,command,result,step)
        if result[:stdout].nil?() || result[:stdout]=='' || result[:stdout].match("Line has invalid autocommand")
          debug "#{@target.description()} has no interfaces trunk output"
        else
          header=true
          parsing_allowed=false
          parsing_active=false
          parsing_unpruned=false
          result[:stdout].split("\n").each do |line|
            next if line.strip==''
            if header && line.match("Vlans allowed on trunk")
              header=false
              parsing_allowed=true
              next
            elsif header
              next
            end
            sp_line=line.split()
            interface=expand_interface_shortname(sp_line[0])
            vlans=nil
            unless sp_line[1..-1].join().match(/none/)
              vlans=parse_allowed_vlans(sp_line[1..-1])
            end
            if parsing_allowed && line.match("Vlans allowed and active in management domain")
              parsing_allowed=false
              parsing_active=true
              next
            elsif parsing_allowed
              if @interfaces.include?(interface.intern)
                if @interfaces[interface.intern].include?(:allowed_vlans)
                  unless @interfaces[interface.intern][:allowed_vlans]==vlans
                    warn "Different allowed vlan list from configuration and interfaces trunk for interface #{interface} on #{@target.description()}", "Configuration: #{@interfaces[interface.intern][:allowed_vlans]}", "Interfaces trunk: #{vlans}"
                  end
                else
                  @interfaces[interface.intern][:allowed_vlans]=vlans
                end
              else
                warn "No interface #{interface} on #{@target.description()}"
              end
            end
            if parsing_active && line.match("Vlans in spanning tree forwarding state and not pruned")
              parsing_active=false
              parsing_unpruned=true
              next
            elsif parsing_active
              if @interfaces.include?(interface.intern)
                if @interfaces[interface.intern].include?(:vtp_vlans)
                  unless @interfaces[interface.intern][:vtp_vlans]==vlans
                    warn "Different vtp vlan list from vlan and interfaces trunk for interface #{interface} on #{@target.description()}", "Vlan: #{@interfaces[interface.intern][:vtp_vlans]}", "Interfaces trunk: #{vlans}"
                  end
                else
                  @interfaces[interface.intern][:vtp_vlans]=vlans
                end
              else
                warn "No interface #{interface} on #{@target.description()}"
              end
            end
            if parsing_unpruned
              if @interfaces.include?(interface.intern)
                @interfaces[interface.intern][:unpruned_vlans]=vlans
              else
                warn "No interface #{interface} on #{@target.description()}"
              end
            end
          end
          unless parsing_unpruned
            error "Unexpected state after parsing; assuming error parsing interfaces trunk for #{@target.description()}"
          end
        end
      end

      def parse_switch(label,command,result,step)
        if result[:stdout].nil?() || result[:stdout]=='' || result[:stdout].match("Line has invalid autocommand")
          debug "#{@target.description()} has no switch output"
        else
          @temporary_storage[:switch_result]=true
          if @temporary_storage[:switch_result] && @temporary_storage[:long_interface_names]
            @temporary_storage[:multi_chassis]=true
            @temporary_storage[:switch_ids]=Array.new() unless @temporary_storage.include?(:switch_ids)
            header=true
            result[:stdout].split("\n").each do |line|
              if header && line.match(/^-+.*$/)
                header=false
                next
              end
              next if header
              next if line.strip()==''
              id=line.split()[0].strip.sub(/\*/,'')
              @temporary_storage[:switch_ids] << id unless @temporary_storage[:switch_ids].include?(id)
            end
          end
          if @temporary_storage.include?(:switch_ids)
            @temporary_storage[:switch_ids].uniq!()
            @temporary_storage[:switch_ids].sort!()
          end
        end
      end

      def parse_redundancy(label,command,result,step)
        if result[:stdout].nil?() || result[:stdout]=='' || result[:stdout].match("Line has invalid autocommand")
          debug "#{@target.description()} has no redundancy output"
        else
          @temporary_storage[:redundancy_result]=true
          if @temporary_storage[:redundancy_result] && @temporary_storage[:long_interface_names]
            @temporary_storage[:switch_ids]=Array.new() unless @temporary_storage.include?(:switch_ids)
            result[:stdout].split("\n").each do |line|
              #FIXME what about vwg-c65??-.*
              next unless line.downcase.match(/slot/)
              id=line.downcase.sub(/.*slot /,'').sub(/\/[0-9]+/,'').strip()
              @temporary_storage[:switch_ids] << id unless @temporary_storage[:switch_ids].include?(id)
            end
          end
          if @temporary_storage.include?(:switch_ids)
            @temporary_storage[:switch_ids].uniq!()
            @temporary_storage[:switch_ids].sort!()
          end
        end
      end

      def parse_vtp_status(label,command,result,step)
        #TODO implement me
      end

      def parse_vtp_devices(label,command,result,step)
        #TODO implement me
      end

      def parse_capabilities(c)
        cap=Array.new()
        cap << :"R" if c.match(/Router/)
        cap << :"T" if c.match(/Trans Bridge/)
        cap << :"T" if c.match(/Trans-Bridge/)
        cap << :"B" if c.match(/Source Route Bridge/)
        cap << :"S" if c.match(/Switch/)
        cap << :"H" if c.match(/Host/)
        cap << :"I" if c.match(/IGMP/)
        cap << :"r" if c.match(/Repeater/)
        cap << :"P" if c.match(/Phone/)
        cap << :"D" if c.match(/Remote/)
        cap << :"C" if c.match(/CVTA/)
        cap << :"M" if c.match(/Two-port Mac Relay/)
        cap
      end

      def expand_interface_shortname(i)
        is_short_name=false
        is_short_name||=i.match(/^Po[0-9]/)
        is_short_name||=i.match(/^Te[0-9]/)
        is_short_name||=i.match(/^Gi[0-9]/)
        is_short_name||=i.match(/^Fa[0-9]/)
        return i unless is_short_name
        long=i
        long.sub!(/^Po/,'Port-channel') if i.match(/^Po[0-9]/)
        long.sub!(/^Te/,'TenGigabitEthernet') if i.match(/^Te[0-9]/)
        long.sub!(/^Gi/,'GigabitEthernet') if i.match(/^Gi[0-9]/)
        long.sub!(/^Fa/,'FastEthernet') if i.match(/^Fa[0-9]/)
        long
      end

      def parse_acl(line)
        line_orig=line.clone()
        #TODO add support for mac and ipv6 access-lists
        acl=Iof::Data::NO::Layer4ACLEntry.new()
        if line[0]=='dynamic' then
          line.shift(2)
          line.shift(2) if line[0]=='timeout'
        end
        case line[0]
        when "deny"
          acl[:no_acl_action]=:deny
        when "permit"
          acl[:no_acl_action]=:permit
        when "remark"
          debug "Ignoring acl comment line '#{line.join(' ')}'"
          return nil
        else
          debug "Ignoring line '#{line_orig.join(' ')}' in ACL on device #{@target.description()} with unknown action '#{line.join(' ')}'"
          return nil
        end
        line.shift
        protocol_given=true
        if line[0]=="ip" then
          acl[:no_acl_protocol]=:ipv4
          line.shift
        elsif line[0]=="tcp"
          acl[:no_acl_protocol]=:tcpv4
          line.shift
        elsif line[0]=="udp"
          acl[:no_acl_protocol]=:udpv4
          line.shift
        elsif line[0]=="icmp"
          acl[:no_acl_protocol]=:icmpv4
          line.shift
        elsif line[0]=="ahp" || line[0]=="esp"
          #TODO support ahp and esp
          warn "Protocol #{line[0]} in ACL '#{line_orig.join(' ')}' on device #{@target.description()} is not supported"
          return nil
        else
          debug "No protocol in ACL '#{line_orig.join(' ')}' on device #{@target.description()} given; assuming ipv4"
          protocol_given=false
          acl[:no_acl_protocol]=:ipv4
        end
        unless protocol_given then
          if line.size==1 then
            acl[:no_source_ipv4_address]=@data_handler.get_ipv4_address_name(line[0])
            line.shift
          elsif line.size==2 then
            acl[:no_source_ipv4_network]=@data_handler.get_ipv4_network_name(line[0],Iof::Data.ipv4_to_netmask(line[1]))
            line.shift(2)
          else
            error "unexpected line format in ACL '#{line_orig.join(' ')}' on device #{@target.description()}: '#{line.join(' ')}'"
            return nil
          end
        else
          if line[0]=='any' then
            line.shift
          elsif line[0]=='host' then
            line.shift
            acl[:no_source_ipv4_address]=@data_handler.get_ipv4_address_name(line[0])
            line.shift
          else
            acl[:no_source_ipv4_network]=@data_handler.get_ipv4_network_name(line[0],Iof::Data.ipv4_to_netmask(line[1]))
            line.shift(2)
          end
          if line[0]=='eq' || line[0]=='neq' || line[0]=='lt' || line[0]=='gt' || line[0]=='established' then
            if line[0]=='established'
              warn "Ignoring established keyword in ACL '#{line_orig.join(' ')}' on device #{@target.description()}; not supported in data model"
              line.shift
            end
            unless line[0]=='eq' then
              #TODO other operators
              warn "operator '#{line[0]}' in ACL '#{line_orig.join(' ')}' on device #{@target.description()} is unsupported"
              return  nil
            else
              unless line[1].to_i()==0 then
                if acl[:no_acl_protocol]==:udpv4 then
                  acl[:no_source_udp_port]=@data_handler.get_udp_port_name(line[1].to_i)
                elsif acl[:no_acl_protocol]==:tcpv4 then
                  acl[:no_source_tcp_port]=@data_handler.get_tcp_port_name(line[1].to_i)
                end
              else
                begin
                  if acl[:no_acl_protocol]==:udpv4 then
                    acl[:no_source_udp_port]=@data_handler.get_udp_port_name(::Socket.getservbyname(line[1],acl[:no_acl_protocol].to_s.gsub!(/v4$/,'')))
                  elsif acl[:no_acl_protocol]==:tcpv4 then
                    acl[:no_source_tcp_port]=@data_handler.get_tcp_port_name(::Socket.getservbyname(line[1],acl[:no_acl_protocol].to_s.gsub!(/v4$/,'')))
                  end
                rescue => e
                  error "service name #{line[1]} in ACL '#{line_orig.join(' ')}' on device #{@target.description()} not recognized; skipping rule", e
                  return nil
                end
              end if !line[0].nil?() && line[0]!=''
            end
            line.shift(2)
          end if !line[0].nil?() && line[0]!='' &&  (acl[:no_acl_protocol]==:udpv4 || acl[:no_acl_protocol]==:tcpv4)
          if line[0]=='any' then
            line.shift
          elsif line[0]=='host' then
            line.shift
            acl[:no_destination_ipv4_address]=@data_handler.get_ipv4_address_name(line[0])
            line.shift
          else
            acl[:no_destination_ipv4_network]=@data_handler.get_ipv4_network_name(line[0],Iof::Data.ipv4_to_netmask(line[1]))
            line.shift(2)
          end

          if acl[:no_acl_protocol]==:udpv4 || acl[:no_acl_protocol]==:tcpv4 then
            if line[0]=='established'
              debug "Ignoring established in ACL '#{line_orig.join(' ')}' on device #{@target.description()} key word; not supported in data model"
              line.shift
            end
            unless line[0]=='eq' then
              #TODO other operators
              warn "operator '#{line[0]}' in ACL '#{line_orig.join(' ')}' on device #{@target.description()} is unsupported"
              return nil
            else
              unless line[1].to_i()==0 then
                if acl[:no_acl_protocol]==:udpv4 then
                  acl[:no_destination_udp_port]=@data_handler.get_udp_port_name(line[1].to_i)
                elsif acl[:no_acl_protocol]==:tcpv4 then
                  acl[:no_destination_tcp_port]=@data_handler.get_tcp_port_name(line[1].to_i)
                end
              else
                begin
                  if acl[:no_acl_protocol]==:udpv4 then
                    acl[:no_destination_udp_port]=@data_handler.get_udp_port_name(::Socket.getservbyname(line[1],acl[:no_acl_protocol].to_s.gsub!(/v4$/,'')))
                  elsif acl[:no_acl_protocol]==:tcpv4 then
                    acl[:no_destination_tcp_port]=@data_handler.get_tcp_port_name(::Socket.getservbyname(line[1],acl[:no_acl_protocol].to_s.gsub!(/v4$/,'')))
                  end
                rescue => e
                  error "service name #{line[1]} in ACL '#{line_orig.join(' ')}' on device #{@target.description()} not recognized; skipping rule", e
                  return nil
                end
              end
              line.shift(2)
            end if !line[0].nil?() && line[0]!=''
          elsif acl[:no_acl_protocol]==:icmpv4 then
            acl[:no_destination_icmpv4_type]=@data_handler.get_icmpv4_type_name(line[0])
            line.shift
          else
            warn "Could not parse additional information for protocol #{acl.protocol.to_s} in ACL on device #{@target.description()}; Syntax unknown"
          end if !line[0].nil?()  && line[0]!='' && (acl[:no_acl_protocol]==:udpv4 || acl[:no_acl_protocol]==:tcpv4 || acl[:no_acl_protocol]==:icmpv4)
        end
        debug "Ignoring rest of acl line: '#{line.join(' ')}'" unless line.size==0
        acl
      end

      def parse_allowed_vlans(vlans)
        begin
          result_i=Array.new
          result=Array.new()
          parts=Array.new()
          vlans.each do |a|
            parts.concat(a.split(','))
          end
          parts.each do |p|
            if p.match(/-/) then
              first,last,garbage=p.split('-')
              debug "Unexpected format for vlan definition: '#{p}', ignoring '#{garbage.to_s}" unless garbage.nil?()
              first.strip!
              last.strip!
              Range.new(first.to_i,last.to_i).to_a.each do |r|
                next if r.nil?()
                result_i << r
              end
            else
              next if p.nil?() || p==''
              result_i << p.to_i
            end
          end
          result_i=result_i.uniq.sort.compact
          result_i.each do |r|
            result << r
          end
          result
        rescue => e
          warn "Failed to parse #{vlans.to_s}", e
          return []
        end
      end

      def generate_vlan_name_list(vlans)
        begin
          result=Array.new()
          vlans.each do |v|
            result << @data_handler.get_vlan_name(v)
          end
          result
        rescue => e
          warn "Failed to parse #{vlans.to_s}", e
          return []
        end
      end

     def generate_output()
       host_name=@target.name
       @cdp_neighbors_by_interface.each do |name,neighbors|
         interface_poe_capable=false
         interface_powers_device=false
         cdp_device_type=''
         interface_poe_capable=!@power_outgoing.nil?() && @power_outgoing[:interfaces].include?(name) && !@power_outgoing[:interfaces][name].nil?() && !@power_outgoing[:interfaces][name].empty?()
         cdp_neighbor=!(neighbors.nil?() || neighbors.empty?())

         next unless cdp_neighbor

         if interface_poe_capable
           interface_powers_device||=(@power_outgoing[:interfaces][name][:operational]=="on")
           cdp_device_type=@power_outgoing[:interfaces][name][:device] unless @power_outgoing[:interfaces][name][:device].nil?() || @power_outgoing[:interfaces][name][:device]=="" || @power_outgoing[:interfaces][name][:device]=="n/a"
         end

         if interface_powers_device && neighbors.size()>1
           cdp_device_type="multiple devices"
           interface_powers_device=false
           warn "Found multiple PoE powered devices at interface #{name.to_s} on #{@target.description()}"
           warn "Could not determine device powered by #{@target.description()}"
         end

         base=Hash.new()
         base[:device]=host_name.clone()
         base[:interface]=name.to_s()
         base[:is_poe_capable]=interface_poe_capable.to_s
         base[:powers_device]=interface_powers_device.to_s
         base[:power_device_type]=cdp_device_type.to_s
         if cdp_neighbor
           neighbors.each do |neighbor|
             data=base.clone()
             data[:reports_power]=neighbor[:reports_power]
             data[:non_zero_power]=neighbor[:powered]
             data[:cdp_device_name]=neighbor[:name]
             data[:cdp_device_type]=neighbor[:platform]
             #XXX sort Accesspoint and phones
             Ghun::Output::DISTRIBUTED_HANDLER[:poe_aps].append_data(data) if Ghun::Output::DISTRIBUTED_HANDLER[:poe_aps]
             Ghun::Output::DISTRIBUTED_HANDLER[:poe_phones].append_data(data) if Ghun::Output::DISTRIBUTED_HANDLER[:poe_phones]
           end
         else
           base[:reports_power]=""
           base[:non_zero_power]=""
           base[:cdp_device_type]=""
           base[:cdp_device_name]=""
           #XXX sort Accesspoint and phones
           Ghun::Output::DISTRIBUTED_HANDLER[:poe_aps].append_data(base) if Ghun::Output::DISTRIBUTED_HANDLER[:poe_aps]
           Ghun::Output::DISTRIBUTED_HANDLER[:poe_phones].append_data(base) if Ghun::Output::DISTRIBUTED_HANDLER[:poe_phones]
         end
       end
     end




      def intersect_vlan(native_mine,voice_mine,tagged_mine,native_theirs,voice_theirs,tagged_theirs,interface)
        native=native_mine
        native=native_theirs if native.nil?() || native==''
        native="1" if native.nil?() || native==''
        warn "Conflicting native vlan configuration on link at interface #{interface} of device #{@target.description()}" if native_mine != native_theirs && !(native_mine.nil?() || native_mine=='') && !(native_theirs.nil?() || native_theirs=='')
        voice=voice_mine
        voice=voice_theirs if voice.nil?() || voice==''
        warn "Conflicting voice vlan configuration on link at interface #{interface} of device #{@target.description()}" if voice_mine != voice_theirs && !(voice_mine.nil?() || voice_mine=='') && !(voice_theirs.nil?() || voice_theirs=='')
        tagged=nil
        if tagged_mine.nil?() || tagged_mine.empty?()
          tagged=tagged_theirs
        elsif tagged_theirs.nil?() || tagged_theirs.empty?()
          tagged=tagged_mine
        else
          tagged=tagged_mine & tagged_theirs
        end
        tagged=parse_allowed_vlans("1-4096") if tagged.nil?()
      end


      def process_incoming()
        #TODO process incoming cdp sightings
        #TODO process incoming power reports
        #TODO process incoming vlan reports
        #process data written into the target this target
        #CDP Neighbors report sightings to seen target
      end

      def postprocess_parsed_data
        debug "Postprocessing #{@target.description()}"

        sub_components=Hash.new()
        component=Iof::Data::NO::Switch.new()
        component.category=0
        l2_suffix=@@l2eth_suffix
        l3v4_suffix=@@l3v4_suffix
        l3v6_suffix=@@l3v6_suffix

        @neighbors=Array.new()
        @connections=Array.new()
        @components=Array.new()

        component[:no_serial_number]=@target.special[:serial] if @target.special.include?(:serial)
        component[:no_vendor_name]=@target.device_vendor
        component[:no_old_administrative_domain_internal_label]=@target.special[:administrative_domain_old_label]
        component[:no_assigned_administrative_domain]=@target.special[:administrative_domain_name]
        component[:no_old_location_internal_label]=@target.special[:location_old_label]
        component[:no_physical_location]=@target.special[:location_name]
        component.name=@target.name()

        component[:no_switch]=true
        component[:no_router]=false
        component[:no_identifier]=@target.identifier
        component[:no_ipv4_forward_set]=@ipv4_forwards unless @ipv4_forwards.nil?()
        component[:no_ipv6_forward_set]=@ipv6_forwards unless @ipv6_forwards.nil?()

        @temporary_storage[:switch_ids].each do |id|
          c=Iof::Data::NO::Switch.new()
          c.category=0
          c[:no_vendor_name]=@target.device_vendor
          c.name="#{@target.name()}_sub#{id}"
          c[:no_switch]=true
          c[:no_router]=false
          component[:no_sub_managed_network_component]=c.name
          sub_components[id.intern]=c
        end if @temporary_storage.include?(:switch_ids)

        @port_channel_to_interface.each do |portchannel,interfaces|
          interfaces.each do |interface|
            if @interfaces.include?(interface.intern)
              @interfaces[interface.intern][:layer]=2 if @interfaces[interface.intern][:layer]<2
              @interfaces[interface.intern][:mode]=:trunk unless @interfaces[interface.intern].include?(:mode)
              @interfaces[interface.intern][:name]=interface unless @interfaces[interface.intern].include?(:name)
              @interfaces[interface.intern][:virtual]=false unless @interfaces[interface.intern].include?(:virtual)
              @interfaces[interface.intern][:monitor]=false unless @interfaces[interface.intern].include?(:monitor)
              @interfaces[interface.intern][:is_in_channel_group]=true unless @interfaces[interface.intern].include?(:is_in_channel_group)
            else
              warn "Creating interface #{interface} as #{portchannel} is based on it"
              i=Hash.new()
              i[:layer]=2
              i[:mode]=:trunk
              i[:name]=interface
              i[:virtual]=false
              i[:monitor]=false
              i[:is_in_channel_group]=true
              @interfaces[interface.intern]=i
            end
          end
        end

        @interfaces.each do |name,interface|
          debug "Preparing config postprocessing config for interface #{name.to_s} on device #{@target.description()}"
          interface[:name]=name.to_s unless interface.include?(:name)

          if interface[:name].match(/^Vlan/)
            interface[:mode]=:access
            interface[:virtual]=true
            interface[:native_vlan]=interface[:virtual_id]
            interface[:voice_vlan]=nil
            interface[:vtp_vlan]=nil
            interface[:unpruned_vtp_vlan]=nil
            interface[:allowed_vlans]=[interface[:virtual_id]]
          end
          next if interface[:virtual]
          if interface[:name].match(/Port-channel/)
            base_interface=@port_channel_to_interface[interface[:name].intern]
            b_int=nil
            base_interface.each do |b|
              b_int||=@interfaces[b.intern]
              break if b_int
            end
            b_int.each do |key,value|
              next if key==:name || key==:description || key==:is_channel_group || key==:is_in_channel_group
              if interface.include?(key)
                debug "Keeping config value for key #{key} at interface #{interface[:name]} on device #{@target.description}"
              else
                debug "Using config value for key #{key} from interface #{@interfaces[base_interface[0].intern]} at interface #{interface[:name]} on device #{@target.description}"
                interface[key]=value
              end
            end unless b_int.nil?()
          end
          next if interface[:name].match(/Port-channel/)
          if @temporary_storage[:long_interface_names] && !@temporary_storage[:switch_ids].nil?() && !@temporary_storage[:switch_ids].empty?()
            id=interface[:name].sub(/.*([0-9]+)\/[0-9]+\/[0-9]+/,'\1')
            interface[:sub_id]=id if id.match(/^[0-9]+$/) && sub_components.include?(id.intern)
          end
        end

        @interfaces.each do |name,interface|
          debug "Postprocessing config for interface #{name.to_s} on device #{@target.description()}"
          l1=Iof::Data::NO::Layer1Interface.new()
          interface[:l1]=l1
          l1.name="#{interface[:name]}"
          l1[:no_monitor_port]=interface[:monitor]
          if interface[:virtual]
            l1[:no_interface_type]=:virtual
          elsif interface[:name].match(/Port-channel/)
            l1[:no_interface_type]=:etherchannel
          else
            l1[:no_interface_type]=:physical
          end

          @data_handler.declare_interface_name_change("#{component.name}___#{l1.name}","#{sub_components[interface[:sub_id].intern].name}___#{l1.name}") if interface.include?(:sub_id) && !interface[:sub_id].nil?()

          #handle power
          l1[:no_poe_capable]=false
          l1[:no_poe_capable]=interface[:poe_capable] if interface.include?(:poe_capable)
          l1[:no_powering_poe_network_component]=interface[:powers_device] if interface.include?(:powers_device)

          l2=nil
          if interface[:layer]>=2
            l2=Iof::Data::NO::EthernetInterface.new()
            l2.name=l2_suffix.clone()
            l1[:no_ethernet_interface]=l2
            l2[:no_trunk_mode]=interface[:mode]
            l2[:no_native_vlan]=@data_handler.get_vlan_name(interface[:native_vlan]) unless interface[:native_vlan].nil?()
            l2[:no_voice_vlan]=@data_handler.get_vlan_name(interface[:voice_vlan]) unless interface[:voice_vlan].nil?()
            l2[:no_tagged_vlan]=generate_vlan_name_list(interface[:allowed_vlans]) unless interface[:allowed_vlans].nil?()
            l2[:no_vtp_vlan]=generate_vlan_name_list(interface[:vtp_vlans]) unless interface[:vtp_vlans].nil?()
            l2[:no_unpruned_vtp_vlan]=generate_vlan_name_list(interface[:unpruned_vlans]) unless interface[:unpruned_vlans].nil?()
            @data_handler.declare_interface_name_change("#{component.name}___#{l1.name}___#{l2.name}","#{sub_components[interface[:sub_id].intern].name}___#{l1.name}___#{l2.name}") if interface.include?(:sub_id) && !interface[:sub_id].nil?()
          end

          if interface[:layer]>=3 && !interface[:ipv4_addresses].nil?() && !interface[:ipv4_addresses].empty?()
            l3v4=Iof::Data::NO::IPv4Interface.new()
            l3v4.name=l3v4_suffix.clone()
            @data_handler.declare_interface_name_change("#{component.name}___#{l1.name}___#{l2.name}___#{l3v4.name}","#{sub_components[interface[:sub_id].intern].name}___#{l1.name}___#{l2.name}___#{l3v4.name}") if interface.include?(:sub_id) && !interface[:sub_id].nil?()
            l3v4[:no_interface_type]=:virtual
            interface[:ipv4_addresses].each do |v4|
              l3v4[:no_ipv4_address]=@data_handler.get_ipv4_address_name(v4[:address],v4[:netmask])
              l3v4[:no_ipv4_network]=@data_handler.get_ipv4_network_name(Iof::Data.ipv4_network_part(v4[:address],Iof::Data.ipv4_to_netmask(v4[:netmask])),Iof::Data.ipv4_to_netmask(v4[:netmask]))
            end
            l2[:no_ipv4_interface]=l3v4
            #TODO: try to detect management addresses from incoming cdp data
            #TO DO: apply acls from line section
            #TO DO: modify this acls to match only addresses on this device
            if !interface[:acl].nil?() && !interface[:acl].empty?() then
              l3s=Array.new()
              l2[:no_ipv4_interface].each do |l3|
                l3s << l3
              end unless l2[:no_ipv4_interface].nil?()
              interface[:acl].each do |a|
                if a[:direction]=="in" then
                  l3s.each do |l3|
                    if l3[:no_input_layer4_acl].nil?() then
                      l3[:no_input_layer4_acl]=clone_acl_list(@acls_list[a[:id]]) unless @acls_list[a[:id]].nil?()
                      l3[:no_input_layer4_acl][:no_yaml_representation]=@acls_yaml[a[:id]] unless l3[:no_input_layer4_acl].nil?() || @acls_yaml[a[:id]].nil?()
                    else
                      error "More than one input acl defined for interface '#{name}' on device #{@target.description()}; discarding all"
                      l3[:no_input_layer4_acl]=nil
                    end
                    debug "Resulting incoming ACL for interface '#{interface[:name]}': #{l3[:no_input_layer4_acl]}" unless l3[:no_input_layer4_acl].nil?()
                  end
                elsif a[:direction]=="out" then
                  l3s.each do |l3|
                    if l3[:no_output_layer4_acl].nil?() then
                      l3[:no_output_layer4_acl]=clone_acl_list(@acls_list[a[:id]]) unless @acls_list[a[:id]].nil?()
                      l3[:no_output_layer4_acl][:no_yaml_representation]=@acls_yaml[a[:id]] unless l3[:no_output_layer4_acl].nil?() || @acls_yaml[a[:id]].nil?()
                    else
                      error "More than one output acl defined for interface '#{name}' on device #{@target.description()}; discarding all"
                      l3[:no_output_layer4_acl]=nil
                    end
                  debug "Resulting outgoing ACL for interface '#{interface[:name]}': #{l3[:no_output_layer4_acl]}" unless l3[:no_output_layer4_acl].nil?()
                  end
                else
                  warn "Unknown ACL direction #{a[:direction]} for ACL #{a.to_s} on interface #{name} on device #{@target.description()}"
                end
              end
            end
          end

          if interface[:layer]>=3 && !interface[:ipv6_addresses].nil?() && !interface[:ipv6_addresses].empty?()
            l3v6=Iof::Data::NO::IPv6Interface.new()
            l3v6.name=l3v6_suffix.clone()
            @data_handler.declare_interface_name_change("#{component.name}___#{l1.name}___#{l2.name}___#{l3v6.name}","#{sub_components[interface[:sub_id].intern].name}___#{l1.name}___#{l2.name}___#{l3v6.name}") if interface.include?(:sub_id) && !interface[:sub_id].nil?()
            l3v6[:no_interface_type]=:virtual
            interface[:ipv6_addresses].each do |v6|
              l3v6[:no_ipv6_address]=@data_handler.get_ipv6_address_name(v6[:address],v6[:netmask])
              l3v6[:no_ipv6_network]=@data_handler.get_ipv6_network_name(Iof::Data.ipv6_network_part_uncompressed(v6[:address],v6[:netmask]),v6[:netmask])
            end
            l2[:no_ipv6_interface]=l3v6
            if !interface[:acl].nil?() && !interface[:acl].empty?() then
              l3s=Array.new()
              l2[:no_ipv4_interface].each do |l3|
                l3s << l3
              end unless l2[:no_ipv4_interface].nil?()
              interface[:acl].each do |a|
                if a[:direction]=="in" then
                  l3s.each do |l3|
                    if l3[:no_input_layer4_acl].nil?() then
                      l3[:no_input_layer4_acl]=clone_acl_list(@acls_list[a[:id]]) unless @acls_list[a[:id]].nil?()
                      l3[:no_input_layer4_acl][:no_yaml_representation]=@acls_yaml[a[:id]] unless l3[:no_input_layer4_acl].nil?() || @acls_yaml[a[:id]].nil?()
                    else
                      error "More than one input acl defined for interface '#{name}' on device #{@target.description()}; discarding all"
                      l3[:no_input_layer4_acl]=nil
                    end
                    debug "Resulting incoming ACL for interface '#{interface[:name]}': #{l3[:no_input_layer4_acl]}" unless l3[:no_input_layer4_acl].nil?()
                  end
                elsif a[:direction]=="out" then
                  l3s.each do |l3|
                    if l3[:no_output_layer4_acl].nil?() then
                      l3[:no_output_layer4_acl]=clone_acl_list(@acls_list[a[:id]]) unless @acls_list[a[:id]].nil?()
                      l3[:no_output_layer4_acl][:no_yaml_representation]=@acls_yaml[a[:id]] unless l3[:no_output_layer4_acl].nil?() || @acls_yaml[a[:id]].nil?()
                    else
                      error "More than one output acl defined for interface '#{name}' on device #{@target.description()}; discarding all"
                      l3[:no_output_layer4_acl]=nil
                    end
                  debug "Resulting outgoing ACL for interface '#{interface[:name]}': #{l3[:no_output_layer4_acl]}" unless l3[:no_output_layer4_acl].nil?()
                  end
                else
                  warn "Unknown ACL direction #{a[:direction]} for ACL #{a.to_s} on interface #{name} on device #{@target.description()}"
                end
              end
            end
          end
        end

        @interfaces.each do |name,interface|
          if interface[:name].match(/^Port-channel/)
            debug "Processing interface name changes for interface #{name.to_s} on device #{@target.description()}"
            interface[:l1][:no_ethernet_interface].each do |l2|
              interfaces=@port_channel_to_interface[interface[:name].intern]
              interfaces.each do |i|
                if @data_handler.changed_interface_names.include?("#{component.name}___#{i}___#{l2_suffix}".intern)
                  debug "Changing name of sub interface from #{component.name}___#{i}___eth to #{@data_handler.changed_interface_names["#{component.name}___#{i}___#{l2_suffix}".intern]}"
                  l2[:no_sub_layer2_interface]=@data_handler.changed_interface_names["#{component.name}___#{i}___#{l2_suffix}".intern]
                elsif @interfaces.include?(i.intern) && @interfaces[i.intern].include?(:sub_id)
                  debug "Changing name of sub interface from #{component.name}___#{i}___eth to #{component.name}_sub_#{@interfaces[i.intern][:sub_id]}___#{i}___#{l2_suffix} by manual change"
                  l2[:no_sub_layer2_interface]="#{component.name}_sub_#{@interfaces[i.intern][:sub_id]}___#{i}___#{l2_suffix}"
                elsif @interfaces.include?(i.intern) && @interfaces[i.intern].include?(:name)
                  id=@interfaces[i.intern][:name].sub(/.*([0-9]+)\/[0-9]+\/[0-9]+/,'\1')
                  if id.match(/^[0-9]+$/) && sub_components.include?(id.intern)
                    @interfaces[i.intern][:sub_id]=id
                    debug "Changing name of sub interface from #{component.name}___#{i}___eth to #{component.name}_sub_#{id}___#{i}___#{l2_suffix} after determing id for manual change"
                    l2[:no_sub_layer2_interface]="#{component.name}_sub_#{id}___#{i}___#{l2_suffix}"
                  else
                    debug "Not changing name of sub interface #{component.name}___#{i}___#{l2_suffix}"
                    l2[:no_sub_layer2_interface]="#{component.name}___#{i}___#{l2_suffix}"
                  end
                else
                  debug "Not changing name of sub interface #{component.name}___#{i}___#{l2_suffix}"
                  l2[:no_sub_layer2_interface]="#{component.name}___#{i}___#{l2_suffix}"
                end
              end
            end unless interface[:l1].nil?() || interface[:l1][:no_ethernet_interface].nil?()
          end
        end

        @interfaces.each do |name,interface|
          module_id=nil
          if @temporary_storage[:long_interface_names]
            module_id=interface[:name].sub(/.*[0-9]+\/([0-9]+)\/[0-9]+/,'\1')
          else
            module_id=interface[:name].sub(/.*([0-9]+)\/[0-9]+/,'\1')
          end

          module_id=nil if !module_id.match(/^[0-9]+$/)
          if !module_id.nil?()
            if interface.include?(:sub_id) && @modules.include?(interface[:sub_id].intern) && @modules[interface[:sub_id].intern].include?(module_id.intern)
              @modules[interface[:sub_id].intern][module_id.intern][:interfaces] << interface[:name].intern
              interface[:module_id]=module_id
            elsif !interface.include?(:sub_id) && @modules.include?(nil) && @modules[nil].include?(module_id.intern)
              @modules[nil][module_id.intern][:interfaces] << interface[:name].intern
              interface[:module_id]=module_id
            end
          end

          next if interface[:name].match(/Port-channel/)
          debug "Postprocessing config for interface #{name.to_s} on device #{@target.description()}"
          l1=interface[:l1]
          if interface.include?(:sub_id)
            c=sub_components[interface[:sub_id].intern]
            debug "Adding #{l1.name} to #{c.name}"
            c[:no_simple_layer1_interface]=l1
          else
            debug "Adding #{l1.name} to #{component.name}"
            component[:no_simple_layer1_interface]=l1
          end
        end

        @interfaces.each do |name,interface|
          next unless interface[:name].match(/Port-channel/)
          debug "Postprocessing config for interface #{name.to_s} on device #{@target.description()}"
          l1=interface[:l1]
          if interface.include?(:sub_id)
            c=sub_components[interface[:sub_id].intern]
            debug "Adding #{l1.name} to #{c.name}"
            c[:no_simple_layer1_interface]=l1
          else
            debug "Adding #{l1.name} to #{component.name}"
            component[:no_simple_layer1_interface]=l1
          end
        end

        @modules.each do |sub_id,modules|
          modules.values.each do |mod|
            m=case mod[:type]
            when :interface
              Iof::Data::NO::InterfaceModule.new()
            when :firewall
              Iof::Data::NO::FirewallModule.new()
            when :wireless
              Iof::Data::NO::WirelessModule.new()
            when :supervisor
              Iof::Data::NO::SupervisorModule.new()
            else
              Iof::Data::NO::Module.new()
            end

            m.name=mod[:name]
            m[:no_module_id]=mod[:id]
            m[:no_module_card_type]=mod[:card_type]
            m[:no_module_model]=mod[:model]
            m[:no_module_number_of_ports]=mod[:number_of_ports]
            m[:no_serial_number]=mod[:serial]
            mod[:interfaces].each do |interface|
              i="#{component.name}___#{@interfaces[interface][:l1].name}"
              if @data_handler.changed_interface_names.include?(i.intern)
                m[:no_part]=@data_handler.changed_interface_names[i.intern]
              else
                m[:no_part]=i
              end
            end
            unless sub_id.nil?()
              c=sub_components[sub_id.intern]
              debug "Adding #{m.name} to #{c.name}"
              c[:no_module]=m
            else
              debug "Adding #{m.name} to #{component.name}"
              component[:no_module]=m
            end
          end
        end

        @cdp_neighbors_by_interface.each do |interface,neighbors|
          #create connections to cdp neighbors
          #Neighbors are created by cdp parser
          me="#{component.name}___#{Iof::Data::Base.sanitize_name(@interfaces[interface][:name])}"
          neighbors.each do |neighbor|
            target=nil
            target=@target_handler.get_target_by_name(neighbor[:name])
            neighbor[:mac_addresses].each do |a|
              t=@target_handler.get_target_by_mac(a)
              unless t.nil?()
                target=t
                break
              end
            end if target.nil?()
            neighbor[:ipv4_addresses].each do |a|
              t=@target_handler.get_target_by_ipv4(a)
              unless t.nil?()
                target=t
                break
              end
            end if target.nil?()
            if target.nil?()
              warn "Found no target for neighbor; this may be caused by active runner filters; #{neighbor.inspect()}"
            else
              n="#{Iof::Data::Base.sanitize_name(target.name)}___#{Iof::Data::Base.sanitize_name(neighbor[:interface])}"
              connection=Iof::Data::NO::IsConnectedTo.new()
              connection.subject=me.clone()
              connection.object=n
              @connections << connection
            end
          end unless neighbors.nil?()
        end

        @mac_by_interface_parsed.each do |interface,macs|
          #create neighbors and connections for mac only neighbors
          if macs.nil?() || macs.size() <= 0
            debug "No mac neighbors to create at interface #{interface} on device #{@target.description()}"
          elsif macs.size==1
            #direct connection
            mac=macs[0]
            me="#{component.name}___#{Iof::Data::Base.sanitize_name(@interfaces[interface][:name])}"
            neighbor=Iof::Data::NO::ManagedNetworkComponent.new()
            neighbor.category=3
            neighbor.name="host_#{mac.to_s}"
            l2=Iof::Data::NO::EthernetInterface.new()
            l2.name="l2_eth"
            l2[:no_mac_address]=@data_handler.get_mac_address_name(mac.to_s)
            neighbor[:no_endpoint]=true
            #FIXME check for safety via @target_handler
            neighbor[:no_identifier]=mac

            #mac only neighbor has no associated vlans; using interface configuration
            l2[:no_trunk_mode]=@interfaces[interface][:mode] unless @interfaces[interface][:mode].nil?()
            l2[:no_native_vlan]=@data_handler.get_vlan_name(@interfaces[interface][:native_vlan]) unless @interfaces[interface][:native_vlan].nil?()
            l2[:no_voice_vlan]=@data_handler.get_vlan_name(@interfaces[interface][:voice_vlan]) unless @interfaces[interface][:voice_vlan].nil?()
            l2[:no_tagged_vlan]=generate_vlan_name_list(@interfaces[interface][:allowed_vlans]) unless @interfaces[interface][:allowed_vlans].nil?()

            l1=Iof::Data::NO::Layer1Interface.new()
            l1.name="l1"
            l1[:no_interface_type]=:physical
            l1[:no_ethernet_interface]=l2
            neighbor[:no_simple_layer1_interface]=l1
            @neighbors << neighbor
            connection=Iof::Data::NO::IsConnectedTo.new()
            connection.subject=me.clone()
            connection.object="#{neighbor.name}___#{l1.name}"
            @connections << connection
          else
            #create unmanaged switch and connect mac neighbors
            me="#{component.name}___#{Iof::Data::Base.sanitize_name(@interfaces[interface][:name])}"
            switch=Iof::Data::NO::UnmanagedNetworkComponent.new()
            switch.category=2
            switch.name="umnc_at_#{component.name}_#{@interfaces[interface][:name]}"
            switch[:no_switch]=true
            l1=Iof::Data::NO::Layer1Interface.new()
            l1[:no_interface_type]=:physical
            l1.name="upstream"
            switch[:no_simple_layer1_interface]=l1
            @components << switch
            connection=Iof::Data::NO::IsConnectedTo.new()
            connection.subject=me.clone()
            connection.object="#{switch.name}___#{l1.name}"
            @connections << connection
            macs.each do |m|
              l1_sw=Iof::Data::NO::Layer1Interface.new()
              l1_sw[:no_interface_type]=:physical
              l1_sw.name="l1_to_#{m}"
              switch[:no_simple_layer1_interface]=l1_sw
              sw="#{switch.name}___#{l1_sw.name}"

              neighbor=Iof::Data::NO::ManagedNetworkComponent.new()
              neighbor.category=3
              neighbor.name="host_#{m.to_s}"
              l2=Iof::Data::NO::EthernetInterface.new()
              l2.name="l2_eth"
              l2[:no_mac_address]=@data_handler.get_mac_address_name(m.to_s)
              neighbor[:no_endpoint]=true
              #FIXME check for safety via @target_handler
              neighbor[:no_identifier]=m

              #mac only neighbor has no associated vlans; using interface configuration
              l2[:no_trunk_mode]=@interfaces[interface][:mode] unless @interfaces[interface][:mode].nil?()
              l2[:no_native_vlan]=@data_handler.get_vlan_name(@interfaces[interface][:native_vlan]) unless @interfaces[interface][:native_vlan].nil?()
              l2[:no_voice_vlan]=@data_handler.get_vlan_name(@interfaces[interface][:voice_vlan]) unless @interfaces[interface][:voice_vlan].nil?()
              l2[:no_tagged_vlan]=generate_vlan_name_list(@interfaces[interface][:allowed_vlans]) unless @interfaces[interface][:allowed_vlans].nil?()

              l1=Iof::Data::NO::Layer1Interface.new()
              l1.name="l1"
              l1[:no_interface_type]=:physical
              l1[:no_ethernet_interface]=l2
              neighbor[:no_simple_layer1_interface]=l1
              @neighbors << neighbor
              connection=Iof::Data::NO::IsConnectedTo.new()
              connection.subject=sw
              connection.object="#{neighbor.name}___#{l1.name}"
              @connections << connection
            end
          end
        end

        @components=sub_components.values+[component]+@components


        @interfaces=nil
        @acls_list=nil
        @acls_yaml=nil
        @ipv4_forwards=nil
        @ipv6_forwards=nil

        @temporary_storage=nil
        @mac_addresses=nil
        @mac_by_interface=nil
        @mac_by_interface_parsed=nil
        @interface_by_mac=nil

        @vtp_vlan_by_interface=nil
        @vtp_vlans=nil

        @power_outgoing=nil
        @powered_devices=nil
        @cdp_neighbors_by_interface=nil
      end

      def clone_acl_list(acl)
        return nil if acl.nil?()
        container=nil
        head=nil
        tail=nil
        if acl.is_a?(Iof::Data::NO::Layer4ACL) then
          container=Iof::Data::NO::Layer4ACL.new()
          head=:no_layer4_acl_entry
          tail=:no_next_layer4_acl_entry
        elsif acl.is_a?(Iof::Data::NO::IPv4ForwardSet) then
          container=Iof::Data::NO::IPv4ForwardSet.new()
          head=:no_layer3_forward
          tail=:no_next_layer3_forward
        elsif acl.is_a?(Iof::Data::NO::IPv6ForwardSet) then
          container=Iof::Data::NO::IPv6ForwardSet.new()
          head=:no_layer3_forward
          tail=:no_next_layer3_forward
        else
          error "Unknown data strcuture '#{acl.class.to_s}'"
        end
        current_old=acl[head]
        container[head]=current_old.clone(false) unless current_old.nil?()
        current_new=container[head]

        until current_old.nil?() do
          current_new[tail]=current_old[tail].clone(false) unless current_old[tail].nil?()
          current_old=current_old[tail]
          current_new=current_new[tail]
        end
        container
      end
      alias :clone_forwards_list :clone_acl_list

      def acl_array_to_list(acl,protocol=nil)
        container=nil
        head=nil
        tail=nil
        if protocol.nil?() then
          container=Iof::Data::NO::Layer4ACL.new()
          head=:no_layer4_acl_entry
          tail=:no_next_layer4_acl_entry
        elsif protocol==:ipv4 then
          container=Iof::Data::NO::IPv4ForwardSet.new()
          head=:no_layer3_forward
          tail=:no_next_layer3_forward
        elsif protocol==:ipv6 then
          container=Iof::Data::NO::IPv6ForwardSet.new()
          head=:no_layer3_forward
          tail=:no_next_layer3_forward
        else
          error "Invalid protocol '#{protocol.to_s}' for calling '{acl|forwards}_array_to_list'"
        end
        current=nil
        acl.each do |a|
          next if a.nil?()
          if container[head].nil?() then
            container[head] = a
            current=container[head]
          elsif current.nil?() then
            error "Something gone wrong", "ACL may be truncated", "input: #{acl}", "current state #{container}"
            container[head] = a
            current=container[head]
          else
            current[tail]=a
            current=current[tail]
          end
        end
        container
      end
      alias :forwards_array_to_list :acl_array_to_list

      def concat_acl_lists(acl1,acl2)
        current=acl1[:no_layer4_acl_entry]
        done=false
        until done do
          if !current[:no_next_layer4_acl_entry].nil?() && current[:no_next_layer4_acl_entry][:no_next_layer4_acl_entry].nil?() then
            current[:no_next_layer4_acl_entry]=acl2[:no_next_layer4_acl_entry]
            done=true
          else
            current=current[:no_next_layer4_acl_entry]
          end
        end
        acl1
      end

      def concat_acl_yaml(acl1,acl2)
        a1=YAML::load(acl1)
        a2=YAML::load(acl2)
        a1_sliced=a1.slice(0..-2)
        a1_sliced.concat(a2)
        a1_sliced.to_yaml
      end

      def send_results
        debug "Sending results to data handler"
        @components.each do |c|
          debug "Sending component #{c.name}"
          @data_handler.set_component(c,c.category)
        end
        @neighbors.each do |n|
          debug "Sending neighbor #{n.name}"
          @data_handler.set_component(n,n.category)
        end
        @connections.each do |c|
          debug "Sending connection #{c.subject}<->#{c.object}"
          @data_handler.set_connection(c)
        end
        @components=nil
        @neighbors=nil
        @connections=nil
      end
    end
  end
end
