# encoding: utf-8
# license: gpl2p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST
 #FIXME add mutex
 module Iof
  module Query
    class Lookup < Ghun::Base::Module

      #NEVER configure dependency cycles
      DEPENDENCIES = {
        :layer1_connections => [:switches, :switch_neighbors],
        :all_layer1_connections => [:switches, :switch_neighbors],
        :switches => [],
        :switch_neighbors => [:switches],
        :voice_vlans => [],
        :voice_switches => [:voice_vlans],
        :mac_address_to_interface => [],
        :ipv4_address_to_interface => [],
        :ipv4_address_to_network => [:ipv4_address_to_interface,:ipv4_network_to_interface],
        :ipv4_network_to_interface => [],
        :ipv6_address_to_interface => [],
        :ipv6_address_to_network => [:ipv6_address_to_interface,:ipv6_network_to_interface],
        :ipv6_network_to_interface => [],
        :voice_vlan_to_interface_slow => [],
        :access_vlan_to_interface_slow => [],
        :tagged_vlan_to_interface_slow => [],
        :vtp_vlan_to_interface_slow => [],
        :voice_vlan_to_interface_fast => [],
        :access_vlan_to_interface_fast => [],
        :tagged_vlan_to_interface_fast => [],
        :vtp_vlan_to_interface_fast => [],
        :voice_vlan_to_interface => [:voice_vlan_to_interface_slow],
        :access_vlan_to_interface => [:access_vlan_to_interface_slow],
        :tagged_vlan_to_interface => [:tagged_vlan_to_interface_slow],
        :vtp_vlan_to_interface => [:vtp_vlan_to_interface_slow],
        :vlan_to_interface_slow => [:voice_vlan_to_interface_slow, :access_vlan_to_interface_slow, :tagged_vlan_to_interface_slow, :vtp_vlan_to_interface_slow],
        :vlan_to_interface_fast => [:voice_vlan_to_interface_fast, :access_vlan_to_interface_fast, :tagged_vlan_to_interface_fast, :vtp_vlan_to_interface_fast],
        :vlan_to_interface => [:voice_vlan_to_interface, :access_vlan_to_interface, :tagged_vlan_to_interface, :vtp_vlan_to_interface],
        :vlan_to_interface_slow_fast_equality => [:vlan_to_interface_slow,:vlan_to_interface_fast],
        :interface_to_vlan => [:vlan_to_interface],
      }

      def initialize(ontology)
        super(Ghun::Log::Source::QUERY)
        @ontology=ontology
        @timer=Ghun::Timing::Timer.new(Ghun::Log::Source::QUERY)

        @ipv4_addresses=Array.new()
        @ipv6_addresses=Array.new()
        @ipv4_networks=Array.new()
        @ipv6_networks=Array.new()
        @timer.register(:ipv4an,"IPv4 address to network")
        @timer.register(:ipv4ai,"IPv4 address to interface")
        @timer.register(:ipv4ni,"IPv4 network to interface")
        @timer.register(:ipv6an,"IPv6 address to network")
        @timer.register(:ipv6ai,"IPv6 address to interface")
        @timer.register(:ipv6ni,"IPv6 network to interface")
        @timer.register(:ethai,"MAC address to interface")
        @timer.register(:sw,"All switches")
        @timer.register(:neighbors,"All switch neighbors")
        @timer.register(:bfs,"BFS")
        @timer.register(:vvlan,"Voice Vlans")
        @timer.register(:voice_slow,"Voice vlans to interface slow")
        @timer.register(:access_slow,"Access vlans to interface slow")
        @timer.register(:tagged_slow,"Tagged vlans to interface slow")
        @timer.register(:vtp_slow,"VTP vlans to interface slow")
        @timer.register(:voice_fast,"Voice vlans to interface fast")
        @timer.register(:access_fast,"Access vlans to interface fast")
        @timer.register(:tagged_fast,"Tagged vlans to interface fast")
        @timer.register(:vtp_fast,"VTP vlans to interface fast")
        @timer.register(:vlans_slow,"Vlans to interface slow")
        @timer.register(:vlans_fast,"Vlans to interface fast")
        @timer.register(:vlans,"Vlans to interface")
        @timer.register(:vlans_equal,"Vlans to interface slow/fast equality check")

        prepare_reverse_dependencies()
        @lookup_read=0
        @lookup_write=0
        @required_lookup_tables=Array.new()
        @dependency_lookup_tables=Array.new()
        @valid_lookup_tables=Hash.new()
        @invalid_lookup_tables=Array.new()
        @known_query_modules=Hash.new()
      end

      ####
      #### Public lookup table methods
      ####

      #temp method
      def thread_optimization()
        Iof::Timing.register(:thread_optimization,"Thread number optimization for huge vlan results")
        Iof::Timing.start(:thread_optimization)
        @ontology.finish
        [2,4,8,16,64].each do |t|
          Iof::Timing.start(:reasoner)
          @ontology=Iof::Ontology::JenaGlue.new(@logger,false,false,true)
          @ontology.start
          Iof::Timing.stop(:reasoner)
          Iof::Timing.register("thread_optimization_#{t}".intern,"Thread number optimization for huge vlan results (#{t} threads)")
          Iof::Timing.start("thread_optimization_#{t}".intern)
          create_lookup_x_vlan_to_interface_helper_fast("no:hasTaggedVlan",t)
          Iof::Timing.stop("thread_optimization_#{t}".intern)
          @ontology.finish
        end
        Iof::Timing.stop(:thread_optimization)
      end

      def get_lookup_table(table,*args)
        @lookup_read+=1
        if args.nil?() || args.empty?() then
          return @valid_lookup_tables[table]
        else
          result=@valid_lookup_tables[table]
          args.each do |a|
            next if a.nil?()
            if result.is_a?(Hash) && result.include?(a) then
              result=result[a]
            else
              result=nil
              break
            end
          end unless result.nil?()
          return result
        end
      end

      def generate_lookup_tables()
        invalid_lookup_tables=@invalid_lookup_tables
        @invalid_lookup_tables=[]
        invalid_lookup_tables.each do |t|
          if @valid_lookup_tables.include?(t) then
            debug "Already updated lookup table '#{t.to_s}' after last request"
            next
          else
            generate_lookup_table(t)
          end
        end
      end

      def register_query_module(mod,required_lookup_tables=[])
        warn "Query module '#{mod.to_s}' already known, updating required lookup tables" if @known_query_modules.include?(mod)
        @known_query_modules[mod]=required_lookup_tables
        generate_required_tables()
        cleanup_lookup_tables()
      end

      def unregister_query_module(mod)
        if @known_query_modules.include?(mod) then
          @known_query_modules.delete(mod)
          generate_required_tables()
          cleanup_lookup_tables()
        else
          warn "Query module '#{mod.to_s}' not known"
        end
      end

      def request_lookup_table_update(table)
        if @valid_lookup_tables.include?(table) then
          r_deps=@reverse_deps[table]
          r_deps.each do |r|
            request_lookup_table_update(r)
          end
          @invalid_lookup_tables << table
          @valid_lookup_tables.delete(table)
        elsif @invalid_lookup_tables.include?(table) then
          debug "Lookup table '#{table.to_s}' already marked for update"
        else
          warn "Lookup table '#{table.to_s}' unknown"
        end
      end

      #FIXME integrate in new structure
      def add_ipv4_address(address)
        @lookup_write+=1
        generate_lookup_table(:ipv4_address_to_network,true,address)
      end

      def add_ipv6_address(address)
        @lookup_write+=1
        generate_lookup_table(:ipv6_address_to_network,true,address)
      end

      ####
      #### Internal lookup table management methods
      ####

    private

      def generate_lookup_table(table,append=false,*args)
        deps=@deps[table]
        deps.each do |d|
          generate_lookup_table(d) unless @valid_lookup_tables.include?(d)
        end unless deps.nil?()
        method="create_lookup_#{table.to_s}".intern
        debug "Running method #{method.to_s}"
        if @valid_lookup_tables.include?(table) && !append then
          debug "Already updated lookup table '#{table.to_s}' after last request"
          return
        end
        result=nil
        begin
          result=self.send(method,*args)
        rescue => e
          error "Failed to generate lookup table '#{table.to_s}'", e
        end

        if append then
          if result.is_a?(Array) then
            @valid_lookup_tables[table]=Array.new() unless @valid_lookup_tables.include?(table) || !@valid_lookup_tables[table].is_a?(Array)
            @valid_lookup_tables[table].merge(result)
          elsif result.is_a?(Hash) then
            @valid_lookup_tables[table]=Hash.new() unless @valid_lookup_tables.include?(table) || !@valid_lookup_tables[table].is_a?(Hash)
            @valid_lookup_tables[table].merge(result)
          else
            warn "No append supported for class '#{result.class}'"
            @valid_lookup_tables[table]=result
          end
        else
          @valid_lookup_tables[table]=result
        end
      end


      def prepare_reverse_dependencies()
        @deps=Query::Lookup::DEPENDENCIES
        @reverse_deps=Hash.new()
        @deps.each do |table,deps|
          deps.each do |d|
            @reverse_deps[d]=Array.new() unless @reverse_deps.include?(d)
            @reverse_deps[d] << table
          end
        end
      end

      def generate_required_tables()
        required_lookup_tables=Array.new()
        dependency_lookup_tables=Array.new()
        @known_query_modules.each do |mod,tables|
          required_lookup_tables.concat(tables)
          tables.each do |t|
            dependency_lookup_tables.concat(@deps[t]) unless @deps[t].nil?()
          end
        end
        required_lookup_tables.uniq!
        dependency_lookup_tables.uniq!
        @required_lookup_tables=required_lookup_tables
        @dependency_lookup_tables=dependency_lookup_tables
      end

      def cleanup_lookup_tables()
        @valid_lookup_tables.each do |table,values|
          @valid_lookup_tables.delete(table) unless @required_lookup_tables.include?(table) || @dependency_lookup_tables.include?(table)
        end
        @invalid_lookup_tables.each do |table,values|
          @invalid_lookup_tables.delete(table) unless @required_lookup_tables.include?(table) || @dependency_lookup_tables.include?(table)
        end
        @required_lookup_tables.each do |t|
          @invalid_lookup_tables << t unless @valid_lookup_tables.include?(t) || @invalid_lookup_tables.include?(t)
        end
      end

      ####
      #### Misc internal methods
      ####

      def output_counter(message=nil)
        debug Ghun::Log::Type::PROFILING, message unless message.nil?()
        @ontology.output_counter()
        debug Ghun::Log::Type::PROFILING, "Handled #{@lookup_read} lookup table reads"
        debug Ghun::Log::Type::PROFILING, "Handled #{@lookup_write} additions to lookup tables"
      end

      def reset_counter()
        @ontology.reset_counter()
        @lookup_read=0
        @lookup_write=0
      end

      def output_and_reset_counter(message=nil)
        output_counter(message)
        reset_counter()
      end

      ####
      ####  Internal lookup table creation
      ####

      def create_lookup_switches(*args)
        switches=Array.new()
        debug "Querying 'switches' data from ontology"
        result=@ontology.query("SELECT DISTINCT ?c WHERE { ?c a no:Switch }")
        debug "Got result #{result.inspect}" unless result.nil?() || result.empty?()
        debug "Got no or empty result" if result.nil?() || result.empty?()
        result.each do |r|
          switches << r[:c]
        end
        switches.sort!
        switches.uniq!
        return switches
      end

      def create_lookup_switch_neighbors(*args)
        neighbors=Hash.new()
        get_lookup_table(:switches).each do |node|
          debug "Prefetching neighbors for #{node}"
          neighbors[node.intern]=@ontology.query("SELECT ?c2 ?i1_l1 ?i2_l1 WHERE { ?c2 a no:Switch . no:#{node} no:hasLayer1Connection ?c2 . no:#{node} no:hasLayer1Interface ?i1_l1 . ?c2 no:hasLayer1Interface ?i2_l1 . ?i1_l1 no:isConnectedTo ?i2_l1 }")
          neighbors[node.intern].each do |n|
            n[:c1]=node
          end
        end
        return neighbors
      end

      def create_lookup_voice_vlans(*args)
        debug "Querying 'voice vlans' data from ontology"
        voice_vlans=Array.new()
        result=Array.new()
        result_raw_voice_on_phones=@ontology.query("SELECT DISTINCT ?voice WHERE { ?tmp a no:ManagedNetworkComponent . ?tmp no:isPhone \"true\" . ?tmp no:hasLayer1Interface ?tmp_l1 . ?tmp_l1 no:hasLayer2Interface ?tmp_l2 . ?tmp_l2 no:hasVoiceVlan ?tmp_voice . ?tmp_voice no:hasVlanID ?voice }",true)
        result_raw_voice_on_phones.map { |v| result << v[:voice].to_i }
        result_raw_voice_on_switches=@ontology.query("SELECT DISTINCT ?voice WHERE { ?tmp a no:Switch . ?tmp no:hasLayer1Interface ?tmp_l1 . ?tmp_l1 no:hasLayer2Interface ?tmp_l2 . ?tmp_l2 no:hasVoiceVlan ?tmp_voice . ?tmp_voice no:hasVlanID ?voice }",true)
        result_raw_voice_on_switches.map { |v| result << v[:voice].to_i }
        debug "Got result #{result.inspect}" unless result.nil?() || result.empty?()
        debug "Got no or empty result" if result.nil?() || result.empty?()
        result.sort!
        result.uniq!
        result.each do |r|
          voice_vlans << r.to_s
        end
        return voice_vlans
      end

      def create_lookup_voice_switches(*args)
        voice_switches=Array.new()
        debug "Querying 'voice switches' data from ontology"
        result=@ontology.query("SELECT DISTINCT ?c WHERE { ?c a no:Switch . ?c no:hasLayer1Connection ?p . ?p a no:ManagedNetworkComponent . ?p no:isPhone \"true\" }")
        debug "Got result #{result.inspect}" unless result.nil?() || result.empty?()
        debug "Got result #{result.inspect}" unless result.nil?() || result.empty?()
        result.each do |s|
          voice_switches << s[:c]
        end

        get_lookup_table(:voice_vlans).each do |vlan|
          result=@ontology.query("SELECT DISTINCT ?c WHERE { ?c a no:Switch . ?c no:hasLayer1Interface ?c_l1 . ?c_l1 no:hasLayer2Interface ?c_l2 . ?c_l2 no:hasVlan ?tmp1 . ?tmp1 no:hasVlanID \"#{vlan}\" }")
          debug "Got result #{result.inspect}" unless result.nil?() || result.empty?()
          debug "Got result #{result.inspect}" unless result.nil?() || result.empty?()
          result.map { |s| voice_switches << s[:c]}
        end
        voice_switches.sort!
        voice_switches.uniq!
        return voice_switches
      end

      def create_lookup_ipv4_network_to_interface(*args)
        ipv4_network_to_interface=Hash.new
        debug "Querying 'ipv4 network to interface' data from ontology"
        result_mask=@ontology.query("SELECT DISTINCT ?c ?i_l1 ?i_l2 ?i_l3 ?v4_addr ?v4_netmask ?v4_length WHERE {?c a no:Switch .  ?c no:hasLayer1Interface ?i_l1 . ?i_l1 no:hasEthernetInterface ?i_l2 . ?i_l2 no:hasIPv4Interface ?i_l3 . ?i_l3 no:hasIPv4Network ?tmp1 . ?tmp1 no:hasIPv4AddressValue ?v4_addr . ?tmp1 no:hasIPv4NetmaskValue ?v4_netmask . OPTIONAL { ?tmp1 no:hasIPv4NetmaskLength ?v4_length } }",true)
        result_length=@ontology.query("SELECT DISTINCT ?c ?i_l1 ?i_l2 ?i_l3 ?v4_addr ?v4_netmask ?v4_length WHERE {?c a no:Switch .  ?c no:hasLayer1Interface ?i_l1 . ?i_l1 no:hasEthernetInterface ?i_l2 . ?i_l2 no:hasIPv4Interface ?i_l3 . ?i_l3 no:hasIPv4Network ?tmp1 . ?tmp1 no:hasIPv4AddressValue ?v4_addr . ?tmp1 no:hasIPv4NetmaskLength ?v4_length . OPTIONAL { ?tmp1 no:hasIPv4NetmaskValue ?v4_netmask } }",true)
        result=(result_mask+result_length).uniq
        debug "Got result #{result.inspect}" unless result.nil?() || result.empty?()
        debug "Got no or empty result" if result.nil?() || result.empty?()
        result.each do |r|
          network=nil
          unless r[:v4_length].nil?() then
            debug "Found network #{r[:v4_addr]}/#{r[:v4_length]}"
            @ipv4_networks << IPAddress::IPv4.new("#{r[:v4_addr]}/#{r[:v4_length]}")
            network=Iof::Data.ipv4_network_part(r[:v4_addr],r[:v4_length])
          else
            mask=Iof::Data.ipv4_to_netmask(r[:v4_netmask])
            debug "Found network #{r[:v4_addr]}/#{mask}"
            @ipv4_networks << IPAddress::IPv4.new("#{r[:v4_addr]}/#{mask}")
            network=Iof::Data.ipv4_network_part(r[:v4_addr],r[:v4_netmask])
          end
          interface=Layer3Interface.new(r[:c],r[:i_l1],r[:i_l2],r[:i_l3])
          if ipv4_network_to_interface.include?(network.intern) then
            ipv4_network_to_interface[network.intern] << interface
          else
            ipv4_network_to_interface[network.intern]=[interface]
          end unless network.nil?()
        end
        return ipv4_network_to_interface
      end

      def create_lookup_ipv6_network_to_interface(*args)
        ipv6_network_to_interface=Hash.new()
        debug "Querying 'ipv6 network to interface' data from ontology"
        result=@ontology.query("SELECT DISTINCT ?c ?i_l1 ?i_l2 ?i_l3 ?v6_addr ?v6_length WHERE {?c a no:Switch . ?c no:hasLayer1Interface ?i_l1 . ?i_l1 no:hasEthernetInterface ?i_l2 . ?i_l2 no:hasIPv6Interface ?i_l3 . ?i_l3 no:hasIPv6Network ?tmp1 . ?tmp1 no:hasIPv6AddressValue ?v6_addr . ?tmp1 no:hasIPv6NetmaskLength ?v6_length }",true)
        debug "Got result #{result.inspect}" unless result.nil?() || result.empty?()
        debug "Got no or empty result" if result.nil?() || result.empty?()
        result.each do |r|
          network=Iof::Data.ipv6_network_part(r[:v6_addr],r[:v6_length])
          @ipv6_networks << IPAddress::IPv6.new("#{r[:v6_addr]}/#{r[:v6_length]}")
          interface=Layer3Interface.new(r[:c],r[:i_l1],r[:i_l2],r[:i_l3])
          if ipv6_network_to_interface.include?(network.intern) then
            ipv6_network_to_interface[network.intern] << interface
          else
            ipv6_network_to_interface[network.intern]=[interface]
          end
        end
        return ipv6_network_to_interface
      end

      def create_lookup_ipv4_address_to_network(*args)
        ipv4_address_to_network=Hash.new()
        args.each do |a|
          @ipv4_addresses << IPAddress::IPv4.new(a) unless a.nil?() || a==''
        end unless args.nil?()
        @ipv4_addresses.each do |a|
          debug "Searching network for ipv4 address #{a.to_s}"
          if ipv4_address_to_network.include?(a.to_s.intern) then
            debug "Found existing mapping '#{ipv4_address_to_network[a.to_s.intern].to_s}' for address '#{a.to_s}'" unless ipv4_address_to_network[a.to_s.intern].nil?()
            debug "Found no mapping for address '#{a.to_s}' in prior run" if ipv4_address_to_network[a.to_s.intern].nil?()
            next
          end
          match=nil
          @ipv4_networks.each do |n|
            if match.nil?() then
              match=n
            else
              old_length=match.prefix.to_i
              new_length=n.prefix.to_i
              if old_length < new_length then
                #Better match
                match=n
              elsif old_length > new_length
                #Worse match
                next
              else
                #equal match
                next if match.network.to_s == n.network.to_s
                #conflicting matches
                error "Network '#{match.network.to_s}/#{match.prefix.to_s}' '#{n.network.to_s}/#{n.prefix.to_s}' conflict for address '#{a.to_s}'"
              end
            end if n.include?(a)
          end
          if match.nil?() then
            warn "Found no match for address '#{a.to_s}'"
            ipv4_address_to_network[a.to_s.intern]=nil
          else
            debug "Found match '#{match.network.to_s}' for address '#{a.to_s}'"
            ipv4_address_to_network[a.to_s.intern]=match.network.to_s.intern
          end
        end
        @ipv4_addresses.clear
        return ipv4_address_to_network
      end

      def create_lookup_ipv6_address_to_network(*args)
        ipv6_address_to_network=Hash.new()
        args.each do |a|
          @ipv6_addresses << IPAddress::IPv6.new(a) unless a.nil?() || a==''
        end unless args.nil?()
        @ipv6_addresses.each do |a|
          debug "Searching network for ipv6 address #{a.to_s}"
          if ipv6_address_to_network.include?(a.to_s.intern) then
            debug "Found existing mapping '#{ipv6_address_to_network[a.to_s.intern].to_s}' for address '#{a.to_s}'"
            next
          end
          match=nil
          @ipv6_networks.each do |n|
            if match.nil?() then
              match=n
            else
              old_length=match.prefix.to_i
              new_length=n.prefix.to_i
              if old_length < new_length then
                #Better match
                match=n
              elsif old_length > new_length
                #Worse match
                next
              else
                #equal match
                next if match.network.to_s == n.network.to_s
                #conflicting matches
                error "Network '#{match.network.to_s}/#{match.prefix.to_s}' '#{n.network.to_s}/#{n.prefix.to_s}' conflict for address '#{a.to_s}'"
              end
            end if n.include?(a)
          end
          if match.nil?() then
            warn "Found no match for address '#{a.to_s}'"
            ipv6_address_to_network[a.to_s.intern]=nil
          else
            debug "Found match '#{match.network.to_s}' for address '#{a.to_s}'"
            ipv6_address_to_network[a.to_s.intern]=match.network.to_s.intern
          end
        end
        @ipv6_addresses.clear
        return ipv6_address_to_network
      end

      def create_lookup_layer1_connections_bfs(*args)
        layer1_connections=Hash.new()
        get_lookup_table(:switches).each do |s|
          layer1_connections[s.intern]=bfs(s.intern)
        end
        return layer1_connections
      end
      alias :create_lookup_layer1_connections :create_lookup_layer1_connections_bfs

      def bfs(start_node)
        debug "BFS for start node #{start_node}"
        path=Hash.new()
        visited=Hash.new
        queue=Array.new()
        queue.push([start_node,Iof::Query::Path.new()])
        visited[start_node]=1
        until queue.empty?() do
          node=queue.shift()
          break if node.nil?()
          path[node[0]]=node[1]
          neighbors=get_lookup_table(:switch_neighbors,node[0])
          neighbors.each do |n|
            if visited[n[:c2].intern].nil?() then
              visited[n[:c2].intern]=1
              p=node[1].clone()
              p << Iof::Query::PathEntry.new(n[:c1],n[:i1_l1])
              p << Iof::Query::PathEntry.new(n[:c2],n[:i2_l1])
              queue.push([n[:c2].intern,p])
            end
          end
        end
        path
      rescue =>e
        error "Failed to calculate path for start node #{start_node}", e
      end

      def create_lookup_all_layer1_connections_bfs(*args)
        all_layer1_connections=Hash.new()
        get_lookup_table(:switches).each do |s|
          all_layer1_connections[s.intern]=all_bfs(s.intern)
        end
        return all_layer1_connections
      end
      alias :create_lookup_all_layer1_connections :create_lookup_all_layer1_connections_bfs

      def all_bfs(start_node)
        debug "BFS for start node #{start_node}"
        path=Hash.new()
        visited=Hash.new
        queue=Array.new()
        queue.push([start_node,Iof::Query::Path.new()])
        visited[start_node]=1
        node=nil
        until queue.empty?() do
          last=node
          node=queue.shift()
          break if node.nil?()
          if path.include?(node[0])
            path[node[0]] << node[1] unless path[node[0]].include?(node[1])
          else
            path[node[0]]=[node[1]]
          end
          neighbors=get_lookup_table(:switch_neighbors,node[0])
          neighbors.each do |n|
            if visited[n[:c2].intern].nil?() then
              #never visited before
              visited[n[:c2].intern]=1
              p=node[1].clone()
              p << Iof::Query::PathEntry.new(n[:c1],n[:i1_l1])
              p << Iof::Query::PathEntry.new(n[:c2],n[:i2_l1])
              queue.push([n[:c2].intern,p])
            elsif start_node!=n[:c2] && last!=n[:c2] && !node[1].include?(n[:c2])
              #node already visited for this start node
              visited[n[:c2].intern]+=1
              p=node[1].clone()
              p << Iof::Query::PathEntry.new(n[:c1],n[:i1_l1])
              p << Iof::Query::PathEntry.new(n[:c2],n[:i2_l1])
              queue.push([n[:c2].intern,p])
            end
          end
        end
        path
      rescue =>e
        error "Failed to calculate path for start node #{start_node}", e
      end



      def create_lookup_ipv4_address_to_interface(*args)
        ipv4_address_to_interface=Hash.new()
        debug "Querying 'ipv4 address to interface' data from ontology"
        result=@ontology.query("SELECT DISTINCT ?c ?i_l1 ?i_l2 ?i_l3 ?v4 WHERE {?c a no:ManagedNetworkComponent . ?c no:hasLayer1Interface ?i_l1 . ?i_l1 no:hasEthernetInterface ?i_l2 . ?i_l2 no:hasIPv4Interface ?i_l3 . ?i_l3 no:hasIPv4Address ?tmp1 . ?tmp1 no:hasIPv4AddressValue ?v4}",true)
        debug "Got result #{result.inspect}" unless result.nil?() || result.empty?()
        debug "Got no or empty result" if result.nil?() || result.empty?()
        result.each do |r|
          v4=r[:v4]
          @ipv4_addresses << IPAddress::IPv4.new(v4)
          interface=Layer3Interface.new(r[:c],r[:i_l1],r[:i_l2],r[:i_l3])
          if ipv4_address_to_interface.include?(v4.intern) then
            ipv4_address_to_interface[v4.intern] << interface
            warn "IPv4 address #{v4} is not unique", "Got '#{ipv4_address_to_interface[v4.intern].inspect}' and '#{interface.inspect}'"
          else
            ipv4_address_to_interface[v4.intern]=[interface]
          end
        end
        return ipv4_address_to_interface
      end

      def create_lookup_ipv6_address_to_interface(*args)
        ipv6_address_to_interface=Hash.new()
        debug "Querying 'ipv6 address to interface' data from ontology"
        result=@ontology.query("SELECT DISTINCT ?c ?i_l1 ?i_l2 ?i_l3 ?v6 WHERE {?c a no:ManagedNetworkComponent . ?c no:hasLayer1Interface ?i_l1 . ?i_l1 no:hasEthernetInterface ?i_l2 . ?i_l2 no:hasIPv6Interface ?i_l3 . ?i_l3 no:hasIPv6Address ?tmp1 . ?tmp1 no:hasIPv6AddressValue ?v6}",true)
        debug "Got result #{result.inspect}" unless result.nil?() || result.empty?()
        debug "Got no or empty result" if result.nil?() || result.empty?()
        result.each do |r|
          v6=r[:v6]
          @ipv6_addresses << IPAddress::IPv6.new(v6)
          interface=Layer3Interface.new(r[:c],r[:i_l1],r[:i_l2],r[:i_l3])
          if ipv6_address_to_interface.include?(v6.intern) then
            ipv6_address_to_interface[v6.intern] << interface
            warn "IPv6 address #{v6} is not unique", "Got '#{ipv6_address_to_interface[v6.intern].inspect}' and '#{interface.inspect}'"
          else
            ipv6_address_to_interface[v6.intern]=[interface]
          end
        end
        return ipv6_address_to_interface
      end

      def create_lookup_mac_address_to_interface(*args)
        mac_address_to_interface=Hash.new()
        debug "Querying 'mac address to interface' data from ontology"
        result=@ontology.query("SELECT DISTINCT ?c ?i_l1 ?i_l2 ?m WHERE {?c a no:ManagedNetworkComponent . ?c no:hasLayer1Interface ?i_l1 . ?i_l1 no:hasEthernetInterface ?i_l2 . ?i_l2 no:hasMacAddress ?tmp1 . ?tmp1 no:hasMacAddressValue ?m}",true)
        debug "Got result #{result.inspect}" unless result.nil?() || result.empty?()
        debug "Got no or empty result" if result.nil?() || result.empty?()
        result.each do |r|
          mac=r[:m]
          interface=Layer2Interface.new(r[:c],r[:i_l1],r[:i_l2])
          if mac_address_to_interface.include?(mac.intern) then
            next if mac_address_to_interface[mac.intern]==interface
            warn "MAC address #{mac} is not unique, using only first occurence", "Got '#{mac_address_to_interface[mac.intern].inspect}' and '#{interface.inspect}'"
          else
            mac_address_to_interface[mac.intern]=interface
          end
        end
        return mac_address_to_interface
      end

      def create_lookup_x_vlan_to_interface_helper(property_name)
        vlans=Hash.new()
        result=@ontology.query("SELECT DISTINCT ?c ?l1 ?l2 ?v WHERE { ?c a no:Switch . ?c no:hasLayer1Interface ?l1 . ?l1 no:hasLayer2Interface ?l2 . ?l2 #{property_name} ?vlan . ?vlan no:hasVlanID ?v }",true)
        foo=0
        result.each do |v|
          foo+=1
          vlans[v[:v].to_i]=Array.new() unless vlans.include?(v[:v].to_i)
          l2=Layer2Interface.new(v[:c],v[:l1],v[:l2])
          vlans[v[:v].to_i] << l2 unless vlans[v[:v].to_i].include?(l2)
        end unless result.nil?()
        debug "Got no or empty result" if result.nil?() || result.empty?()
        (0..4095).each do |v|
          vlans[v]=Array.new() unless vlans.include?(v)
          vlans[v].uniq!
        end
        return vlans
      end

      def create_lookup_x_vlan_to_interface_helper_huge_result(property_name)
        vlans=Hash.new()
        (0..4095).each do |v|
          vlans[v]=Array.new() unless vlans.include?(v)
#          result=@ontology.query("SELECT DISTINCT ?c ?l1 ?l2 WHERE { ?c a no:Switch . ?c no:hasLayer1Interface ?l1 . ?l1 no:hasLayer2Interface ?l2 . ?l2 #{property_name} ?vlan . ?vlan no:hasVlanID \"#{v.to_s}\" }",true)
          result=@ontology.query("SELECT DISTINCT ?c ?l1 ?l2 WHERE { ?c a no:Switch . ?c no:hasLayer1Interface ?l1 . ?l1 no:hasLayer2Interface ?l2 . ?l2 #{property_name} no:__vlan_#{v.to_s}__ }",true)
          result.each do |r|
            l2=Layer2Interface.new(r[:c],r[:l1],r[:l2])
            vlans[v] << l2 unless vlans.include?(l2)
          end unless result.nil?()
          debug "Got no or empty result for vlan #{v.to_s}" if result.nil?() || result.empty?()
        end
        return vlans
      end

      def create_lookup_x_vlan_to_interface_helper_fast_helper(vlan,result,property_name)
#        result_raw=@ontology.query("SELECT DISTINCT ?c ?l1 ?l2 WHERE { ?c a no:Switch . ?c no:hasLayer1Interface ?l1 . ?l1 no:hasLayer2Interface ?l2 . ?l2 #{property_name} ?vlan . ?vlan no:hasVlanID \"#{v.to_s}\" }",true)
        result_raw=@ontology.query("SELECT DISTINCT ?c ?l1 ?l2 WHERE { ?c a no:Switch . ?c no:hasLayer1Interface ?l1 . ?l1 no:hasLayer2Interface ?l2 . ?l2 #{property_name} no:__vlan_#{vlan.to_s}__ }",true)
        result_raw.each do |r|
          l2=Layer2Interface.new(r[:c],r[:l1],r[:l2])
          result << l2 unless result.include?(l2)
        end unless result_raw.nil?()
        debug "Got no or empty result for vlan #{vlan.to_s}" if result.nil?() || result.empty?()
      end

      def create_lookup_x_vlan_to_interface_helper_fast(property_name,number_of_threads=32)
        num_of_threads=2**Math.log2(number_of_threads).floor
        num_of_threads=[num_of_threads,4096].min
        num_of_threads=[1,num_of_threads].max
        num_of_steps=(4096/num_of_threads)-1
        vlans=Hash.new()
        (0..num_of_steps).each do |step|
          v_min=step*num_of_threads
          v_max=(step+1)*num_of_threads-1
          threads=Array.new()
          (v_min..v_max).each do |v|
            vlans[v]=Array.new
          end
          (0..(num_of_threads-1)).each do |v|
            v_real=step*num_of_threads+v
            threads[v]=Ghun::Base::Thread.run("query_vlan#{v_real}",0,@__log_source,@__log_type) do
              create_lookup_x_vlan_to_interface_helper_fast_helper(v_real,vlans[v_real],property_name)
            end
          end
          (0..(num_of_threads-1)).each do |v|
            v_real=step*num_of_threads+v
            threads[v].join
            threads.delete(v)
          end
        end
        return vlans
      end


      def create_lookup_voice_vlan_to_interface_fast(*args)
        debug "Querying 'voice vlan to interface' data from ontology"
        @timer.start(:voice_fast)
        result=create_lookup_x_vlan_to_interface_helper_fast("no:hasVoiceVlan")
        @timer.stop(:voice_fast)
        return result
      end

      def create_lookup_access_vlan_to_interface_fast(*args)
        debug "Querying 'access vlan to interface' data from ontology"
        @timer.start(:access_fast)
        result=create_lookup_x_vlan_to_interface_helper_fast("no:hasNativeVlan")
        @timer.stop(:access_fast)
        return result
      end

      def create_lookup_tagged_vlan_to_interface_fast(*args)
        debug "Querying 'tagged vlan to interface' data from ontology"
        @timer.start(:tagged_fast)
        result=create_lookup_x_vlan_to_interface_helper_fast("no:hasTaggedVlan")
        @timer.stop(:tagged_fast)
        return result
      end

      def create_lookup_vtp_vlan_to_interface_fast(*args)
        debug "Querying 'access vlan to interface' data from ontology"
        @timer.start(:vtp_fast)
        result=create_lookup_x_vlan_to_interface_helper_fast("no:hasVtpVlan")
        @timer.stop(:vtp_fast)
        return result
      end

      def create_lookup_voice_vlan_to_interface_slow(*args)
        debug "Querying 'voice vlan to interface' data from ontology"
        @timer.start(:voice_slow)
        result=create_lookup_x_vlan_to_interface_helper("no:hasVoiceVlan")
        @timer.stop(:voice_slow)
        return result
      end

      def create_lookup_access_vlan_to_interface_slow(*args)
        debug "Querying 'access vlan to interface' data from ontology"
        @timer.start(:acess_slow)
        result=create_lookup_x_vlan_to_interface_helper("no:hasNativeVlan")
        @timer.stop(:access_slow)
        return result
      end

      def create_lookup_tagged_vlan_to_interface_slow(*args)
        debug "Querying 'tagged vlan to interface' data from ontology"
        @timer.start(:tagged_slow)
        result=create_lookup_x_vlan_to_interface_helper_huge_result("no:hasTaggedVlan")
        @timer.stop(:tagged_slow)
        return result
      end

      def create_lookup_vtp_vlan_to_interface_slow(*args)
        debug "Querying 'access vlan to interface' data from ontology"
        @timer.start(:vtp_slow)
        result=create_lookup_x_vlan_to_interface_helper("no:hasVtpVlan")
        @timer.stop(:vtp_slow)
        return result
      end

      def create_lookup_voice_vlan_to_interface(*args)
        return get_lookup_table(DEPENDENCIES[:voice_vlan_to_interface][0])
      end

      def create_lookup_access_vlan_to_interface(*args)
        return get_lookup_table(DEPENDENCIES[:access_vlan_to_interface][0])
      end

      def create_lookup_tagged_vlan_to_interface(*args)
        return get_lookup_table(DEPENDENCIES[:tagged_vlan_to_interface][0])
      end

      def create_lookup_vtp_vlan_to_interface(*args)
        return get_lookup_table(DEPENDENCIES[:vtp_vlan_to_interface][0])
      end

      def create_lookup_vlan_to_interface_fast(*args)
        @timer.start(:vlans_fast)
        vlans=Hash.new()
        (0..4095).each do |v|
          vlans[v]=Array.new()
          DEPENDENCIES[:vlan_to_interface_fast].each do |d|
            vlans[v].concat(get_lookup_table(d, v)) unless get_lookup_table(d, v).nil?()
          end
          vlans[v].uniq!
        end
        @timer.stop(:vlans_fast)
        return vlans
      end

      def create_lookup_vlan_to_interface_slow(*args)
        @timer.start(:vlans_slow)
        vlans=Hash.new()
        (0..4095).each do |v|
          vlans[v]=Array.new()
          DEPENDENCIES[:vlan_to_interface].each do |d|
            vlans[v].concat(get_lookup_table(d, v)) unless get_lookup_table(d, v).nil?()
          end
          vlans[v].uniq!
        end
        @timer.stop(:vlans_slow)
        return vlans
      end

      def create_lookup_vlan_to_interface(*args)
        @timer.start(:vlans)
        vlans=Hash.new()
        (0..4095).each do |v|
          vlans[v]=Array.new()
          DEPENDENCIES[:vlan_to_interface].each do |d|
            vlans[v].concat(get_lookup_table(d, v)) unless get_lookup_table(d, v).nil?()
          end
          vlans[v].uniq!
        end
        @timer.stop(:vlans)
        return vlans
      end

      def create_lookup_interface_to_vlan(*args)
        components=Hash.new()
        (0..4095).each do |v|
          lookup=get_lookup_table(:vlan_to_interface,v)
          lookup.each do |interface|
            components[interface[:component].intern]=Hash.new() unless components.include?(interface[:component].intern)
            components[interface[:component].intern][interface[:layer1_interface].intern]=Array.new() unless components[interface[:component].intern].include?(interface[:layer1_interface].intern)
            components[interface[:component].intern][interface[:layer1_interface].intern] << v
          end unless lookup.nil?()
        end
        return components
      end

      def create_lookup_vlan_to_interface_slow_fast_equality(*args)
        @timer.start(:vlans_equal)
        result=Hash.new()
        result[:summary]=true
        ["voice_vlan_to_interface","access_vlan_to_interface","tagged_vlan_to_interface","vtp_vlan_to_interface"].each do |l|
          result[l.to_s]=((get_lookup_table("#{l}_slow".intern))==(get_lookup_table("#{l}_fast".intern)))
          result[:summary]&&=result[l.to_s]
        end
        @timer.stop(:vlans_equal)
        return result
      end
    end
  end
end
