# encoding: utf-8
# license: gpl2p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST
require 'ipaddress'
module Iof
  module Target
    class Handler < Ghun::Base::AutoloadingHandler
      SELECTOR=Ghun::Base::Selector.new("TargetSource",[:type],false,Ghun::Log::Source::TARGET,Ghun::Log::Type::BASE)
      Ghun::Base::Blackboard.config.declare_key(:"target.unique_mac", :bool, true)
      Ghun::Base::Blackboard.config.declare_key(:"target.unique_ipv4", :bool, true)
      Ghun::Base::Blackboard.config.declare_key(:"target.unique_ipv6", :bool, true)
      Ghun::Base::Blackboard.config.declare_key(:"target.unique_name", :bool, true)

      Ghun::Base::Blackboard.config.declare_key(:"target.unique_mac_addresses", :array,[])
      Ghun::Base::Blackboard.config.declare_key(:"target.non_unique_mac_addresses", :array, [/.*/])
      Ghun::Base::Blackboard.config.declare_key(:"target.assume_unique_for_unknown_mac_addresses", :bool, false)
      Ghun::Base::Blackboard.config.declare_key(:"target.report_unknown_mac_addresses", :bool, false)
      Ghun::Base::Blackboard.config.declare_key(:"target.unique_ipv4_addresses", :array,[])
      Ghun::Base::Blackboard.config.declare_key(:"target.non_unique_ipv4_addresses", :array, ["192.168.0.0/16","172.16.0.0/12","10.0.0.0/8","127.0.0.0/8","169.254.0.0/16","224.0.0.0/4"])
      Ghun::Base::Blackboard.config.declare_key(:"target.assume_unique_for_unknown_ipv4_addresses", :bool, true)
      Ghun::Base::Blackboard.config.declare_key(:"target.report_unknown_ipv4_addresses", :bool, true)
      Ghun::Base::Blackboard.config.declare_key(:"target.unique_ipv6_addresses", :array,[])
      Ghun::Base::Blackboard.config.declare_key(:"target.non_unique_ipv6_addresses", :array, ["fe80::/64","fc00::/8","fd00::/8","f00::/8","::/128","::1/128","::/128"])
      Ghun::Base::Blackboard.config.declare_key(:"target.assume_unique_for_unknown_ipv6_addresses", :bool, true)
      Ghun::Base::Blackboard.config.declare_key(:"target.report_unknown_ipv6_addresses", :bool, true)
      Ghun::Base::Blackboard.config.declare_key(:"target.unique_names", :array,[])
      Ghun::Base::Blackboard.config.declare_key(:"target.non_unique_names", :array, ['localhost'])
      Ghun::Base::Blackboard.config.declare_key(:"target.assume_unique_for_unknown_names", :bool, true)
      Ghun::Base::Blackboard.config.declare_key(:"target.report_unknown_names", :bool, true)
      Ghun::Base::Blackboard.config.declare_key(:"target.broken_name_mappings", :array, [[/\.$/,'']])

      Ghun::Base::Blackboard.config.declare_key(:"target.vlan_name_distribution_report", :bool, false)

      def initialize(mode=:execute_and_store,write_timestamp=nil,read_timestamp=nil)
        super(Ghun::Log::Source::TARGET)
        @unique_mac=config[:"target.unique_mac"]
        @unique_ipv4=config[:"target.unique_ipv4"]
        @unique_ipv6=config[:"target.unique_ipv6"]
        @unique_name=config[:"target.unique_name"]
        @mode=mode

        @output=Ghun::Output::Versioned.new(nil,write_timestamp,read_timestamp)

        @unique_mac_addresses=config[:"target.unique_mac_addresses"]
        @non_unique_mac_addresses=config[:"target.non_unique_mac_addresses"]
        @assume_unique_for_unknown_mac_addresses=config[:"target.assume_unique_for_unknown_mac_addresses"]
        @report_unknown_mac_addresses=config[:"target.report_unknown_mac_addresses"]
        @unique_ipv4_addresses_raw=config[:"target.unique_ipv4_addresses"]
        @non_unique_ipv4_addresses_raw=config[:"target.non_unique_ipv4_addresses"]
        @assume_unique_for_unknown_ipv4_addresses=config[:"target.assume_unique_for_unknown_ipv4_addresses"]
        @report_unknown_ipv4_addresses=config[:"target.report_unknown_ipv4_addresses"]
        @unique_ipv6_addresses_raw=config[:"target.unique_ipv6_addresses"]
        @non_unique_ipv6_addresses_raw=config[:"target.non_unique_ipv6_addresses"]
        @assume_unique_for_unknown_ipv6_addresses=config[:"target.assume_unique_for_unknown_ipv6_addresses"]
        @report_unknown_ipv6_addresses=config[:"target.report_unknown_ipv6_addresses"]
        @unique_names=config[:"target.unique_names"]
        @non_unique_names=config[:"target.non_unique_names"]
        @assume_unique_for_unknown_names=config[:"target.assume_unique_for_unknown_names"]
        @report_unknown_names=config[:"target.report_unknown_names"]
        @broken_name_mappings=config[:"target.broken_name_mappings"]

        @vlan_name_distribution_report=config[:"target.vlan_name_distribution_report"]

        @unique_ipv4_addresses=Array.new()
        @unique_ipv4_addresses_raw.each do |a|
          begin
            @unique_ipv4_addresses << ::IPAddress::IPv4.new(a).network
          rescue => e
            error "Failed to parse network #{a}", e
          end
        end
        @unique_ipv4_addresses_raw=nil
        @non_unique_ipv4_addresses=Array.new()
        @non_unique_ipv4_addresses_raw.each do |a|
          begin
            @non_unique_ipv4_addresses << ::IPAddress::IPv4.new(a).network
          rescue => e
            error "Failed to parse network #{a}", e
          end
        end
        @non_unique_ipv4_addresses_raw=nil
        @unique_ipv6_addresses=Array.new()
        @unique_ipv6_addresses_raw.each do |a|
          begin
            @unique_ipv6_addresses << ::IPAddress::IPv6.new(a).network
          rescue => e
            error "Failed to parse network #{a}", e
          end
        end
        @unique_ipv6_addresses_raw=nil
        @non_unique_ipv6_addresses=Array.new()
        @non_unique_ipv6_addresses_raw.each do |a|
          begin
            @non_unique_ipv6_addresses << ::IPAddress::IPv6.new(a).network
          rescue => e
            error "Failed to parse network #{a}", e
          end
        end
        @non_unique_ipv6_addresses_raw=nil

        @target_by_mac=Hash.new()
        @target_by_ipv4=Hash.new()
        @target_by_ipv6=Hash.new()
        @target_by_name=Hash.new()
        @target_by_internal=Hash.new()
        @auth=Ghun::Auth::Handler.new()
        @mutex=Monitor.new()

        @mac_to_ipv4=Hash.new()
        @ipv4_to_mac=Hash.new()
        @mac_to_ipv6=Hash.new()
        @ipv6_to_mac=Hash.new()

        @mac_overview=Hash.new()
        @vlan_name_distribution=Hash.new()
        @vlan_overview=Array.new()
        Range.new(1,4095).each do |i|
          @vlan_overview[i]=Hash.new()
          @vlan_overview[i][:seen]=0
          @vlan_overview[i][:name]=nil
          @vlan_overview[i][:tagged]=0
          @vlan_overview[i][:native]=0
          @vlan_overview[i][:voice]=0
          @vlan_overview[i][:vtp]=0
        end

        enable_all
        load_targets
      end

      def clear_targets
        @target_by_mac.clear()
        @target_by_ipv4.clear()
        @target_by_ipv6.clear()
        @target_by_name.clear()
        @target_by_internal.clear()
      end

      def load_targets
        @members.each do |n,member|
          member.load_targets(self,@mode,@output)
        end unless @members.nil?
      end

      def filter_target(target)
        result=false
        return result if target.filter_protected_by_cdp?()
        @members.values.each do |member|
          next unless member.enabled?()
          result||=member.filter_target_by_name(target.names)
          result||=member.filter_target_by_ipv4(target.ipv4_addresses)
          result||=member.filter_target_by_ipv6(target.ipv6_addresses)
          break if result
        end unless @members.nil?
        target.filtered=result
      end

      def add_mac_to_item(target,mac_raw)
        mac=add_mac(mac_raw)
        ipv4=get_ipv4_addresses_by_mac(mac)
        ipv4.each do |a|
          unless target.ipv4_addresses.include?(ipv4)
            debug "Found associated IPv4 address #{a} for MAC address #{mac}; adding to #{target.description()}"
            add_to_item("IPv4 addresses",a,@unique_ipv4,@unique_ipv4_addresses,@non_unique_ipv4_addresses,@assume_unique_for_unknown_ipv4_addresses,@report_unknown_ipv4_addresses,nil,@target_by_ipv4,target.ipv4_addresses,nil,target) unless a.nil?()
          end
        end unless ipv4.nil?()
        ipv6=get_ipv6_addresses_by_mac(mac)
        ipv6.each do |a|
          unless target.ipv6_addresses.include?(ipv6)
            debug "Found associated IPv6 address #{a} for MAC address #{mac}; adding to #{target.description()}"
            add_to_item("IPv6 addresses",a,@unique_ipv6,@unique_ipv6_addresses,@non_unique_ipv6_addresses,@assume_unique_for_unknown_ipv6_addresses,@report_unknown_ipv6_addresses,nil,@target_by_ipv6,target.ipv6_addresses,nil,target) unless a.nil?()
          end
        end unless ipv6.nil?()
        add_to_item("MAC addresses",mac,@unique_mac,@unique_mac_addresses,@non_unique_mac_addresses,@assume_unique_for_unknown_mac_addresses,@report_unknown_mac_addresses,nil,@target_by_mac,target.mac_addresses,nil,target) unless mac.nil?()
      rescue => e
        error "Failed to add mac address #{mac_raw} to target #{target.description()}", e
        return target
      end

      def add_ipv4_to_item(target,ipv4_raw,type)
        items_var=nil
        items_var="@#{type.to_s}_ipv4_address".intern unless type.nil?()
        ipv4=add_ipv4(ipv4_raw)
        mac=get_mac_address_by_ipv4(ipv4)
        if !mac.nil?() && !target.mac_addresses.include?(mac)
          debug "Found associated mac address #{mac} for IPv4 address #{ipv4}; adding to #{target.description()}"
          add_to_item("MAC addresses",mac,@unique_mac,@unique_mac_addresses,@non_unique_mac_addresses,@assume_unique_for_unknown_mac_addresses,@report_unknown_mac_addresses,nil,@target_by_mac,target.mac_addresses,nil,target) unless mac.nil?()
        end
        add_to_item("IPv4 addresses",ipv4,@unique_ipv4,@unique_ipv4_addresses,@non_unique_ipv4_addresses,@assume_unique_for_unknown_ipv4_addresses,@report_unknown_ipv4_addresses,nil,@target_by_ipv4,target.ipv4_addresses,items_var,target) unless ipv4.nil?()
      rescue => e
        error "Failed to add ipv4 address #{ipv4_raw} to target #{target.description()}", e
        return target
      end

      def add_ipv6_to_item(target,ipv6_raw,type)
        items_var=nil
        items_var="@#{type.to_s}_ipv6_address".intern unless type.nil?()
        ipv6=add_ipv6(ipv6_raw)
        mac=get_mac_address_by_ipv6(ipv6)
        if !mac.nil?() && !target.mac_addresses.include?(mac)
          debug "Found associated mac address #{mac} for IPv4 address #{ipv6}; adding to #{target.description()}"
          add_to_item("MAC addresses",mac,@unique_mac,@unique_mac_addresses,@non_unique_mac_addresses,@assume_unique_for_unknown_mac_addresses,@report_unknown_mac_addresses,nil,@target_by_mac,target.mac_addresses,nil,target) unless mac.nil?()
        end
        add_to_item("IPv6 addresses",ipv6,@unique_ipv6,@unique_ipv6_addresses,@non_unique_ipv6_addresses,@assume_unique_for_unknown_ipv6_addresses,@report_unknown_ipv6_addresses,nil,@target_by_ipv6,target.ipv6_addresses,items_var,target) unless ipv6.nil?()
      rescue => e
        error "Failed to add ipv6 address #{ipv6_raw} to target #{target.description()}", e
        return target
      end

      def add_name_to_item(target,name,type)
        items_var=nil
        items_var="@#{type.to_s}_name".intern unless type.nil?()
        add_to_item("names",name.clone(),@unique_name,@unique_names,@non_unique_names,@assume_unique_for_unknown_names,@report_unknown_names,@broken_name_mappings,@target_by_name,target.names,items_var,target)
      rescue => e
        error "Failed to add name #{name} to target #{target.description()}", e
        return target
      end


      def add_to_item(desc,raw_value,unique_enabled,unique,non_unique,assume_unique,report_unknown,broken,target_by,items_array,items_type_var,item)
        @mutex.synchronize {
          item.output=@output if item.output.nil?()
          ip_address_mode=broken.nil?() && (raw_value.match(/:/) || raw_value.match(/\./))
          mac_address_mode=broken.nil?() && !ip_address_mode
          value=nil
          second_item=item
          if ip_address_mode || mac_address_mode
            value=raw_value.strip
          else
            value=raw_value.strip
            broken.each do |pattern|
              value.gsub!(pattern[0],pattern[1]) if value.match(pattern[0])
            end
            unless value==raw_value
              info Ghun::Log::Type::REPORT, "Name changed from #{raw_value} to #{value} due to broken_name_mappings"
              debug "Name changed from #{raw_value} to #{value} due to broken_name_mappings"
            end
          end
          return item if value==''
          unless unique_enabled
            debug "Unique mode is not enabled for #{desc}; skipping checks for '#{value}'"
            item.instance_variable_set(items_type_var, value) unless items_type_var.nil?()
            @target_by_internal[item.__id__]=item unless @target_by_internal.include?(item.__id__)
            unless items_array.include?(value)
              items_array << value
              target_by[value.intern]=Array.new() unless target_by.include?(value.intern)
              target_by[value.intern] << item
            end
            #in non unique mode the value can not be an identifier
          else
            debug "Unique mode is enabled for #{desc}; checking '#{value}'"
            known_unique=false
            known_non_unique=false
            if ip_address_mode
              unique.each do |net|
                begin
                  known_unique||=net.include?(::IPAddress.parse(value))
                rescue => e
                  error "Failed to compare #{net} and #{value}", e
                end
              end
              non_unique.each do |net|
                begin
                  known_non_unique||=net.include?(::IPAddress.parse(value))
                rescue => e
                  error "Failed to compare #{net} and #{value}", e
                end
              end
            else
              unique.each do |pattern|
                begin
                  known_unique||=pattern.match(value)
                rescue => e
                  error "Failed to compare #{pattern.to_s} and #{value}", e
                end
              end
              non_unique.each do |pattern|
                begin
                  known_non_unique||=pattern.match(value)
                rescue => e
                  error "Failed to compare #{pattern.to_s} and #{value}", e
                end
              end
            end
            if known_unique && known_non_unique
              warn "#{value} belongs to known unique and non unique #{desc}; preferring non unique"
              known_unique=false
            elsif known_unique
              debug "#{value} is part of known unique #{desc}"
            elsif known_non_unique
              debug "#{value} is part of known non unique #{desc}"
            elsif assume_unique
              known_unique=true
              known_non_unique=false
              debug "#{value} is assumed to be part of unique #{desc}"
              info Ghun::Log::Type::REPORT, "Found unknown value #{value}", "#{value} is assumed to be part of unique #{desc}" if report_unknown
            else
              known_unique=false
              known_non_unique=true
              debug "#{value} is assumed to be part of non unique #{desc}"
              info Ghun::Log::Type::REPORT, "Found unknown value #{value}", "#{value} is assumed to be part of non unique #{desc}" if report_unknown
            end
            if known_non_unique
              #given value is not unique, so adding only to corresponding array and given variable
              debug "Given value was not recognized as unique; adding to item but not to lookup tables"
              item.instance_variable_set(items_type_var, value) unless items_type_var.nil?()
              items_array << value unless items_array.include?(value)
            elsif known_unique
              #given value is unique
              #looking up in lookup table
              if target_by.include?(value.intern)
                #target exists; checking identity
                if target_by[value.intern].__id__==item.__id__
                  debug "Found given target #{item.description()} while adding #{value}"
                  #given target is already known
                  #adding value only to corresponding array and given variable
                  item.instance_variable_set(items_type_var, value) unless items_type_var.nil?()
                  items_array << value unless items_array.include?(value)
                  item.identifier << value unless item.identifier.include?(value)
                else
                  #given target is not the known one
                  #joining given target to existing one
                  second_item=target_by[value.intern]
                  warn "Found conflicting target #{second_item.description()} while adding #{value} to #{item.description()}"
                  second_item.join(self,item)
                end
              else
                #target is new; adding value to item fields and into lookup tables
                if @target_by_internal.include?(item.__id__)
                  debug "Target #{item.description()} is already known; adding #{value} to item and to lookup table"
                  item.instance_variable_set(items_type_var, value) unless items_type_var.nil?()
                  items_array << value unless items_array.include?(value)
                  item.identifier << value unless item.identifier.include?(value)
                  target_by[value.intern]=item
                else
                  debug "Target is new; adding #{value} to item and to lookup table"
                  item.instance_variable_set(items_type_var, value) unless items_type_var.nil?()
                  items_array << value unless items_array.include?(value)
                  item.identifier << value unless item.identifier.include?(value)
                  @target_by_internal[item.__id__]=item unless @target_by_internal.include?(item.__id__)
                  target_by[value.intern]=item
                end
              end
            end
          end
          return second_item
        }
      end

      def remove_target(item)
        @mutex.synchronize {
          item.mac_addresses.each do |a|
            if @unique_mac
              @target_by_mac.delete(a.intern) if @target_by_mac.include?(a.intern)
            else
              if @target_by_mac.include?(a.intern) && !@target_by_mac[a.intern].nil?()
                result=Array.new()
                @target_by_mac[a.intern].each do |i|
                  result << i unless i._id__==item.__id__
                end
                @target_by_mac[a.intern]=result
              end
            end
          end
          item.ipv4_addresses.each do |a|
            if @unique_ipv4
              @target_by_ipv4.delete(a.intern) if @target_by_ipv4.include?(a.intern)
            else
              if @target_by_ipv4.include?(a.intern) && !@target_by_ipv4[a.intern].nil?()
                result=Array.new()
                @target_by_ipv4[a.intern].each do |i|
                  result << i unless i._id__==item.__id__
                end
                @target_by_ipv4[a.intern]=result
              end
            end
          end
          item.ipv6_addresses.each do |a|
            if @unique_ipv6
              @target_by_ipv6.delete(a.intern) if @target_by_ipv6.include?(a.intern)
            else
              if @target_by_ipv6.include?(a.intern) && !@target_by_ipv6[a.intern].nil?()
                result=Array.new()
                @target_by_ipv6[a.intern].each do |i|
                  result << i unless i._id__==item.__id__
                end
                @target_by_ipv6[a.intern]=result
              end
            end
          end
          item.names.each do |n|
            if @unique_name
              @target_by_name.delete(n.intern) if @target_by_name.include?(n.intern)
            else
              if @target_by_name.include?(n.intern) && !@target_by_name[n.intern].nil?()
                result=Array.new()
                @target_by_name[n.intern].each do |i|
                  result << i unless i._id__==item.__id__
                end
                @target_by_name[n.intern]=result
              end
            end
          end
          @target_by_internal.delete(item.__id__) if @target_by_internal.include?(item.__id__)
        }
      end

      def get_target_by_any(identifier)
        result=Array.new()
        begin
          result << get_target_by_mac(identifier)
        rescue => e
          debug "Failed to get_target_by_mac for #{identifier}", e
        end
        begin
          result << get_target_by_ipv4(identifier)
        rescue => e
          debug "Failed to get_target_by_ipv4 for #{identifier}", e
        end
        begin
          result << get_target_by_ipv6(identifier)
        rescue => e
          debug "Failed to get_target_by_ipv6 for #{identifier}", e
        end
        begin
          result << get_target_by_name(identifier)
        rescue => e
          debug "Failed to get_target_by_name for #{identifier}", e
        end
        result.flatten!
        result.compact!
        result.uniq!
        if result.empty? then
          nil
        else
          result
        end
      end

      def get_target_by_mac(mac_raw)
        mac=add_mac(mac_raw)
        @target_by_mac[mac.intern]
      end

      def get_target_by_ipv4(ipv4_raw)
        ipv4=add_ipv4(ipv4_raw)
        @target_by_ipv4[ipv4.intern]
      end

      def get_target_by_ipv6(ipv6_raw)
        ipv6=add_ipv6(ipv6_raw)
        @target_by_ipv6[ipv6.intern]
      end

      def get_target_by_name(raw_name)
        name=raw_name.clone
        @broken_name_mappings.each do |pattern|
          name.gsub!(pattern[0],pattern[1]) if name.match(pattern[0])
        end
        debug "Name changed from #{raw_name} to #{name} due to broken_name_mappings while looking up" unless name==raw_name
        return nil if raw_name==''
        @target_by_name[name.intern]
      end

      def get_ipv4_addresses_by_mac(mac_raw)
        mac=add_mac(mac_raw)
        @mac_to_ipv4[mac.intern]
      end

      def get_ipv6_addresses_by_mac(mac_raw)
        mac=add_mac(mac_raw)
        @mac_to_ipv6[mac.intern]
      end

      def get_mac_address_by_ipv4(ipv4_raw)
        ipv4=add_ipv4(ipv4_raw)
        @ipv4_to_mac[ipv4.intern]
      end

      def get_mac_address_by_ipv6(ipv6_raw)
        ipv6=add_ipv6(ipv6_raw)
        @ipv6_to_mac[ipv6.intern]
      end

      def assign_mac_ipv4(mac_raw,ipv4_raw)
        mac=add_mac(mac_raw)
        ipv4=add_ipv4(ipv4_raw)
        #FIXME arp cache needs further investigation
        debug "ARP cache seems to be faulty; skipping assignment"
        return nil
        @mac_to_ipv4[mac.intern] << ipv4 if @mac_to_ipv4.include?(mac.intern) && !@mac_to_ipv4[mac.intern].include?(ipv4)
        if @ipv4_to_mac[ipv4.intern]!=mac && !@ipv4_to_mac[ipv4.intern].nil?()
          warn "Found conflicting MAC addresses #{mac} and #{@ipv4_to_mac[ipv4.intern]} for IPv4 address #{ipv4}"
        elsif @ipv4_to_mac[ipv4.intern].nil?()
          @ipv4_to_mac[ipv4.intern]=mac
        end
      end

      def assign_mac_ipv6(mac_raw,ipv6_raw)
        mac=add_mac(mac_raw)
        ipv6=add_ipv6(ipv6_raw)
        @mac_to_ipv6[mac.intern] << ipv6 if @mac_to_ipv6.include?(mac.intern) && !@mac_to_ipv6[mac.intern].include?(ipv6)
        if @ipv6_to_mac[ipv6.intern]!=mac && !@ipv6_to_mac[ipv6.intern].nil?()
          warn "Found conflicting MAC addresses #{mac} and #{@ipv6_to_mac[ipv6.intern]} for IPv6 address #{ipv6}"
        elsif @ipv6_to_mac[ipv6.intern].nil?()
          @ipv6_to_mac[ipv6.intern]=mac
        end
      end

      def add_mac(mac_raw)
        mac=Iof::Data.mac_plain(mac_raw)
        @mac_to_ipv6[mac.intern]=Array.new() unless mac.nil?() || @mac_to_ipv6.include?(mac.intern)
        @mac_to_ipv4[mac.intern]=Array.new() unless mac.nil?() || @mac_to_ipv4.include?(mac.intern)
        @mac_overview[mac.intern]={:seen => 0, :used => 0, :error => 0, :not_implemented => 0, :switch_filter => 0, :needs_interaction => 0 } unless mac.nil?() || @mac_overview.include?(mac.intern)
        return mac
      end

      def update_mac(mac_raw,type)
        mac=add_mac(mac_raw)
        @mac_overview[mac.intern][type]+=1 if !mac.nil?() && !@mac_overview.nil?() && @mac_overview.include?(mac.intern) && !@mac_overview[mac.intern].nil?() && @mac_overview[mac.intern].include?(type)
      end

      def add_ipv4(ipv4_raw)
        ipv4=::IPAddress::IPv4.new(ipv4_raw).address.to_s
        @ipv4_to_mac[ipv4.intern]=nil unless ipv4.nil?() || @ipv4_to_mac.include?(ipv4.intern)
        return ipv4
      end

      def add_ipv6(ipv6_raw)
        ipv6=::IPAddress::IPv6.new(ipv6_raw).to_string_uncompressed.downcase
        @ipv6_to_mac[ipv6.intern]=nil unless ipv6.nil?() || @ipv6_to_mac.include?(ipv6.intern)
        return ipv6
      end

      def name_vlan(id_raw,name,seen_at)
        id=id_raw.to_s.to_i
        return if id==0
        if !@vlan_overview[id][:name].nil?() && @vlan_overview[id][:name]!=name
          debug "Conflicting names '#{name}' and #{@vlan_overview[id][:name]} for vlan #{id}"
        elsif @vlan_overview[id][:name].nil?()
          @vlan_overview[id][:name]=name
        end
        if @vlan_name_distribution_report && !seen_at.nil?()
          @vlan_name_distribution[id]=Hash.new() unless @vlan_name_distribution.include?(id)
          @vlan_name_distribution[id][name.intern]=Array.new() unless @vlan_name_distribution[id].include?(name.intern)
          @vlan_name_distribution[id][name.intern] << seen_at
        end
      end

      def generate_vlan_distribution_report(full=false)
        if @vlan_name_distribution_report
          info Ghun::Log::Type::REPORT, "Printing #{(full)?'complete ':''}vlan name distribution report"
          @vlan_name_distribution.keys.sort.each do |id|
            names=@vlan_name_distribution[id]
            if names.nil?() || names.keys.size()==0
              info Ghun::Log::Type::REPORT, "Vlan #{id} has no associated names"
            elsif names.keys.size()==1
              if full
                info Ghun::Log::Type::REPORT, "Vlan #{id} is known under 1 name"
                names.each do |name,devices|
                  info Ghun::Log::Type::REPORT, "Vlan #{id} is known as '#{name.to_s}' on #{devices.size()} devices: #{devices.sort.uniq.join(', ')}"
                end
              else
                info Ghun::Log::Type::REPORT, "Vlan #{id} is known as '#{names.keys[0].to_s}' on #{names.values[0].uniq.size()} devices"
              end

            elsif names.keys.size()>1
              info Ghun::Log::Type::REPORT, "Vlan #{id} is known under #{names.keys.size} names"
              names.keys.sort_by { |k| names[k].size() }.each do |name|
                devices=names[name]
                info Ghun::Log::Type::REPORT, "Vlan #{id} is known as '#{name.to_s}' on #{devices.size()} devices: #{devices.sort.uniq.join(', ')}"
              end
            end
          end
        else
          debug Ghun::Log::Type::REPORT, "Vlan name distribution report is disabled"
        end
      end

      def update_vlan(id,type)
        id=id.to_s.to_i
        @vlan_overview[id][type]+=1
      end

      def get_vlan(id)
        id=id.to_s.to_i
        @vlan_overview[id]
      end


      def each()
        targets=nil
        @mutex.synchronize {
          targets=@target_by_internal.dup()
        }
        targets.each do |key,value|
          next if value.filtered?()
          yield value
        end
      end

      def size()
        @mutex.synchronize {
          return @target_by_internal.keys.size()
        }
      end

      def has_not_preparsed_targets?()
        undone=false
        @mutex.synchronize {
          @target_by_internal.values.each do |value|
            next if value.filtered?()
            undone||=(!value.failed_until_preparse?() && value.undone_until_preparse?())
            break if undone
          end
        }
        undone
      end

      def has_not_parsed_targets?()
        undone=false
        @mutex.synchronize {
          @target_by_internal.values.each do |value|
            next if value.filtered?()
            undone||=(!value.failed_until_parse?() && value.undone_until_parse?())
            break if undone
          end
        }
        undone
      end

      def has_undone_targets?()
        undone=false
        @mutex.synchronize {
          @target_by_internal.values.each do |value|
            next if value.filtered?()
            undone||=(!value.failed?() && value.undone?())
            break if undone
          end
        }
        undone
      end

      def member_class(type)
        result=SELECTOR.select_member(type)
        if result.nil?()
          self.error "Unknown target source #{type.to_s}"
          raise InvalidTargetSourceError, "Unknown config storage engine #{type}"
        end
        return result
      end

      def init_member_list
        @member_list=Array.new()
        member_list=Array.new()
        SELECTOR.members.values.each do |m|
          member_list << m[:values][0][0]
        end unless SELECTOR.members.nil?()
        member_list.each do |m|
          config.declare_dynamic_key("target.source.#{m.to_s}.enabled".intern,:bool,false)
          @member_list << m.intern
        end unless member_list.nil?()
      end
    end
  end
end
