# encoding: utf-8
# license: gpl2p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST
module Iof
  module Target
    class Item
      include Ghun::Base::Logging
      def initialize()
        init_logging(Ghun::Log::Source::TARGET)
        @mac_addresses=Array.new()
        @ipv4_addresses=Array.new()
        @primary_ipv4_address=nil
        @dns_ipv4_address=nil
        @mgmt_ipv4_address=nil
        @ipv6_addresses=Array.new()
        @primary_ipv6_address=nil
        @dns_ipv6_address=nil
        @mgmt_ipv6_address=nil
        @names=Array.new()
        @primary_name=nil
        @dns_name=nil
        @own_name=nil

        @identifier=Array.new()

        @auth_name=nil
        @auth_item=nil
        @connection_type=nil
        @connection_version=nil
        @connection_class=nil
        @connection_reconnect_per_command=false
        @parser_class=nil
        @parser=nil

        @filtered=false

        @commands=nil
        @results=nil

        #all @status_* variables contain values in ternary logic
        #true: this step was successful
        #false: this step failed
        #nil: this step was not done yet or the result is unknown
        @status_identify=nil
        @status_prepare=nil
        @status_collect=nil
        @status_preparse=nil
        @status_parse=nil
        @status_postparse=nil
        @status_cleanup=nil

        @device_vendor=nil
        @device_type=nil
        @special=Hash.new()
        @add_to_handler_mutex=Monitor.new()
        @output=nil
        @special_mutex=Monitor.new()
      end

      attr_accessor :output
      attr_accessor :status_identify, :status_prepare, :status_collect, :status_preparse, :status_parse, :status_postparse, :status_cleanup
      attr_reader :connection_type, :connection_version, :connection_class
      attr_reader :auth_name, :auth_item
      attr_reader :device_vendor, :device_type, :special, :special_mutex
      attr_reader :primary_ipv4_address, :dns_ipv4_address, :mgmt_ipv4_address, :ipv4_addresses
      attr_reader :primary_ipv6_address, :dns_ipv6_address, :mgmt_ipv6_address, :ipv6_addresses
      attr_reader :mac_addresses
      attr_reader :primary_name, :dns_name, :own_name, :names
      attr_reader :commands, :results, :identifier
      attr_writer :auth_name

      def connection_reconnect_per_command?()
        @connection_reconnect_per_command
      end

      def name()
        @own_name || @primary_name || @dns_name || @names.uniq.compact[0]
      end

      def filter_protected_by_cdp()
        @special_mutex.synchronize() {
          @special[:filtered_by_runner]=false
          @filtered=false
          @special[:filter_protected_by_cdp]=true
        }
      end

      def filter_protected_by_cdp?()
        @special_mutex.synchronize() {
          @special[:filter_protected_by_cdp]
        }
      end

      def filtered?()
        @filtered
      end

      def failed?()
        @status_identify==false || @status_prepare==false || @status_collect==false || @status_preparse==false || @status_parse==false || @status_postparse==false || @status_cleanup==false
      end

      def undone?()
        @status_identify.nil?() || @status_prepare.nil?() || @status_collect.nil?() || @status_preparse.nil?() || @status_parse.nil?() || @status_postparse.nil?() || @status_cleanup.nil?()
      end

      def failed_until_parse?()
        @status_identify==false || @status_prepare==false || @status_collect==false || @status_preparse==false || @status_parse==false
      end

      def undone_until_parse?()
        @status_identify.nil?() || @status_prepare.nil?() || @status_collect.nil?() || @status_preparse.nil?() || @status_parse.nil?()
      end

      def failed_until_preparse?()
        @status_identify==false || @status_prepare==false || @status_collect==false || @status_preparse==false
      end

      def undone_until_preparse?()
        @status_identify.nil?() || @status_prepare.nil?() || @status_collect.nil?() || @status_preparse.nil?()
      end


      def filtered=(filtered)
        return if @filtered==filtered
        @filtered=filtered
        if filtered
          @status_prepare=false
          @status_collect=false
          @status_preparse=false
          @status_parse=false
          @status_postparse=false
          @status_cleanup=false
        else
          @status_prepare=nil
          @status_collect=nil
          @status_preparse=nil
          @status_parse=nil
          @status_postparse=nil
          @status_cleanup=nil
        end
      end

      def device_vendor=(device_vendor)
        @device_vendor=device_vendor
        @status_identify=!(@device_vendor.nil? || @device_vendor=='' || @device_type.nil?)
      end

      def device_type=(device_type)
        @device_type=device_type
        @status_identify=!(@device_vendor.nil? || @device_vendor=='' || @device_type.nil?)
      end


      def description(mode=:default)
        result=""
        case mode
        when :full
          if @names.empty?()
            result="unnamed"
          else
            result=@names.join(' ')
          end
          result+=' ('
          if @ipv4_addresses.empty?()
            result+='no IPv4 address'
          else
            result+=@ipv4_addresses.join(' ')
          end
          result+=', '
          if @ipv6_addresses.empty?()
            result+='no IPv6 address'
          else
            result+=@ipv6_addresses.join(' ')
          end
          result+=', '
          if @mac_addresses.empty?()
            result+='no MAC address'
          else
            result+=@mac_addresses.join(' ')
          end
          result+=')'
        when :identifier
          result+=@identifier.join(' ')
        else
          result=(@primary_name || @dns_name || @own_name || @names[0] || 'unnamed')
          result+=' ('
          result+=(@primary_ipv4_address || @dns_ipv4_address || @mgmt_ipv4_address || @ipv4_addresses[0] || 'no IPv4 address')
          result+=', '
          result+=(@primary_ipv6_address || @dns_ipv6_address || @mgmt_ipv6_address || @ipv6_addresses[0] || 'no IPv6 address')
          result+=')'
        end
        result="'#{result.strip()}'"
      end

      def identify(mode)
        #TODO implement me

        debug "Identify called for target #{description()}"
        error "Identification is not implemented yet: #{description()}"
        @status_identify=false
        return @status_identify
      rescue => e
        error "Failed to identify target #{description()}", e
        @status_identify=false
        return @status_identify
      end

      def prepare(auth_handler)
        unless @status_identify
          error "Could not prepare unidentified target #{description()}"
          @status_prepare=false
        else
          @ipv4_addresses.sort!()
          vendor=nil
          vendor=@device_vendor.intern unless @device_vendor.nil?()
          type=nil
          type=@device_type.intern unless @device_type.nil?()
          connection=CONNECTION_SELECTOR.select_member([vendor,type])
          @connection_type=connection[0] unless connection.nil?()
          @connection_type=:none if auth_handler.nil?()
          @connection_version=connection[1] unless connection.nil?()
          @auth_item=auth_handler[@auth_name.intern] unless @connection_type==:none || auth_handler.nil?() || @auth_name.nil?()
          ctype=nil
          ctype=@connection_type.intern unless @connection_type.nil?()
          cversion=nil
          cversion=Integer(@connection_version) unless @connection_version.nil?()
          @commands=COMMAND_SELECTOR.select_member([vendor,type,ctype,cversion])
          @connection_reconnect_per_command=RECONNECT_SELECTOR.select_member([vendor,type,ctype,cversion])
          @connection_class=Ghun::Remote::REMOTE_CONNECTION_SELECTOR.select_member([ctype,cversion])
          @parser_class=Iof::Parser::SELECTOR.select_member([vendor,type,ctype,cversion])
          @status_prepare=false if @connection_type.nil?()
          @status_prepare=false if @connection_class.nil?() && @connection_type!=:none
          @status_prepare=false if @parser_class.nil?()
          if @commands.nil?() && @connection_type!=:none
            error "No known commands for vendor '#{[vendor,type,ctype,cversion]}'"
            @status_prepare=false
          else
            @results=Hash.new()
            @commands.keys.each do |label|
              @results[label]=nil
            end unless @commands.nil?()
          end
          @status_prepare=true if @status_prepare.nil?()
        end
        return @status_prepare
      rescue => e
        error "Failed to prepare target #{description()}", e
        @status_prepare=false
        return @status_prepare
      end

      def collect(thread_handler,mode)
        if !@status_prepare
          error "Could not run collect for unprepared target #{description()}"
          @status_collect=false
        elsif @connection_type==:none || @commands.empty?()
          debug "Nothing to do for target #{description()}"
          @status_collect=true
        elsif @commands.nil?() && @connection_type!=:none
          error "Undefined command list for target #{description()}"
          @status_collect=false
        else
          address=@primary_ipv4_address || @mgmt_ipv4_address || @dns_ipv4_address || @ipv4_addresses[0] || @primary_ipv6_address || @mgmt_ipv6_address || @dns_ipv6_address || @ipv6_addresses[0] || @primary_name || @dns_name || @own_name || @names[0]
          debug "Using address #{address} for target #{description()}"
          remote_connection=@connection_class.new(address,@auth_item,thread_handler)
          remote_connection.keep_connection=!@connection_reconnect_per_command
          case mode
          when :execute
            result=@results.clone()
            remote_connection.connect()
            @commands.each do |label,command|
              result[label]=remote_connection.run_command(command[:command],command[:constraints][:failed_output_stdout],command[:constraints][:failed_output_stderr],(!command[:check_stdout] || command[:expect_empty_stdout]))
            end
            remote_connection.disconnect()
            @results=result
          when :read, :read_and_store
            @results=@output.send(mode,['target.item',address,"collect"],{})
          when :execute_or_read, :execute_and_store_or_read, :execute_and_store, :read_or_execute, :read_or_execute_and_store
            @results=@output.send(mode,['target.item',address,"collect"],{}) {
              result=@results.clone()
              remote_connection.connect()
              @commands.each do |label,command|
                result[label]=remote_connection.run_command(command[:command],command[:constraints][:failed_output_stdout],command[:constraints][:failed_output_stderr],(!command[:check_stdout] || command[:expect_empty_stdout]))
              end
              remote_connection.disconnect()
              result
            }
          end
          remote_connection=nil

          @connection_type=nil
          @connection_version=nil
          @auth_item=nil
          @connection_reconnect_per_command=nil
          @connection_class=nil

          @status_collect=true
          @status_collect=false if @results.nil?()
          @status_collect=false if @results.empty?()
          failed=false
          @commands.each do |label,command|
            r=nil
            r=@results[label] unless @results.nil?()
            if r.nil?()
              warn "Result for command with label '#{label}' is missing for #{description()}"
              @status_collect=false
              break
            end

            if command[:constraints][:check_stdout]
              if command[:constraints][:expect_empty_stdout] && !r[:stdout].nil?() && r[:stdout]!=''
                warn "Result for command with label '#{label}' contains unexpected non empty stdout for #{description()}"
                failed=true
              end
              if command[:constraints][:expect_nonempty_stdout] && (r[:stdout].nil?() || r[:stdout]=='')
                warn "Result for command with label '#{label}' contains unexpected empty stdout for #{description()}"
                failed=true
              end
            end
            if command[:constraints][:check_stderr]
              if command[:constraints][:expect_empty_stderr] && !r[:stderr].nil?() && r[:stderr]!=''
                warn "Result for command with label '#{label}' contains unexpected non empty stderr for #{description()}"
                failed=true
              end
              if command[:constraints][:expect_nonempty_stderr] && (r[:stderr].nil?() || r[:stderr]=='')
                warn "Result for command with label '#{label}' contains unexpected empty stderr for #{description()}"
                failed=true
              end
            end
            if command[:constraints][:check_exit_code] && command[:constraints][:expect_exit_code]!=r[:exit]
              warn "Result for command with label '#{label}' returned unexpected exit code '#{r[:exit]}' for #{description()}"
              failed=true
            end
            if command[:constraints][:check_signal] && command[:constraints][:expect_signal]!=r[:signal]
              warn "Result for command with label '#{label}' exited with unexpected signal '#{r[:signal]}' for #{description()}"
              failed=true
            end
            break if failed
          end
          if failed
            @status_collect=false
            error "Failure during data constraint check for target #{description()}"
          else
            debug "Successful passed data constraint check for target #{description()}"
          end
        end
        return @status_collect
      rescue => e
        error "Failed to collect for target #{description()}", e
        @status_collect=false
        return @status_collect
      end

      def preparse(target_handler)
        unless @status_collect
          error "Could not preparse uncollected target #{description()}"
          @status_preparse=false
        else
          @parser=@parser_class.new(self,target_handler)
          if @parser.preparse()==false then
            @status_preparse=false
          else
            @status_preparse=true
          end
          @parser_class=nil
        end
        return @status_preparse
      rescue => e
        error "Failed to preparse data for target #{description()}", e
        @status_preparse=false
        @parser=nil
        return @status_preparse
      end

      def parse(data_handler)
        unless @status_preparse
          error "Could not parse not successful preparsed target #{description()}"
          @status_parse=false
        else
          if @parser.parse(data_handler)==false then
            @status_parse=false
          else
            @status_parse=true
          end
        end
        return @status_parse
      rescue => e
        error "Failed to parse data for target#{description()}", e
        @status_parse=false
        @parser=nil
        return @status_parse
      end

      def postparse()
        unless @status_parse
          error "Could not postparse not successful parsed target #{description()}"
          @status_parse=false
        else
          if @parser.postparse()==false then
            @status_postparse=false
          else
            @status_postparse=true
          end
        end
        @parser=nil
        return @status_postparse
      rescue => e
        error "Failed to postparse data for target #{description()}", e
        @status_postparse=false
        return @status_postparse
      end

      def cleanup(force=false)
        if (@status_preparse && @status_parse && @status_postparse) || force
          @commands=nil
          @results=nil
          @status_cleanup=true
        else
          warn "Ignoring cleanup request for not successful parsed target #{description()}"
          @status_cleanup=false
        end
      end

      def add_mac(target_handler,mac)
        item=self
        @add_to_handler_mutex.synchronize {
          item=target_handler.add_mac_to_item(self,mac)
          target_handler.filter_target(item)
        }
        info "Target #{description()} was joined with target #{item.description()}" unless item.__id__==self.__id__
        return item
      end

      #allowed values for type :primary, :dns, :mgmt, nil
      def add_ipv4(target_handler,ipv4,type=nil)
        item=self
        @add_to_handler_mutex.synchronize {
          item=target_handler.add_ipv4_to_item(self,ipv4,type)
          target_handler.filter_target(item)
        }
        info "Target #{description()} was joined with target #{item.description()}" unless item.__id__==self.__id__
        return item
      end

      #allowed values for type :primary, :dns, :mgmt, nil
      def add_ipv6(target_handler,ipv6,type=nil)
        item=self
        @add_to_handler_mutex.synchronize {
          item=target_handler.add_ipv6_to_item(self,ipv6,type)
          target_handler.filter_target(item)
        }
        info "Target #{description()} was joined with target #{item.description()}" unless item.__id__==self.__id__
        return item
      end

      #allowed values for type :primary, :dns, :own, nil
      def add_name(target_handler,name,type=nil)
        item=self
        @add_to_handler_mutex.synchronize {
          item=target_handler.add_name_to_item(self,name,type)
          target_handler.filter_target(item)
        }
        info "Target #{description()} was joined with target #{item.description()}" unless item.__id__==self.__id__
        return item
      end

      def join(target_handler,other_item)
        debug "Joining #{other_item.description()} into #{self.description()}"
        target_handler.remove_target(other_item)
        other_item.mac_addresses.each do |a|
          add_mac(target_handler, a) unless a.nil?() || a==''
        end
        other_item.ipv4_addresses.each do |a|
          add_ipv4(target_handler, a) unless a.nil?() || a==''
        end
        if @primary_ipv4_address
          warn "Dropping incoming primary IPv4 address #{other_item.primary_ipv4_address}" unless @primary_ipv4_address==other_item.primary_ipv4_address
        else
          add_ipv4(target_handler,other_item.primary_ipv4_address,:primary)
        end if other_item.primary_ipv4_address
        if @dns_ipv4_address
          warn "Dropping incoming dns IPv4 address #{other_item.dns_ipv4_address}" unless @dns_ipv4_address==other_item.dns_ipv4_address
        else
          add_ipv4(target_handler,other_item.dns_ipv4_address,:dns)
        end if other_item.dns_ipv4_address
        if @mgmt_ipv4_address
          warn "Dropping incoming mgmt IPv4 address #{other_item.mgmt_ipv4_address}" unless @mgmt_ipv4_address==other_item.mgmt_ipv4_address
        else
          add_ipv4(target_handler,other_item.mgmt_ipv4_address,:mgmt)
        end if other_item.mgmt_ipv4_address
        other_item.ipv6_addresses.each do |a|
          add_ipv6(target_handler, a) unless a.nil?() || a==''
        end
        if @primary_ipv6_address
          warn "Dropping incoming primary IPv6 address #{other_item.primary_ipv6_address}" unless @primary_ipv6_address==other_item.primary_ipv6_address
        else
          add_ipv6(target_handler,other_item.primary_ipv6_address,:primary)
        end if other_item.primary_ipv6_address
        if @dns_ipv6_address
          warn "Dropping incoming dns IPv6 address #{other_item.dns_ipv6_address}" unless @dns_ipv6_address==other_item.dns_ipv6_address
        else
          add_ipv6(target_handler,other_item.dns_ipv6_address,:dns)
        end if other_item.dns_ipv6_address
        if @mgmt_ipv6_address
          warn "Dropping incoming mgmt IPv6 address #{other_item.mgmt_ipv6_address}" unless @mgmt_ipv6_address==other_item.mgmt_ipv6_address
        else
          add_ipv6(target_handler,other_item.mgmt_ipv6_address,:mgmt)
        end if other_item.mgmt_ipv6_address
        other_item.names.each do |n|
          add_name(target_handler, n) unless n.nil?() || n==''
        end
        if @primary_name
          warn "Dropping incoming primary name #{other_item.primary_name}" unless @primary_name==other_item.primary_name
        else
          add_name(target_handler,other_item.primary_name,:primary)
        end if other_item.primary_name
        if @dns_name
          warn "Dropping incoming dns name #{other_item.dns_name}" unless @dns_name==other_item.dns_name
        else
          add_name(target_handler,other_item.dns_name,:dns)
        end if other_item.dns_name
        if @own_name
          warn "Dropping incoming own name #{other_item.own_name}" unless @own_name==other_item.own_name
        else
          add_name(target_handler,other_item.own_name,:own)
        end if other_item.own_name
        if other_item.filtered?()
          warn "Target #{description()} is now filtered"
          self.filtered=(true)
        end
        if @auth_name
          warn "Dropping incoming auth name #{other_item.auth_name}" unless @auth_name==other_item.auth_name
        else
          @auth_name=other_item.auth_name
          @auth_item=other_item.auth_item
        end if other_item.auth_name
        if @device_vendor
          warn "Dropping incoming device vendor #{other_item.device_vendor}" unless @device_vendor==other_item.device_vendor
        else
          @device_vendor=other_item.device_vendor
        end if other_item.device_vendor
        if @device_type && !@device_type.to_s.match(/^cdp/)
          warn "Dropping incoming device type #{other_item.device_type} and keeping device type #{@device_type}" unless @device_type==other_item.device_type
        elsif @device_type && @device_type.to_s.match(/^cdp/)
          unless @device_type==other_item.device_type
            warn "Dropping device type #{@device_type} and using incoming device type #{other_item.device_type}"
            @device_type=other_item.device_type
          end
        else
          @device_type=other_item.device_type
        end if other_item.device_type
        tmp_status_identify=@status_identify
        @status_identify=true unless (@device_vendor.nil? || @device_vendor=='' || @device_type.nil?)
        if (tmp_status_identify!=@status_identify) && @status_prepare==false
          warn "Target #{description()} changed identification status with failed prepare status; resetting"
          @auth_item=nil
          @connection_type=nil
          @connection_version=nil
          @connection_class=nil
          @connection_reconnect_per_command=false
          @parser_class=nil
          @parser=nil
          @commands=nil
          @results=nil
          @connection_reconnect_per_command=false
          @parser=nil
          @status_prepare=nil
          @status_collect=nil
          @status_preparse=nil
          @status_parse=nil
          @status_postparse=nil
          @status_cleanup=nil
        end
        other_item.special.each do |key,value|
          if @special[key]
            warn "Dropping incoming special field #{key.to_s} with value #{value}; keeping '#{@special[key]}'" unless @special[key]==value
          else
            @special[key]=value
          end if value
        end unless other_item.special.nil?()
      end
    end
  end
end
