# encoding: utf-8
# license: gpl2p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST

module Ghun
  module Base
    class Connection  < Ghun::Base::Module
      def initialize(mode,type=:unknown,log_source=Ghun::Log::Source::BASE,log_type=Ghun::Log::Type::BASE)
        super(log_source,log_type)
        @async_mode=(mode==:async || mode==:combined)
        @sync_mode=(mode==:sync || mode==:combined)
        @combined_mode=(mode==:misc)
        if @async_mode then
          @shutdown=false
          @requests=::Queue.new()
          @state_mutex=Mutex.new()
          @request_state=Hash.new()
          @responses=::Queue.new()
          @response_mutex=Mutex.new()
          @thread=Ghun::Base::Thread.new("#{self.class}ExecutionRunner#{self.__id__}")
          @thread.start() {
            execution_runner()
          }
        end
        @execution_mutex=Monitor.new()
        @sync_commands_pending=0
        @pending_mutex=Mutex.new()
        @type=type
        @allow_parallel_execution=false
        @retry_mode=false
        @working_before=false
        @accept_nil_as_result=false
      end

      attr_writer :allow_parallel_execution, :accept_nil_as_result
      attr_reader :sync_mode, :async_mode

      def allow_parallel_execution?()
        return @allow_parallel_execution
      end

      def accept_nil_as_result?()
        return @accept_nil_as_result
      end

      def configure_retries(
          execute_retries=Ghun::Base::Blackboard.config[:"connection.#{@type}.execute_retries"],
          execute_retry_wait=Ghun::Base::Blackboard.config[:"connection.#{@type}.execute_retry_wait"],
          execute_retry_wait_first_time=Ghun::Base::Blackboard.config[:"connection.#{@type}.execute_retry_wait_first_time"]
        )
        if execute_retries==0
          #disabling retries
          @working_before=false
          @retry_mode=false
          @execute_retries=0
          @execute_retry_wait=0
          @execute_retry_wait_first_time=0
        else
          @retry_mode=true
          @execute_retries=execute_retries
          @execute_retry_wait=execute_retry_wait
          @execute_retry_wait_first_time=execute_retry_wait_first_time
        end
      end

      def disable_retries
        configure_retries(0, 0, 0)
      end

      def enable_wait_reporting(thread_handler)
        @thread_handler=thread_handler
      end

      def disable_wait_reporting()
        @thread_handler=nil
      end

      def execute_request(args)
        raise NotSupportedInAsyncModeError unless @combined_mode || @sync_mode
        response=nil
        @pending_mutex.synchronize {
          @sync_commands_pending+=1
        }
        @execution_mutex.synchronize {
          begin
            ensure_connection() if self.respond_to?(:ensure_connection,true)
            if args.is_a?(Array)
              response=Array.new()
              args.each do |argument|
                response << execute_retry_helper(argument)
              end
            else
              response=execute_retry_helper(args)
            end
          rescue => e
            error "Failed to execute '#{args.inspect}' for connection type #{self.class}", e
            raise
          ensure
            disconnect() if self.respond_to?(:disconnect,true) && !@keep_connection
          end
        }
        @pending_mutex.synchronize {
          @sync_commands_pending-=1
        }
        return response
      end

      def enqueue_request(id,args)
        raise NotSupportedInSyncModeError unless @combined_mode || @async_mode
        return nil if @shutdown
        @state_mutex.synchronize {
          raise "Id '#{id}' already in used for request in state '#{@request_state[id.intern].to_s}'" if @request_state.include?(id.intern)
          @request_state[id.intern]=:waiting
        }
        @requests.enq({:id => id, :arguments => args})
      end

      def get_response()
        raise NotSupportedInSyncModeError unless @async_mode
        @response_mutex.synchronize {
          return @responses.deq()
        }
      end

      def get_response_non_blocking()
        raise NotSupportedInSyncModeError unless @async_mode
        @response_mutex.synchronize {
          return nil if @responses.length()==0
          return @responses.deq()
        }
      end

      def get_response_by_id(id)
        raise NotSupportedInSyncModeError unless @async_mode
        result=nil
        while result.nil?()
          @response_mutex.synchronize {
            mutex=@responses.instance_variable_get(:@mutex)
            mutex.synchronize {
              queue=@responses.instance_variable_get(:@que)
              queue.each do |q|
                if q[:id]==id
                  result=q
                  break
                end unless q.nil?()
              end
              queue.delete_if(){ |q| q.nil?() || q[:id]==id}
            }
          }
        end
        if !result.nil?() && result.include?(:response)
          return result[:response]
        else
          return nil
        end
      end

      def get_response_by_id_non_blocking(id)
        raise NotSupportedInSyncModeError unless @async_mode
        result=nil
        @response_mutex.synchronize {
          mutex=@responses.instance_variable_get(:@mutex)
          mutex.synchronize {
            queue=@responses.instance_variable_get(:@que)
            queue.each do |q|
              if q[:id]==id
                result=q
                break
              end unless q.nil?()
            end
            queue.delete_if(){ |q| q.nil?() || q[:id]==id}
          }
        }
        if !result.nil?() && result.include?(:response)
          return result[:response]
        else
          return nil
        end
      end

      def request_queue_length()
        raise NotSupportedInSyncModeError unless @async_mode
        @requests.length()
      end

      def requests_before(request_id)
        raise NotSupportedInSyncModeError unless @async_mode
        return nil if request_id.nil?()
        before=0
        mutex=@requests.instance_variable_get(:@mutex)
        mutex.synchronize {
          queue=@requests.instance_variable_get(:@que)
          queue.each do |r|
            break if r[:id]==request_id
            before+=1
          end unless queue.nil?()
        }
        @state_mutex.synchronize {
          if @request_state[request_id.intern]!=:waiting
            before=-1
          end
        }
        return before
      end

      def drop_request_state(request_id)
        raise NotSupportedInSyncModeError unless @async_mode
        @state_mutex.synchronize {
          @request_state.delete(request_id.intern) unless request_id.nil?()
        }
      end

      def request_state(request_id)
        raise NotSupportedInSyncModeError unless @async_mode
        @state_mutex.synchronize {
          if !request_id.nil?() && @request_state.include?(request_id.intern)
            return @request_state[request_id.intern]
          else
            return nil
          end
        }
      end

      def abort_request(request_id)
        if !request_id.nil?()
          mutex=@requests.instance_variable_get(:@mutex)
          mutex.synchronize {
            queue=@requests.instance_variable_get(:@que)
            queue.delete_if(){ |q| q[:id]==request_id}
          }
          @state_mutex.synchronize {
            @request_state[request_id.intern]=:aborted if @request_state.include?(request_id.intern)
          }
        end
        abort_running_request(request_id)
      end

      def abort_running_request(request_id)
        #Overwrite this method for connection types, that are able to abort running requests
        if request_id.nil?()
            warn "Can not abort running synchronous"
        else
          @state_mutex.synchronize {
            if @request_state[request_id.intern]==:running
              warn "Can not abort running job with id '#{request_id}'"
            end
          }
        end
      end

      def stop_connection()
        raise NotSupportedInSyncModeError unless @async_mode
        puts "#{__FILE__}: stop_connection"
        @shutdown=true
        @requests.enq(nil)
        @response_mutex.synchronize {
          @responses.enq(nil)
        }
      end

    private

      def reported_sleep(duration)
        @thread_handler.report_thread_sleep(duration, @type) unless @thread_handler.nil?()
        sleep duration
      ensure
        @thread_handler.report_thread_wakeup(duration, @type) unless @thread_handler.nil?()
      end

      def reported_execution(&block)
        @thread_handler.report_thread_longexec_start(@type) unless @thread_handler.nil?()
        block.call
      ensure
        @thread_handler.report_thread_longexec_stop(@type) unless @thread_handler.nil?()
      end

      def execution_runner()
        return nil unless @async_mode
        until @shutdown do
          begin
            request=@requests.pop()
            if request.nil?()
              sleep(0.1)
              next
            end
            ensure_connection() if self.respond_to?(:ensure_connection,true)
            response=nil
            @state_mutex.synchronize {
              @request_state[request[:id].intern]=:running if @request_state.include?(request[:id].intern)
            }
            if request[:arguments].is_a?(Array)
              response=Array.new()
              request[:arguments].each do |req|
                unless @allow_parallel_execution
                  wait=true
                  while wait
                    @pending_mutex.synchronize {
                      wait&&=(@sync_commands_pending>0)
                    }
                    sleep 0.1
                  end
                end
                @execution_mutex.synchronize {
                  response << execute_retry_helper(req)
                }
              end
            else
              unless @allow_parallel_execution
                wait=true
                while wait
                  @pending_mutex.synchronize {
                    wait&&=(@sync_commands_pending>0)
                  }
                  sleep 0.1
                end
              end
              @execution_mutex.synchronize {
                response=execute_retry_helper(request[:arguments])
              }
            end
            disconnect() if self.respond_to?(:disconnect,true) && !@keep_connection
            @response_mutex.synchronize {
              @responses.enq({:id => request[:id], :response => response})
            }
            @state_mutex.synchronize {
              @request_state[request[:id].intern]=:finished if @request_state.include?(request[:id].intern)
            }
          rescue => e
            error "Failed to execute '#{request.inspect}' for connection type #{self.class}", e
            @state_mutex.synchronize {
              @request_state[request[:id].intern]=:failed if @request_state.include?(request[:id].intern)
            }
            @response_mutex.synchronize {
              @responses.enq({:id => request[:id], :response => e})
            }
          ensure
            disconnect() if self.respond_to?(:disconnect,true) && !@keep_connection
          end
        end
      end

      def execute_retry_helper(args)
        if @retry_mode
          count=0
          response=nil
          while @retry_mode && count<=@execute_retries && response.nil?()
            if count > 0
              disconnect() if self.respond_to?(:disconnect,true)
              if @working_before
                reported_sleep(count*@execute_retry_wait)
              else
                reported_sleep(count*@execute_retry_wait_first_time)
              end if count>0
              connect() if self.respond_to?(:connect,true)
            end
            begin
              count+=1
              response=execute_helper(args)
            rescue => e
              warn "Failed execution try", e
              response=nil
            end
          end
          if response.nil?()
            raise RetryFailedError, "Failed to execute #{args.inspect} after #{@execute_retries} retries"
          else
            @working_before=true
            return response
          end
        else
          response=execute_helper(args)
          raise "Execution returned nil, assuming unknown failure" if !@accept_nil_as_result &&  response.nil?()
          return response
        end
      end

      def execute_helper(args={})
        raise AbstractClassError, "Reimplement 'execute_helper' to make use of connection type '#{self.class}'"
      end
    end

    class StatefullConnection < Connection
      Ghun::Base::Blackboard.config.declare_key(:"connection.statefull.connect_retries", :uint, 0, true)
      Ghun::Base::Blackboard.config.declare_key(:"connection.statefull.connect_retry_wait", :uint, 0, true)
      Ghun::Base::Blackboard.config.declare_key(:"connection.statefull.connect_retry_wait_first_time", :uint, 0, true)
      Ghun::Base::Blackboard.config.declare_key(:"connection.statefull.execute_retries", :uint, 0, true)
      Ghun::Base::Blackboard.config.declare_key(:"connection.statefull.execute_retry_wait", :uint, 0, true)
      Ghun::Base::Blackboard.config.declare_key(:"connection.statefull.execute_retry_wait_first_time", :uint, 0, true)
      def initialize(mode,type=:statefull,keep_connection=true,log_source=Ghun::Log::Source::BASE,log_type=Ghun::Log::Type::BASE)
        super(mode,type,log_source,log_type)
        Ghun::Base::Blackboard.config.declare_dynamic_key(:"connection.#{@type}.connect_retries", :uint, Ghun::Base::Blackboard.config[:"connection.statefull.connect_retries"], true) unless Ghun::Base::Blackboard.config.key_declared?(:"connection.#{@type}.connect_retries")
        Ghun::Base::Blackboard.config.declare_dynamic_key(:"connection.#{@type}.connect_retry_wait", :uint, Ghun::Base::Blackboard.config[:"connection.statefull.connect_retry_wait"], true) unless Ghun::Base::Blackboard.config.key_declared?(:"connection.#{@type}.connect_retry_wait")
        Ghun::Base::Blackboard.config.declare_dynamic_key(:"connection.#{@type}.connect_retry_wait_first_time", :uint, Ghun::Base::Blackboard.config[:"connection.statefull.connect_retry_wait_first_time"], true) unless Ghun::Base::Blackboard.config.key_declared?(:"connection.#{@type}.connect_retry_wait_first_time")
        Ghun::Base::Blackboard.config.declare_dynamic_key(:"connection.#{@type}.execute_retries", :uint, Ghun::Base::Blackboard.config[:"connection.statefull.execute_retries"], true) unless Ghun::Base::Blackboard.config.key_declared?(:"connection.#{@type}.execute_retries")
        Ghun::Base::Blackboard.config.declare_dynamic_key(:"connection.#{@type}.execute_retry_wait", :uint, Ghun::Base::Blackboard.config[:"connection.statefull.execute_retry_wait"], true) unless Ghun::Base::Blackboard.config.key_declared?(:"connection.#{@type}.execute_retry_wait")
        Ghun::Base::Blackboard.config.declare_dynamic_key(:"connection.#{@type}.execute_retry_wait_first_time", :uint, Ghun::Base::Blackboard.config[:"connection.statefull.execute_retry_wait_first_time"], true) unless Ghun::Base::Blackboard.config.key_declared?(:"connection.#{@type}.execute_retry_wait_first_time")
        @keep_connection=keep_connection
        @connecting_mutex=Monitor.new()
        @connected=false
        configure_retries()
      end

      def configure_retries(
          connect_retries=Ghun::Base::Blackboard.config[:"connection.#{@type}.connect_retries"],
          connect_retry_wait=Ghun::Base::Blackboard.config[:"connection.#{@type}.connect_retry_wait"],
          connect_retry_wait_first_time=Ghun::Base::Blackboard.config[:"connection.#{@type}.connect_retry_wait_first_time"],
          execute_retries=Ghun::Base::Blackboard.config[:"connection.#{@type}.execute_retries"],
          execute_retry_wait=Ghun::Base::Blackboard.config[:"connection.#{@type}.execute_retry_wait"],
          execute_retry_wait_first_time=Ghun::Base::Blackboard.config[:"connection.#{@type}.execute_retry_wait_first_time"]
        )
        super(execute_retries,execute_retry_wait,execute_retry_wait_first_time)
        if connect_retries==0
          #disabling retries
          @connect_retries=0
          @connect_retry_wait=0
          @connect_retry_wait_first_time=0
        else
          @connect_retries=connect_retries
          @connect_retry_wait=connect_retry_wait
          @connect_retry_wait_first_time=connect_retry_wait_first_time
        end
      end

      def disable_retries()
        configure_retries(0, 0, 0, 0, 0, 0)
      end

      def keep_connection?()
        @keep_connection
      end

      def keep_connection=(keep_connection)
        @keep_connection=keep_connection
        disconnect() if keep_connection && connected?()
      end

      def connect()
        @connecting_mutex.synchronize {
          @connected=connect_retry_helper()
        }
      end

      def disconnect()
        @connecting_mutex.synchronize {
          @connected=disconnect_helper()
        }
      end

      def connected?()
        @connecting_mutex.synchronize {
          return @connected
        }
      end

    private

      def ensure_connection()
        connect() unless connected?()
        raise Ghun::Base::ConnectionError,"Failed to create connection of class #{self.class}" unless connected?()
      end

      def connect_retry_helper()
        if @retry_mode
          count=0
          response=nil
          while @retry_mode && count<=@connect_retries && response.nil?()
            if @working_before
              reported_sleep(@connect_retry_wait)
            else
              reported_sleep(@connect_retry_wait_first_time)
            end if count>0
            begin
              count+=1
              reported_execution() { response=connect_helper() }
            rescue => e
              warn "Failed connection try", e
              response=nil
            end
          end
          unless connected?()
            raise RetryFailedError, "Failed to connect after #{@connect_retries} retries"
          else
            @working_before=true
            return response
          end
        else
          return connect_helper()
        end
      end


      def connect_helper()
        raise AbstractClassError, "Reimplement 'connect_helper' to make use of connection type '#{self.class}'"
      end

      def disconnect_helper()
        raise AbstractClassError, "Reimplement 'disconnect_helper' to make use of connection type '#{self.class}'"
      end
    end

    class StatelessConnection < Connection
      Ghun::Base::Blackboard.config.declare_key(:"connection.stateless.execute_retries", :uint, 0, true)
      Ghun::Base::Blackboard.config.declare_key(:"connection.stateless.execute_retry_wait", :uint, 0, true)
      Ghun::Base::Blackboard.config.declare_key(:"connection.stateless.execute_retry_wait_first_time", :uint, 0, true)
      def initialize(mode,type=:stateless,log_source=Ghun::Log::Source::BASE,log_type=Ghun::Log::Type::BASE)
        super(mode,type,log_source,log_type)
        Ghun::Base::Blackboard.config.declare_dynamic_key(:"connection.#{@type}.execute_retries", :uint, Ghun::Base::Blackboard.config[:"connection.stateless.execute_retries"], true) unless Ghun::Base::Blackboard.config.key_declared?(:"connection.#{@type}.execute_retries")
        Ghun::Base::Blackboard.config.declare_dynamic_key(:"connection.#{@type}.execute_retry_wait", :uint, Ghun::Base::Blackboard.config[:"connection.stateless.execute_retry_wait"], true) unless Ghun::Base::Blackboard.config.key_declared?(:"connection.#{@type}.execute_retry_wait")
        Ghun::Base::Blackboard.config.declare_dynamic_key(:"connection.#{@type}.execute_retry_wait_first_time", :uint, Ghun::Base::Blackboard.config[:"connection.stateless.execute_retry_wait_first_time"], true) unless Ghun::Base::Blackboard.config.key_declared?(:"connection.#{@type}.execute_retry_wait_first_time")
        configure_retries()
      end
    end

    class ConnectionHandler < Ghun::Base::SimpleHandler
      def initialize(log_source=Ghun::Log::Source::BASE,log_type=Ghun::Log::Type::BASE)
        super(log_source,log_type)
      end
    end
  end
end
