# encoding: utf-8
# license: gpl2p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST

require 'thread'
require 'timeout'

module Ghun
  module Base
    class Thread
      include Ghun::Base::Logging
      def initialize(name,log_source=Ghun::Log::Source::BASE,log_type=Ghun::Log::Type::BASE)
        init_logging(log_source,log_type)
        @name=name
        @state=:stopped
        @state_mutex=Monitor.new()
        @thread=nil
      end

      def self.run(name,timeout=0,log_source=Ghun::Log::Source::BASE,log_type=Ghun::Log::Type::BASE,&block)
        thread=self.new(name,log_source,log_type)
        thread.start(timeout,&block)
        return thread
      end

      def run()
        raise AbstractClassError, "run needs to be reimplemented"
      end
      private :run

      def start(timeout=0,&block)
        @self=self
        status()
        @state_mutex.synchronize() {
          if @state==:running then
            warn "Thread #{@name} is running already"
          else
            @state=:"starting"
            if timeout==0 then
              debug "Starting thread #{@name} without timeout"
              @thread=::Thread.new() do
                enable_thread_local_logger()
                begin
                  if block_given?() then
                    block.call
                  else
                    self.run()
                  end
                rescue => e
                  error "Thread #{@name} terminated with error", e
                else
                  debug "Thread #{@name} terminated"
                end
                disable_thread_local_logger()
              end
              @state=:"running"
            else
              debug "Starting thread #{@name} with #{timeout} seconds timeout"
              begin
                Timeout.timeout(timeout) {
                  begin
                    enable_thread_local_logger()
                    if block_given?() then
                      block.call
                    else
                      self.run()
                    end
                  rescue => e
                    error "Thread #{@name} terminated with error", e
                  else
                    debug "Thread #{@name} terminated"
                  end
                  disable_thread_local_logger()
                }
              rescue Timeout::Error => e
                warn "Thread #{@name} timed out"
              rescue => e
                error "Thread #{@name} terminated with error", e
              else
                debug "Thread #{@name} terminated"
              end
              @state=:"running"
            end
          end
        }
      rescue => e
        error "Failed to start thread '#{@name}' of type '#{self.class}'", e
        @state=:stopped
      end

      def exit
        !@thread.nil?() && @thread.exit()
        until !@thread.status do
          sleep 0.01
        end unless @thread.nil?()
        disable_thread_local_logger()
        status()
      end
      alias :kill :exit
      alias :terminate :exit

      def join()
        @thread.nil?() && @thread.join()
        until !@thread.status do
          sleep 0.01
        end unless @thread.nil?()
        disable_thread_local_logger()
        status()
      end

      def status()
        @state_mutex.synchronize {
          if @thread.nil?() then
            @state=:stopped
          elsif !@thread.status then
            @state=:stopped
            @thread=nil
          end
        }
        !@thread.nil?() && @thread.status()
      end

      def alive?()
        !@thread.nil?() && @thread.alive?()
      end

      def stop?()
        !@thread.nil?() && @thread.stop?()
      end

      def enable_thread_local_logger()
        if ::Thread.current[:logger].nil?()
          debug "Enabling thread local logger for thread #{@name}"
          ::Thread.current[:logger]=Ghun::Base::Blackboard.logger.create_local_logger(@name)
          debug "Enabled thread local logger for thread #{@name}"
        end
      end
      private :enable_thread_local_logger

      def disable_thread_local_logger()
        unless ::Thread.current[:logger].nil?()
          debug "Disabling thread local logger for thread #{@name}"
          logger=::Thread.current[:logger]
          ::Thread.current[:logger]=nil
          logger.outputters.each do |outputter|
            outputter.close() unless outputter.closed?()
          end unless logger.nil?() || logger.outputters.nil?()
          debug "Disabled thread local logger for thread #{@name}"
        end
      end
      private :disable_thread_local_logger
    end
  end
end
