# encoding: utf-8
# license: gpl2p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST
module Ghun
  module Config
    class Handler
      include Ghun::Base::Logging

      def initialize()
        init_logging(Log::Source::CONFIG,Log::Type::BASE)
        @module_mutex=Mutex.new()
        @config_mutex=Mutex.new()
        @key_mutex=Monitor.new()
        @config=Hash.new
        @dynamic_keys=Array.new()
        @key=Hash.new()
        @modules=Array.new
        load_config
      end

      def reload_configuration
        info "Reloading configuration"
        @config_mutex.synchronize {
          @config.clear
        }
        load_config
        notify_modules
      end

      def register(mod)
        if mod.respond_to?(:"_reload_configuration") then
          @module_mutex.synchronize {
            if @modules.include?(mod) then
              warn "Instance #{mod.__id__} of class #{mod.class} alread registered"
            else
              debug "Instance #{mod.__id__} of class #{mod.class} registered"
              @modules << mod
            end
          }
        else
          error "Instance #{mod.__id__} of class #{mod.class} does not support '_reload_configuration'"
        end
      end

      def unregister(mod)
        @module_mutex.synchronize {
          @modules.delete(mod)
          debug "Instance #{mod.__id__} of class #{mod.class} unregistered"
        }
      end

      def key_declared?(name)
        @key_mutex.synchronize {
          return @key.include?(name.intern)
        }
      end

      def declare_key(name,type,default=nil,fail_on_nil=true)
        @key_mutex.synchronize {
          key=ConfigKey.new(name,type,default,fail_on_nil)
          if @key.include?(name.intern) && @key[name.intern]!=key then
           debug "Conflicting key '#{name}' already declared; discarding new key", "Known key: #{@key[name.intern]}", "New key #{key}" unless @key[name.intern].name==key.name && @key[name.intern].type==key.type && @key[name.intern].default==key.default && @key[name.intern].fail_on_nil==key.fail_on_nil
          elsif @key.include?(name.intern) && @key[name.intern]==key then
            debug "Key '#{name}' already declared"
          else
            if TypeChecker.registered?(key.type) then
              debug "Adding key '#{name}': #{key}"
              @key[name.intern]=key
            else
              warn "Key '#{name}' has unknown type '#{key.type.to_s}'; discarding key'#{key}'"
              raise UnknownConfigTypeError, "Key '#{name}' has unknown type '#{key.type.to_s}'"
            end
          end
          if @config.include?(name.intern) && !@config[name.intern].nil?() then
            set_key(name.intern,@config[name.intern])
            @config.delete(name.intern)
          end
        }
        return nil
      end

      def declare_dynamic_key(name,type,default=nil,fail_on_nil=true)
        @key_mutex.synchronize {
          @dynamic_keys << name.intern unless @dynamic_keys.include?(name.intern)
          declare_key(name,type,default,fail_on_nil)
        }
      end

      def undeclare_key(name)
        @key_mutex.synchronize {
          if @key.include?(name.intern) then
            debug "Removing key '#{name}': #{@key[name.intern]}"
          else
            warn "Key '#{name}' not known"
          end
        }
      end

      def get_key(name)
        @key_mutex.synchronize {
          if @key.include?(name.intern) then
            if @key[name.intern].valid then
              debug Log::Type::READ,"get_key called for key #{name} with result #{@key[name.intern].prepared}"
              return @key[name.intern].prepared
            elsif (!@key[name.intern].default.nil?()) || (@key[name.intern].default.nil?() && !@key[name.intern].fail_on_nil)
              debug Log::Type::READ,"get_key called for key #{name} with result #{@key[name.intern].default}"
              return @key[name.intern].default
            else
              error "Found no valid config value for key '#{name}': #{@key[name.intern]}"
              raise InvalidConfigValueError, "Found no valid config value for key '#{name}'"
            end
          else
            raise UnknownKeyError, "Key '#{name}' is unknown"
          end
        }
      end
      alias [] get_key

      def set_key(name,value,silent=true)
        @key_mutex.synchronize {
          if @key.include?(name.intern) then
            TypeChecker.check_and_set_key(@key[name.intern], value)
            notify_modules unless silent
          else
            warn "Key '#{name}' was not declared; only declared keys can be changed during runtime"
          end
        }
      end
      alias []= set_key


    private
      def notify_modules
        @module_mutex.synchronize {
          @modules.compact!
          debug("Notifying modules after configuration reload")
          @modules.each do |m|
              m.reload_config if m.respond_to?(:"_reload_configuration")
              debug("Instance #{m.__id__} of class #{m.class} notified")
          end
        }
      end

      def load_config
        info "Loading configuration"
        key=Hash.new()
        @key_mutex.synchronize {
          @key.each do |name,definition|
            key[name]=ConfigKey.new(definition.name,definition.type,definition.default,definition.fail_on_nil) unless @dynamic_keys.include?(definition.name)
          end
          @dynamic_keys.clear
        }
        @config_mutex.synchronize {
          @config.clear
        }
        ENGINE_SELECTOR.each_member() do |storage_engine|
          debug "Loading config from engine '#{storage_engine.to_s}'"
          engine=storage_engine.new()
          @config_mutex.synchronize {
            @key_mutex.synchronize {
              @config.merge!(engine.load_config(key))
            }
          }
        end
        @key_mutex.synchronize {
          @key=key
        }
      end
    end
  end
end
