# encoding: utf-8
# license: gpl2p 

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST
module Ghun
  module Config
    ConfigKey=Struct.new(:name,:type,:default,:fail_on_nil,:raw,:valid,:prepared)

    class TypeChecker
      @types=Hash.new()

      class << self
        def register_type(type,is,to)
          return if is.nil?() || to.nil?()
          @types[type.intern]={:is => is, :to => to} unless registered?(type)
        end

        def registered?(type)
          @types.include?(type.intern)
        end

        def unregister_type(type)
          @types.delete(type.intern) if registered?(type)
        end

        def check_and_set_key(key,value)
          key.raw=value
          key.prepared=nil
          begin
            if key.raw.nil?() then
              key.valid=!key.fail_on_nil
            else
              key.valid=is_type?(key.type,key.raw)
              key.prepared=to_type(key.type,key.raw) if key.valid
            end
          rescue
            key.valid=false
            key.prepared=nil
            raise
          end
          return key.valid
        end


        def is_string?(value)
          value.is_a?(String)
        rescue
          false
        end

        def is_array?(value)
          value.is_a?(Array)
        rescue
          false
        end

        def is_int_range?(value,min=nil,max=nil)
          result=false
          if value.is_a?(Integer) || (Integer(value).to_s == value.to_s) then
            int_value=Integer(value)
            result = true
            result = result && (int_value >= min) unless min.nil?
            result = result && (int_value <= max) unless max.nil?
          end
          result
        rescue
          false
        end
        alias_method :is_int?, :is_int_range?

        def is_uint_range?(value,min=nil,max=nil)
          if min.nil?() then
            is_int_range?(value,0,max)
          else
            is_int_range?(value,[0,min].max,max)
          end
        end
        alias_method :is_uint?, :is_uint_range?

        def is_float_range?(value,min=nil,max=nil)
          result=false
          if value.is_a?(Float) || (Float(value).to_s == value.to_s) then
            flt_value=Float(value)
            result = true
            result = result && (flt_value >= min) unless min.nil?
            result = result && (flt_value <= max) unless max.nil?
          end
          result
        rescue
          false
        end
        alias_method :is_float?, :is_float_range?

        def is_bool?(value)
          val=to_bool(value)
          val.is_a?(FalseClass) || val.is_a?(TrueClass)
        end

        def to_array(value)
          if value.is_a?(Array) then
            value
          else
            nil
          end
        end

        def to_string(value)
          if value.is_a? String then
            value
          else
            nil
          end
        end

        def to_int_range(value,min=nil,max=nil,clip=false)
          int_value=nil
          if value.is_a?(Integer) || (Integer(value).to_s == value.to_s) then
            int_value=Integer(value)
            if clip then
              int_value=min
            else
              int_value=nil
            end if (!int_value.nil?) && (!min.nil?) && (int_value < min)
            if clip then
              int_value=max
            else
              int_value=nil
            end if (!int_value.nil?) && (!max.nil?) && (int_value > max)
          end
          int_value
        rescue
          nil
        end
        alias_method :to_int, :to_int_range

        def to_uint_range(value,min=nil,max=nil,clip=false)
          if min.nil?() then
            to_int_range(value,0,max,clip)
          else
            to_int_range(value,[0,min].max,max,clip)
          end
        end
        alias_method :to_uint, :to_uint_range

        def to_float_range(value,min=nil,max=nil,clip=false)
          flt_value=nil
          if value.is_a?(Float) || (Float(value).to_s == value.to_s) then
            flt_value=Float(value)
            if clip then
              flt_value=min
            else
              flt_value=nil
            end if (!flt_value.nil?) && (!min.nil?) && (flt_value < min)
            if clip then
              flt_value=max
            else
              flt_value=nil
            end if (!flt_value.nil?) && (!max.nil?) && (flt_value > max)
          end
          flt_value
        rescue
          nil
        end
        alias_method :to_float, :to_float_range

        def to_bool(value)
          if value.is_a?(FalseClass) || value.nil?() then
            false
          elsif value.is_a? TrueClass then
            true
          elsif value.is_a?(String) then
            case value.upcase
            when "TRUE", "YES", "1"
              true
            when "FALSE", "NO", "0"
              false
            else
              nil
            end
          elsif value.is_a? Numeric
            if value == 1 then
              true
            elsif value == 0 then
              false
            else
              nil
            end
          else
            nil
          end
        rescue
          nil
        end

      private

        def is_type?(type,value)
          if @types.include?(type)
            @types[type][:is].call(value)
          else
            raise InvalidTypeError, "Invalid config item: #{type} is not a valid type"
          end
        end

        def to_type(type,value)
          if @types.include?(type)
            @types[type][:to].call(value)
          else
            raise InvalidTypeError, "Invalid config item: #{type} is not a valid type"
          end
        end
      end

      Ghun::Config::TypeChecker.register_type(:"string",method(:"is_string?"),method(:"to_string"))
      Ghun::Config::TypeChecker.register_type(:"int",method(:"is_int?"),method(:"to_int"))
      Ghun::Config::TypeChecker.register_type(:"uint",method(:"is_uint?"),method(:"to_uint"))
      Ghun::Config::TypeChecker.register_type(:"float",method(:"is_float?"),method(:"to_float"))
      Ghun::Config::TypeChecker.register_type(:"bool",method(:"is_bool?"),method(:"to_bool"))
      Ghun::Config::TypeChecker.register_type(:"array",method(:"is_array?"),method(:"to_array"))
    end
  end
end
