# encoding: utf-8
# license: gpl2p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST

require 'log4r'

module Ghun
  module Log
    class GlobalLog4rFileFormatter < ::Log4r::Formatter
      def format(event)
        now="#{Time.now.strftime('%Y-%m-%d %H:%M:%S')}"
        output=""
        output += "[#{now}] "
        output += "[#{Level.output event.level}] "
        output += event.data
        output += "\n"
        return output
      end
    end

    class IOFormatter < ::Log4r::Formatter
      def prepare_messages(event)
        messages=event[:messages]
        prepared=Array.new()
        if messages.is_a?(Enumerable)
          messages.each do |message|
            if message.is_a?(Exception) || (Ghun::Base::Blackboard.ruby_engine==:jruby && ( message.is_a?(Java::JavaLang::Exception) || message.is_a?(Java::JavaLang::Error)))
              prepared << ["Exception: #{message.class}",message.message,message.backtrace[0..24]]
            elsif message.is_a?(String)
              prepared << message
            elsif message.is_a?(Enumerable)
              prepared << prepare_messages(messages.to_a)
            elsif message.respond_to? :to_s then
              prepared << "Class: #{message.class}; Content: #{message.to_s}"
            else
              prepared << "Class: #{message.class}; content not accessible"
            end
          end
        else
          if messages.is_a?(Exception) || (Ghun::Base::Blackboard.ruby_engine==:jruby && ( messages.is_a?(Java::JavaLang::Exception) || messages.is_a?(Java::JavaLang::Error)))
            prepared << ["Exception: #{messages.class}",messages.message,messages.backtrace[0..24]]
          elsif messages.is_a?(String)
            prepared << messages
          elsif messages.respond_to? :to_s then
            prepared << "Class: #{messages.class}; Content: #{messages.to_s}"
          else
            prepared << "Class: #{messages.class}; content not accessible"
          end
        end
        now=Time.now.strftime('%Y-%m-%d %H:%M:%S')
        return now,prepared.flatten
      end
    end

    class LocalFileFormatter < IOFormatter
      def format(event)
        now,prepared=prepare_messages(event.data)
        output=""
        prepared.each do |m|
          output += "[#{now}] "
          output += "[#{Level.output event.data.log_level}] "
          output += "[#{Source.output event.data.log_source}] "
          output += "[#{Type.output event.data.log_type}] "
          output += m
          output += "\n"
        end
        return output
      end
    end

    class GlobalFileFormatter < IOFormatter
      def format(event)
        now,prepared=prepare_messages(event.data)
        output=""
        prepared.each do |m|
          output += "[#{now}] "
          output += "[#{Level.output event.data.log_level}] "
          output += "[#{Source.output event.data.log_source}] "
          output += "[#{Type.output event.data.log_type}] "
          output += m
          output += "\n"
        end
        return output
      end
    end

    class ConsoleFormatter < IOFormatter
      def format(event)
        now,prepared=prepare_messages(event.data)
        first=true
        output=""
        prepared.each do |m|
          output += "\n" unless first
          first=false
          output += "[#{now}] "
          output += "[#{Level.output event.data.log_level}] "
          output += "[#{Source.output event.data.log_source}] "
          output += "[#{Type.output event.data.log_type}] "
          output += m
          output += "\n"
        end
        return output
      end
    end

    ::Log4r.define_levels("DEBUG","INFO","WARN","ERROR","FATAL")
    early_logger=::Log4r::Logger.new('early')
    early_logger.level=::Log4r::INFO
    early_outputter=::Log4r::StderrOutputter.new('early_stderr')
    early_formatter=ConsoleFormatter.new()
    early_outputter.formatter=early_formatter
    early_logger.outputters=early_outputter
    Ghun::Base::Blackboard.early_logger=early_logger

    Message=Struct.new(:log_level,:log_source,:log_type,:messages)

    module Level
      LEVELS=::Log4r::LNAMES
      ALL=::Log4r::ALL
      DEBUG=::Log4r::DEBUG
      INFO=::Log4r::INFO
      WARN=::Log4r::WARN
      ERROR=::Log4r::ERROR
      FATAL=::Log4r::FATAL
      OFF=::Log4r::OFF

      ::Log4r.define_levels("DEBUG","INFO","WARN","ERROR","FATAL")
      ::Log4r.define_levels("DEBUG","INFO","WARN","ERROR","FATAL")


      def self.output(level)
        LEVELS[level]
      end
    end

    module Source
      NONE=0
      ALL=0
      @@sources=Hash.new()
      @@counter=1
      @@none=NONE
      @@all=ALL

      def self.register_log_source(name)
        n=name.upcase.intern
        return if @@sources.include?(n)
        @@sources[n]=n.to_s
        self.const_set(n,@@counter)
        @@all+=@@counter
        self.send(:remove_const,:ALL)
        self.const_set(:ALL,@@all)
        @@counter*=2
      end

      def self.unregister_log_source(name)
        n=name.upcase.intern
        return unless @@sources.include?(n)
        @@sources.delete(n)
        @@all-=self.const_get(n)
        self.send(:remove_const,:ALL)
        self.const_set(:ALL,@@all)
        self.send(:remove_const,n)
      end

      self.register_log_source("BASE")
      self.register_log_source("CONFIG")
      self.register_log_source("FILE")
      self.register_log_source("LOG")

      def self.output(source)
        output=""
        self.constants.each do |c|
          next if c==:ALL
          next if c==:NONE
          output+="#{c.to_s} " if (source&(self.const_get(c)))!=0
        end unless self.constants.nil?()
        output.strip!
        return output
      end
    end

    module Type
      NONE=0
      ALL=0
      @@types=Hash.new()
      @@counter=1
      @@none=NONE
      @@all=ALL

      def self.register_log_type(name)
        n=name.upcase.intern
        return if @@types.include?(n)
        @@types[n]=n.to_s
        self.const_set(n,@@counter)
        @@all+=@@counter
        self.send(:remove_const,:ALL)
        self.const_set(:ALL,@@all)
        @@counter*=2
      end

      def self.unregister_log_type(name)
        n=name.upcase.intern
        return unless @@types.include?(n)
        @@types.delete(n)
        @@all-=self.const_get(n)
        self.send(:remove_const,:ALL)
        self.const_set(:ALL,@@all)
        self.send(:remove_const,n)
      end

      def self.output(type)
        output=""
        self.constants.each do |c|
          next if c==:ALL
          next if c==:NONE
          output+="#{c.to_s} " if (type&(self.const_get(c)))!=0
        end unless self.constants.nil?()
        output.strip!
        return output
      end

      self.register_log_type("BASE")
      self.register_log_type("READ")
      self.register_log_type("WRITE")
      self.register_log_type("REPORT")
      self.register_log_type("EXTERNAL_LIBS")
    end
  end
end
