# encoding: utf-8
# license: gpl2p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST

module Ghun
  module Log
    class Handler
      include Ghun::Base::Configuration
      Ghun::Base::Blackboard.config.declare_key(:"log.developer_mode",:bool,false)
      Ghun::Base::Blackboard.config.declare_key(:"log.threaded",:bool,false)
      Ghun::Base::Blackboard.config.declare_key(:"log.enabled",:bool,true)
      Ghun::Base::Blackboard.config.declare_key(:"log.base_dir".intern,:string,File.join(Ghun::Base::Blackboard.log_path,Ghun::Base::Blackboard.component.internal_name))
      Ghun::Base::Blackboard.config.declare_key(:"log.loglevel".intern,:log_level,Log::Level::INFO)
      Ghun::Base::Blackboard.config.declare_key(:"log.tll.enabled",:bool,true)
      Ghun::Base::Blackboard.config.declare_key(:"log.tll.base_dir".intern,:string,File.join(Ghun::Base::Blackboard.log_path,Ghun::Base::Blackboard.component.internal_name,"tll"))
      Ghun::Base::Blackboard.config.declare_key(:"log.tll.loglevel".intern,:log_level,Log::Level::DEBUG)

      Ghun::Base::Blackboard.config.declare_key(:"log.storage.stderr.enabled",:bool,true)
      Ghun::Base::Blackboard.config.declare_key(:"log.storage.stderr.loglevel",:log_level,Log::Level::INFO)
      Ghun::Base::Blackboard.config.declare_key(:"log.storage.stdout.enabled",:bool,true)
      Ghun::Base::Blackboard.config.declare_key(:"log.storage.stdout.loglevel",:log_level,Log::Level::ERROR)
      Ghun::Base::Blackboard.config.declare_key(:"log.storage.file.enabled",:bool,true)
      Ghun::Base::Blackboard.config.declare_key(:"log.storage.file.loglevel",:log_level,Log::Level::ALL)


      Ghun::Base::Blackboard.config.declare_key(:"log.queue.log4r.enabled",:bool,true)
      Ghun::Base::Blackboard.config.declare_key(:"log.queue.log4r.only_developer",:bool,true)
      Ghun::Base::Blackboard.config.declare_key(:"log.queue.log4r.loglevel",:log_level,Log::Level::DEBUG)
      Ghun::Base::Blackboard.config.declare_key(:"log.queue.log4r.storage",:array,["file"])
      Ghun::Base::Blackboard.config.declare_key(:"log.queue.log4r.logsource",:log_source,Log::Source::NONE)
      Ghun::Base::Blackboard.config.declare_key(:"log.queue.log4r.logtype",:log_type,Log::Type::NONE)

      class Queues < Ghun::Base::AutoloadingHandler
        Ghun::Base::Blackboard.config.declare_key(:"log.queue",:array,[:"default",:"development"])
        def initialize()
          super(Log::Source::LOG)
          reload_config
          @shutdown=false
        end

        def member_class(type)
          Log::Queue
        end

        def init_member_list
          @member_list=Array.new()
          member_list=config[:"log.queue"]
          member_list << "log4r" unless member_list.include?("log4r")
          member_list.each do |m|
            config.declare_dynamic_key("log.queue.#{m.to_s}.enabled".intern,:bool,true)
            if m==:development
              config.declare_dynamic_key("log.queue.#{m.to_s}.only_developer".intern,:bool,true)
              config.declare_dynamic_key("log.queue.#{m.to_s}.loglevel".intern,:log_level,Log::Level::DEBUG)
              config.declare_dynamic_key("log.queue.#{m.to_s}.storage".intern,:array,["file"])
            else
              config.declare_dynamic_key("log.queue.#{m.to_s}.only_developer".intern,:bool,false)
              config.declare_dynamic_key("log.queue.#{m.to_s}.loglevel".intern,:log_level,Log::Level::WARN)
              config.declare_dynamic_key("log.queue.#{m.to_s}.storage".intern,:array,["stderr","file"])
            end
            config.declare_dynamic_key("log.queue.#{m.to_s}.logsource".intern,:log_source,Log::Source::ALL)
            config.declare_dynamic_key("log.queue.#{m.to_s}.logtype".intern,:log_type,Log::Type::ALL)
            @member_list << m.intern
          end unless member_list.nil?()
        end
      end

      def initialize
        register
        @global_logger_list_mutex=Mutex.new()
        @global_logger_list=Hash.new()

        @develop=config[:"log.developer_mode"]
        @threaded=config[:"log.threaded"]

        @enabled=config[:"log.enabled"]
        @base_dir=config[:"log.base_dir"]
        @log_level=config[:"log.loglevel"]
        FileUtils.mkdir_p(@base_dir,{:mode => 0750}) unless Dir.exists?(@base_dir)

        @tll_enabled=config[:"log.tll.enabled"]
        @tll_log_level=config[:"log.tll.loglevel"]
        @tll_base_dir=config[:"log.tll.base_dir"]
        FileUtils.mkdir_p(@tll_base_dir,{:mode => 0750}) unless Dir.exists?(@tll_base_dir)

        @storages=Hash.new
        ['stderr','stdout','file'].each do |storage|
          @storages[storage.intern]=Hash.new()
          @storages[storage.intern][:enabled]=config[:"log.storage.#{storage}.enabled"]
          @storages[storage.intern][:log_level]=config[:"log.storage.#{storage}.loglevel"]
        end

        @queues_mutex=Monitor.new()
        @messages=::Queue.new()
        @queues=Queues.new()
        @queues.enable_all
        @queues.members.values.each do |q|
          q.retrieve_global_logger(self)
        end
      end
      attr_reader :queues

      def self.sanitize_queue_name(name)
        name.to_s.tr("^[A-Za-z0-9]",'_').gsub(/__/,'_').gsub(/_+$/,'').gsub(/^_+/,'')
      end

      def sanitize_queue_name(name)
        Ghun::Log::Handler.sanitize_queue_name(name)
      end

      def create_local_logger(name)
        return unless @tll_enabled
        now=Time.now.strftime('%Y%m%d_%H%M%S')
        sanitized_name=sanitize_queue_name("#{name}")
        full_name="#{sanitized_name}_#{now}"
        logger=::Log4r::Logger.new(sanitized_name)
        logger.level=@tll_log_level
        outputter=::Log4r::FileOutputter.new("local_#{sanitized_name}_file",:filename => File.join(@tll_base_dir,full_name), :trunc => false)
        outputter.formatter=LocalFileFormatter.new()
        outputter.level=config[:"log.storage.file.loglevel"]
        logger.add(outputter)
        return logger
      end

      def create_global_logger(name,storages)
        return unless @enabled
        @global_logger_list_mutex.synchronize {
          sanitized_name=sanitize_queue_name(name)
          unless @global_logger_list.include?(sanitized_name.intern)
            full_name=sanitized_name
            logger=::Log4r::Logger.new(sanitized_name)
            logger.level=[config[:"log.queue.#{name.to_s}.loglevel"],@log_level].max
            storages.each do |storage|
              outputter=nil
              case storage
              when :stderr
                if @storages[storage][:enabled]
                  outputter=::Log4r::StderrOutputter.new("global_#{name}_stderr")
                  outputter.formatter=ConsoleFormatter.new()
                  outputter.level=@storages[storage][:log_level]
                end
              when :stdout
                if @storages[storage][:enabled]
                  outputter=::Log4r::StdoutOutputter.new("global_#{name}_stdout")
                  outputter.formatter=ConsoleFormatter.new()
                  outputter.level=@storages[storage][:log_level]
                end
              else
                if @storages[:file][:enabled]
                  outputter=::Log4r::FileOutputter.new("global_#{sanitized_name}_file",:filename => File.join(@base_dir,full_name), :trunc => false)
                  outputter.level=@storages[storage][:log_level]
                  if sanitized_name=='log4r'
                    outputter.formatter=GlobalLog4rFileFormatter.new()
                  else
                    outputter.formatter=GlobalFileFormatter.new()
                  end
                end
              end
              logger.add(outputter) unless outputter.nil?()
            end
            @global_logger_list[sanitized_name.intern]=logger
          end
          return @global_logger_list[sanitized_name.intern]
        }
      end

      def is_threaded?()
        return @threaded
      end

      def log(message)
        return unless @enabled
        if @threaded
          @messages.enq(message)
        else
          @queues_mutex.synchronize {
            @queues.members.each do |name,queue|
              queue.log(message) if !queue.nil?() && queue.appropriate_queue?(message)
            end unless @queues.members.nil?()
          }
        end
      end
      alias :debug :log
      alias :info :log
      alias :warn :log
      alias :error :log
      alias :fatal :log

      def log_all_messages()
        return nil unless @threaded
        begin
          @shutdown=false
          until @shutdown do
            message=@messages.pop()
            @queues.members.each do |name,queue|
              queue.log(message) if !queue.nil?() && queue.appropriate_queue?(message)
            end unless @queues.members.nil?()
          end
        rescue => e
          log(Message.new(Log::Level::ERROR,Log::Source::LOG,Log::Type::BASE,["Error while logging messages",e]))
        end
      end

      def stop_log_all_messages
        return nil unless @threaded
        @shutdown=true
        log(Message.new(Log::Level::DEBUG,Log::Source::LOG,Log::Type::BASE,"Stopped message logging"))
      end

      def log_all_messages_non_blocking()
        return nil unless @threaded
        begin
          @shutdown=false
          until @shutdown || @messages.length()==0 do
            message=@messages.pop()
            @queues.members.each do |name,queue|
              queue.log(message) if !queue.nil?() && queue.appropriate_queue?(message)
            end unless @queues.members.nil?()
          end
        rescue => e
          log(Message.new(Log::Level::ERROR,Log::Source::LOG,Log::Type::BASE,["Error while logging messages",e]))
        end
      end
    end
  end
end

