# encoding: utf-8
# license: gpl2p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST
require 'fileutils'
require 'csv'
require 'yaml'
require 'base64'
require 'zlib'

module Ghun
  module Output
    class Base < Ghun::Base::Module
      @@dir_mutex=Monitor.new()
      Ghun::Base::Blackboard.config.declare_key(:"output.dump_dir", :string, File.join(Ghun::Base::Blackboard.local.data_path,'dump'))
      Ghun::Base::Blackboard.config.declare_key(:"output.compress_encoded_output", :bool, true)
      Ghun::Base::Blackboard.config.declare_key(:"output.compress_csv_output", :bool, false)
      Ghun::Base::Blackboard.config.declare_key(:"output.compress_readable_output", :bool, true)
      Ghun::Base::Blackboard.config.declare_key(:"output.avoid_empty_files", :bool, true)
      Ghun::Base::Blackboard.config.declare_key(:"output.disable_compression_threshold", :uint, 512)
      def initialize(base=nil)
        super(Ghun::Log::Source::OUTPUT)
        @base=base
        @base=Ghun::Base::Blackboard.config[:"output.dump_dir"] unless @base
        @compress_encoded_output=Ghun::Base::Blackboard.config[:"output.compress_encoded_output"]
        @compress_csv_output=Ghun::Base::Blackboard.config[:"output.compress_csv_output"]
        @compress_readable_output=Ghun::Base::Blackboard.config[:"output.compress_readable_output"]
        @avoid_empty_files=Ghun::Base::Blackboard.config[:"output.avoid_empty_files"]
        @disable_compression_threshold=Ghun::Base::Blackboard.config[:"output.disable_compression_threshold"]
      end

      attr_reader :avoid_empty_files

      def store(data,file,options={})
        opts=parse_options(options)
        additional=nil
        if opts.include?(:dirname) && opts.include?(:filename)
          dirname=opts[:dirname]
          filename=opts[:filename]
        else
          dirname,filename,additional=build_filename_write(file.clone())
        end
        encoded=nil
        case data.class.to_s.downcase.intern
        when :string
          encoded=encode_string(data)
        when :array
          return data if options[:no_encoded_output]
          store_hash_array_as_csv(data,opts,dirname,filename) if opts[:hash_array_as_csv]
          store_array_array_as_csv(data,opts,dirname,filename) if opts[:array_array_as_csv]
          store_readable_array(data,opts,dirname,filename) if opts[:readable_array]
          return data if options[:no_encoded_output]
          encoded=encode_array(data)
          filename+=":array"
        when :hash
          store_hash_as_directory(data,opts,dirname,filename) if opts[:hash_as_directory]
          return data if options[:no_encoded_output]
          encoded=encode_anything(data)
          filename+=":anything"
        else
          return data if options[:no_encoded_output]
          filename+=":anything"
          encoded=encode_anything(data)
        end
        # if compress option is included in options, store was called by a store_*_as_* method
        # if compress option is not included, this is the 'outer' store call and store will write encoded data for
        #   internal use and not formatted data
        if ((!options.include?(:compress) && @compress_encoded_output) || (options.include?(:compress) && options[:compress])) && encoded.size() > @disable_compression_threshold
          filename+=".gz"
          mkdir(file,dirname,additional)
          File.open(File.join(dirname,filename),'wb') do |f|
            gz = Zlib::GzipWriter.new(f)
            gz.write(encoded)
            gz.close()
          end
        else
          mkdir(file,dirname,additional)
          File.open(File.join(dirname,filename),'wb') { |f| f.write(encoded)}
        end
        return data
      end

      def read(file,options={})
        opts=parse_options(options)
        dirname,filename=build_filename_read(file)
        if File.exists?(File.join(dirname,"#{filename}:array"))
          data=File.read(File.join(dirname,"#{filename}:array"))
          result=decode_array(data) if data
        elsif File.exists?(File.join(dirname,"#{filename}:array.gz"))
          data=File.open(File.join(dirname,"#{filename}:array.gz")) do |f|
            gz=Zlib::GzipReader.new(f)
            gz.read()
          end
          result=decode_array(data) if data
        elsif File.exists?(File.join(dirname,"#{filename}:anything"))
          data=File.read(File.join(dirname,"#{filename}:anything"))
          result=decode_anything(data) if data
        elsif File.exists?(File.join(dirname,"#{filename}:anything.gz"))
          data=File.open(File.join(dirname,"#{filename}:anything.gz")) do |f|
            gz=Zlib::GzipReader.new(f)
            gz.read()
          end
          result=decode_anything(data) if data
        elsif File.exists?(File.join(dirname,"#{filename}.gz"))
          data=File.open(File.join(dirname,"#{filename}.gz")) do |f|
            gz=Zlib::GzipReader.new(f)
            gz.read()
          end
          result=decode_string(data) if data
        elsif File.exists?(File.join(dirname,filename))
          data=File.read(File.join(dirname,filename))
          result=decode_string(data) if data
        else
          raise NoOutputFileToReadError, "No file in directory #{dirname} with basename #{filename} found"
        end
        raise FileReadError, "Could not read content from file in directory #{dirname} with basename #{filename}" if data.nil?()
        return result
      end

      def execute_and_store(file,options={},&block)
        opts=parse_options(options)
        data=block.call()
        return store(data, file, opts)
      end

      def read_or_execute(file,options={},&block)
        opts=parse_options(options)
        begin
          data=read(file,opts)
          return data
        rescue NoOutputFileToReadError => e
          debug "Could not read from file", e
        rescue => e
          error "Unexpected error while reading from file", e
        end
        data=block.call()
        return data
      end

      def read_and_store(file,options={})
        opts=parse_options(options)
        data=read(file,opts)
        begin
          store(data,file, options)
        rescue NoOutputFileToReadError => e
          debug "Could not read from file", e
        rescue => e
          error "Unexpected error while writing to file", e
        end
        return data
      end

      def read_or_execute_and_store(file,options={},&block)
        opts=parse_options(options)
        begin
          data=read(file,opts)
          return data
        rescue NoOutputFileToReadError => e
          debug "Could not read from file", e
        rescue => e
          error "Unexpected error while reading from file", e
        end
        execute_and_store(file, options,block)
      end

      def execute_or_read(file,options={},&block)
        opts=parse_options(options)
        begin
          data=block.call()
          return data
        rescue => e
          error "Failed to execute block; trying to read #{file.inspect()}", e
        end
        data=read(file,opts)
        return data
      end

      def execute_and_store_or_read(file,options={},&block)
        opts=parse_options(options)
        begin
          data=execute_and_store(file, options, block)
          return data
        rescue => e
          error "Failed to execute block; trying to read #{file.inspect()}", e
        end
        data=read(file,opts)
        return data
      end

    private

      def parse_options(opts)
        options=Hash.new()
        [:readable_array,:hash_as_directory,:hash_array_as_csv,:array_array_as_csv,:hash_headline,:array_headline,:dirname,:filename,:compress,:no_encoded_output].each do |key|
          if opts.include?(key) && (opts[key].is_a?(TrueClass) || opts[key].is_a?(FalseClass))
            options[key]=opts[key]
          elsif (options[:hash_array_as_csv] && key==:hash_headline && opts.include?(key) && opts[key].is_a?(Hash)) ||
              (options[:array_array_as_csv] && key==:array_headline && opts.include?(key) && opts[key].is_a?(Array))
            options[key]=opts[key].clone()
          elsif (key==:dirname || key==:filename) && (!opts.include?(:dirname) || !opts.include?(:filename))
            next
          elsif (key==:dirname || key==:filename) && opts.include?(:dirname) && opts.include?(:filename)
            options[key]=opts[key].clone()
          elsif key==:no_encoded_output && !opts.include?(:no_encoded_output)
            next
          elsif key==:no_encoded_output && opts.include?(:no_encoded_output)
            options[key]=opts[key].clone()
          elsif key==:compress && !opts.include?(:compress)
            next
          elsif key==:compress && opts.include?(:compress)
            options[key]=opts[key].clone()
          else
            options[key]=false
          end
        end
        return options
      end

      def mkdir(file,dirname,additional)
        @@dir_mutex.synchronize {
          FileUtils.mkdir_p(dirname) unless Dir.exists?(dirname)
        }
      end

      def build_filename_read(args)
        dir=File.join(args[0..-2])
        dir=File.join(@base,dir) unless @base.nil?() || @base.empty?()
        file=args[-1]
        return dir,file
      end

      def build_filename_write(args)
        dir=File.join(args[0..-2])
        dir=File.join(@base,dir) unless @base.nil?() || @base.empty?()
        file=args[-1]
        return dir,file,nil
      end

      def store_hash_as_directory(data,opts,dirname,filename)
        data.each do |key,value|
          dir=File.join(dirname,filename)
          file=key.to_s
          options=opts.clone()
          options[:dirname]=dir
          options[:filename]=file
          options[:compress]=@compress_readable_output
          options[:no_encoded_output]=true
          if @avoid_empty_files && (value.nil?() || (value.respond_to?(:empty?) && value.empty?()) || (value.is_a?(String) && value.strip()==''))
            debug "File #{File.join(dir,file)} has would be empty; skipping"
          else
            # do not use internal mkdir as this dir name already contains timestamps if running in versioning mode
            @@dir_mutex.synchronize {
              FileUtils.mkdir_p(dir) unless Dir.exists?(dir)
            }
            store(value,nil,options)
          end
        end
      end

      def store_hash_array_as_csv(data,opts,dirname,filename)
        column_counter=0
        columns=Hash.new()
        titles=Hash.new()
        if data.empty?()
          opts[:hash_headline].keys.each do |key|
            k=key.to_s.intern
            columns[k]=column_counter
            column_counter+=1
            titles[k]=opts[:hash_headline][key].to_s
          end if opts.include?(:hash_headline) && !opts[:hash_headline].empty?
        else
          data.each do |d|
            unless d.is_a?(Hash)
              error "The given array for #{dirname}/#{filename} contains at least one non hash member"
              return nil
            end
            d.keys.each do |key|
              k=key.to_s.intern
              next if columns.include?(k)
              columns[k]=column_counter
              column_counter+=1
              if opts.include?(:hash_headline) && opts[:hash_headline].include?(key)
                titles[k]=opts[:hash_headline][key].to_s
              else
                titles[k]=k.to_s
              end
            end
          end
        end

        csv=""
        row=Array.new(column_counter)
        titles.each do |key,value|
          row[columns[key]]=value
        end
        csv+=row.to_csv()
        data.each do |d|
          row=Array.new(column_counter)
          d.each do |key,value|
            row[columns[key.to_s.intern]]=value.to_s
          end
          csv+=row.to_csv()
        end
        file=filename.clone
        file+=".csv"
        options=opts.clone()
        options[:dirname]=dirname.clone
        options[:filename]=file
        options[:compress]=@compress_csv_output
        options[:no_encoded_output]=true
        store(csv, nil, options)
      rescue => e
        error "Failed to convert array hash for #{file.inspect()} to CSV", e
      end

      def store_readable_array(data,opts,dirname,filename)
        result=""
        data.each do |d|
          unless d.is_a?(String) || d.respond_to?(:to_s)
            error "The given array for #{filename.inspect()} contains at least one non string member"
            return nil
          end
        end
        data.each do |d|
          if d.is_a?(String) || d.respond_to?(:to_s)
            result+=d.to_s
            result+="\n"
          end
        end
        file=filename.clone
        file+=".txt"
        options=opts.clone()
        options[:dirname]=dirname.clone
        options[:filename]=file
        options[:compress]=@compress_readable_output
        options[:no_encoded_output]=true
        store(result, nil, options)
      rescue => e
        error "Failed to convert array of string for #{file.inspect()} to TXT", e
      end

      def store_array_array_as_csv(data,opts,dirname,filename)
        data.each do |d|
          unless d.is_a?(Array)
            error "The given array for #{file.inspect()} contains at least one non array member"
            return nil
          end
        end
        csv=""
        csv+=opts[:array_headline].to_csv if opts.include?(:array_headline)
        data.each { |d| csv+=d.to_csv }

        file=filename.clone
        file+=".csv"
        options=opts.clone()
        options[:dirname]=dirname.clone
        options[:filename]=file
        options[:compress]=@compress_csv_output
        options[:no_encoded_output]=true
        store(csv, nil, options)
      rescue => e
        error "Failed to convert array hash for #{file.inspect()} to CSV", e
      end


      def encode_string(data)
        return data
      end

      def encode_array(data)
        result=""
        data.each do |d|
          result+="#{Base64.strict_encode64(d.to_yaml)}\n"
        end
        return result
      end

      def encode_anything(data)
        return data.to_yaml()
      end

      def decode_string(data)
        return data
      end

      def decode_array(data)
        result=Array.new()
        data.split("\n").each do |d|
          result << YAML::load(Base64.strict_decode64(d))
        end
        return result
      end

      def decode_anything(data)
        return YAML::load(data)
      end
    end
  end
end
