# encoding: utf-8
# license: gpl2p 

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST
require 'os'

module Ghun
  module Profiler
    class MemoryUsage
      MemoryUsageMonitoringEntry=Struct.new(:name,:description,:usage,:start,:stop,:result)
      MemoryUsageMeasurementEntry=Struct.new(:absolute_time,:relative_time,:usage)
      MemoryUsageMeasurementResult=Struct.new(:min,:max,:sum,:steps,:average)
      include Ghun::Base::Logging

      def initialize(log_source=Ghun::Log::Source::BASE)
        init_logging(log_source,Ghun::Log::Type::PROFILING)
        @monitorings=Hash.new()
        @step_duration=1.0
        @thread=nil
        @monitorings_mutex=Monitor.new()
        @thread_mutex=Mutex.new()
        @active_monitorings=0
        @enabled=true
      end

      def register(name,description)
        return unless @enabled
        @monitorings_mutex.synchronize {
          if @monitorings.include?(name) then
            @monitorings[name].description=description
            @monitorings[name].description=name.to_s if @monitorings[name].description.nil?() || @monitorings[name].description==""
          else
            @monitorings[name]=MemoryUsageMonitoringEntry.new(name.to_s,description)
            @monitorings[name].description=name.to_s if @monitorings[name].description.nil?() || @monitorings[name].description==""
          end
        }
      end

      def unregister(name)
        return unless @enabled
        @monitorings_mutex.synchronize {
          @monitorings.delete(name) if @monitorings.include?(name)
        }
      end

      def enable
        @enabled=true
        @thread_mutex.synchronize {
          start_thread if @thread.nil?() && @active_monitorings>=1
        }
      end

      def disable
        @enabled=false
        @thread_mutex.synchronize {
          stop_thread
        }
      end

      def start(name)
        return unless @enabled
        @thread_mutex.synchronize {
          @active_monitorings+=1
          start_thread if @thread.nil?() || @active_monitorings==1
        }
        @monitorings_mutex.synchronize {
          if @monitorings.include?(name) then
            @monitorings[name].start=Time.now()
            @monitorings[name].stop=nil
            @monitorings[name].usage=[]
            @monitorings[name].result=MemoryUsageMeasurementResult.new(2**64,0,0,0,0)
            debug("#{@monitorings[name].description} started at #{@monitorings[name].start}")
          end
        }
      end

      def state(name)
        return unless @enabled
        @monitorings_mutex.synchronize {
          return unless @monitorings.include?(name)
          return if @monitorings[name].stop.nil?() && @monitorings[name].start.nil?()
          if @monitorings[name].stop.nil?() then
            tmp=Time.now()
            debug("#{@monitorings[name].description} started at #{@monitorings[name].start}")
            debug("#{@monitorings[name].description} is running for #{tmp - @monitorings[name].start} seconds")
            debug("#{@monitorings[name].description} used min/max/avg. #{@monitorings[name].result.min}/#{@monitorings[name].result.max}/#{@monitorings[name].result.average} megabytes of memory")
          else
            debug("#{@monitorings[name].description} started at #{@monitorings[name].start}")
            debug("#{@monitorings[name].description} stopped at #{@monitorings[name].stop}")
            debug("#{@monitorings[name].description} took #{@monitorings[name].stop - @monitorings[name].start} seconds")
            debug("#{@monitorings[name].description} used min/max/avg. #{@monitorings[name].result.min}/#{@monitorings[name].result.max}/#{@monitorings[name].result.average} megabytes of memory")
          end
        }
      end

      def stop(name)
        return unless @enabled
        @thread_mutex.synchronize {
          @active_monitorings-=1
          stop_thread if @active_monitorings==0
        }
        @monitorings_mutex.synchronize {
          if @monitorings.include?(name) && !@monitorings[name].nil?() && !@monitorings[name].start.nil?() then
            @monitorings[name].stop=Time.now()
            debug("#{@monitorings[name].description} stopped at #{@monitorings[name].stop}")
            debug("#{@monitorings[name].description} took #{@monitorings[name].stop - @monitorings[name].start} seconds")
            debug("#{@monitorings[name].description} used min/max/avg. #{@monitorings[name].result.min}/#{@monitorings[name].result.max}/#{@monitorings[name].result.average} megabytes of memory")
          end
        }
      end

      def output(name)
        return unless @enabled
        @monitorings_mutex.synchronize {
            if @monitorings.include?(name) then
            state(name)
            debug("absolute time;time since start;memory used")
            @monitorings[name].usage.each do |u|
              debug("#{u.absolute_time};#{u.relative_time};#{u.usage}")
            end
          end
        }
      end

    private

      def measurement()
        before=nil
        begin
          @monitorings_mutex.synchronize {
            before=Time.now()
            memory=OS.rss_bytes/1024.0
            @monitorings.values.each do |m|
              next if m.start.nil?()
              next unless m.stop.nil?()
              m.usage << MemoryUsageMeasurementEntry.new(before.to_f,before-m.start,memory)
              m.result.min=[m.result.min,memory].min
              m.result.max=[m.result.max,memory].max
              m.result.sum+=memory
              m.result.steps+=1
              m.result.average=m.result.sum/m.result.steps
            end
          }
        rescue => e
          error "Failed to query and store memory usage", e
        end
        after=Time.now
        return after-before
      end

      def endless_measurement()
        while @enabled && @active_monitorings > 0
          measurement_duration = measurement()
          sleep(@step_duration - measurement_duration) if (@step_duration - measurement_duration) > 0.0
        end
      end

      def start_thread()
        return unless @thread.nil?()
        @thread=Ghun::Base::Thread.run("Memory usage monitor ##{self.__id__}",0,@log_source,@log_type) do
          endless_measurement()
        end
      end

      def stop_thread()
        return if @thread.nil?()
        @thread.join()
        @thread=nil
      end
    end
  end
end
