# encoding: utf-8
# license: gpl2p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST

Ghun::Base.java_require("jsch")
module Ghun
  module Remote
    class PlainSSH < Ghun::Remote::BasePlainSSH
      @type='ssh_v2'
      Ghun::Remote::REMOTE_CONNECTION_SELECTOR.register_member(self,[:ssh,2])
      def initialize(host,auth,thread_handler=nil,strip_motd=true,mode=:sync,keep_connection=true)
        super(host,auth,thread_handler,strip_motd,mode,keep_connection)
        unless JSch::JSch.get_logger.is_a?(Ghun::Log::CompatibleLogger)
          JSch::JSch.logger=Ghun::Log::CompatibleLogger.new(global_logger(),Ghun::Log::Source::REMOTE,Ghun::Log::Type::EXTERNAL_LIBS,1)
        end
        @jsch=JSch::JSch.new()
        @session=nil
      end

      def connected?()
        !@session.nil?() && @session.is_connected()
      rescue
        return false
      end

      def run_command(command,failed_output_stdout=[],failed_output_stderr=[],allow_empty_result=false)
        @motd=execute_request({:command => '',:failed_output_stdout => [], :failed_output_stderr => [], :allow_empty_result => true}) if @motd.nil?() && @strip_motd
        if command.is_a?(Array)
          commands=Array.new()
          command.each do |c|
            commands << {:command => c, :failed_output_stdout => failed_output_stdout, :failed_output_stderr => failed_output_stderr, :allow_empty_result => allow_empty_result}
          end
          return execute_request(commands)
        elsif command.is_a?(Hash)
          return execute_request(command)
        else
          return execute_request({:command => command, :failed_output_stdout => failed_output_stdout, :failed_output_stderr => failed_output_stderr, :allow_empty_result => allow_empty_result})
        end
      end

      def enqueue_command(id,command,failed_output_stdout=[],failed_output_stderr=[],allow_empty_result=false)
        @motd=execute_request({:command => '',:failed_output_stdout => [], :failed_output_stderr => []}, :allow_empty_result => true) if @motd.nil?() && @strip_motd
        enqueue_request(id,{:command => command, :failed_output_stdout => failed_output_stdout, :failed_output_stderr => failed_output_stderr, :allow_empty_result => allow_empty_result})
      end

    private

      def connect_helper()
        begin
          #TODO implement more auth schemes in
          debug "Connecting to #{@host}"
          if !@auth.nil?() && @auth.supports_username_password?()
            @session=@jsch.get_session(@auth.username,@host,22)
            config=Java::Util::Properties.new()
            config.put("StrictHostKeyChecking","no")
            @session.set_config(config)
            @session.set_password(@auth.password)
            @session.connect()
          else
            disable_retries()
            raise Ghun::Auth::MissingAuthElementError, "Auth item #{@auth} does not support needed auth elements for host #{@host}"
          end
        rescue => e
          error "Failed to connect to host with address #{@host}: #{e.message}"
          raise
        end
        # run_command in connect_helper will trigger an endless loop
        # when relying on @connected for determining connection status
        # SSH does not rely on the @connected variable but returns the expected true
        return true
      end

      def disconnect_helper()
        debug "Disconnecting from #{@host}"
        @session.disconnect unless (@session.nil?() || !@session.is_connected())
        @session=nil
      rescue => e
        debug "SSH detected disconnect while disconnecting: #{e.message}"
        @session=nil
      end

      def execute_helper(args)
        response=nil
        begin
          response=ssh_exec(args[:command],args[:allow_empty_result])
        rescue => e
          error "Command execution for command '#{args[:command]}' on host #{@host} failed: #{e.message}"
          raise
        end if response.nil?()

        raise "Command result is nil" if response.nil?()
        warn "Command result stdout is nil" if response[:stdout].nil?()
        warn "Command result stderr is nil" if response[:stderr].nil?()
        warn "Command result exit code is nil" if response[:exit].nil?()

        args[:failed_output_stdout].each do |c|
          if response[:stdout].match(c)
            warn "Command output matched defined 'command failed' output '#{c.to_s}'"
            response[:stdout]=nil
          end unless response[:stdout].nil?()
        end
        args[:failed_output_stderr].each do |c|
          if response[:stderr].match(c)
            warn "Command output matched defined 'command failed' output '#{c.to_s}'"
            response[:stderr]=nil
          end unless response[:stderr].nil?()
        end

        raise "Command result has active failed marker" if response[:exit]==-1

        if @strip_motd && !@motd.nil?()
          if !@motd[:stdout].nil?() && @motd[:stdout]!="" && !response[:stdout].nil?() && response[:stdout]!="" && response[:stdout].index(@motd[:stdout])==0
            debug "Removing motd from stdout of command #{args[:command]} of host #{@host}"
            response[:stdout].sub!(@motd[:stdout],'')
          else
            debug "No motd in stdout of command #{args[:command]} of host #{@host}"
          end
          if !@motd[:stderr].nil?() && @motd[:stderr]!="" && !response[:stderr].nil?() && response[:stderr]!="" && response[:stderr].index(@motd[:stderr])==0
            debug "Removing motd from stderr of command #{args[:command]} of host #{@host}"
            response[:stderr].sub!(@motd[:stderr],'')
          else
            debug "No motd in stderr of command #{args[:command]} of host #{@host}"
          end
        end
        debug "Command #{args[:command]} returned with exit code #{response[:exit]} and wrote #{(response[:stdout].nil?())?"NIL":response[:stdout].size} characters in stdout and #{(response[:stderr].nil?())?"NIL":response[:stderr].size} characters in stderr"
        return response
      end

      def ssh_exec(command,allow_empty_result=false)
        result=Hash.new()
        result[:stdout]=""
        result[:stderr]=""
        result[:exit]=nil
        result[:signal]=nil

        channel=@session.open_channel("exec")
        channel.set_command(command)
        channel.set_input_stream(nil)
        stdout=channel.get_input_stream();
        stderr=channel.get_err_stream();
        channel.connect();

        tmp=Java::byte[1024].new
        while true
          while stdout.available()>0
            i=stdout.read(tmp,0,1024)
            break if i<0
            result[:stdout]+=(String.from_java_bytes(tmp[0..(i-1)]))
          end
          while stderr.available()>0
            i=stderr.read(tmp,0,1024)
            break if i<0
            result[:stderr]+=(String.from_java_bytes(tmp[0..(i-1)]))
          end
          if channel.is_closed()
            unless stdout.available()>0 || stderr.available()>0
              result[:exit]=channel.get_exit_status()
              break
            end
          end
          sleep(0.25)
        end
        channel.disconnect() if channel.is_connected  && !channel.is_closed()
        return result
      rescue => e
        error "Failed to connect to host with address #{@host}: #{e.message}"
        raise
      end
    end
  end
end
