# encoding: utf-8
# license: gpl2p

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST

require 'net/ssh'
module Ghun
  module Remote
    class PlainSSH < Ghun::Remote::BasePlainSSH
      @type='ssh_v2'
      Ghun::Remote::REMOTE_CONNECTION_SELECTOR.register_member(self,[:ssh,2])
      def initialize(host,auth,thread_handler=nil,strip_motd=true,mode=:sync,keep_connection=true)
        super(host,auth,thread_handler,strip_motd,mode,keep_connection)
        @logger_compatible=Log::CompatibleLogger.new(self,Log::Source::REMOTE,Log::Type::EXTERNAL_LIBS)
        @session=nil
      end

      def connected?()
        !@session.nil?() && !@session.closed?()
      rescue
        return false
      end

    private

      def connect_helper()
        begin
          #TODO implement more auth schemes in
          if !@auth.nil?() && @auth.supports_username_password?()
            @session=Net::SSH.start(@host,@auth.username, :logger => @logger_compatible, :"auth_methods" => ["password"], :password => @auth.password, :paranoid => false, :timeout => 15, :config => false)
          else
            disable_retries()
            raise Ghun::Auth::MissingAuthElementError, "Auth item #{@auth} does not support needed auth elements for host #{@host}"
          end
        rescue Net::SSH::AuthenticationFailed
          error "Could not authenticate for host with address #{@host}"
          error Ghun::Log::Type::REPORT, "Could not authenticate for host with address #{@host}"
          raise
        rescue Errno::ECONNREFUSED
          error "Connection attempt was refused by host with address #{@host}"
          error Ghun::Log::Type::REPORT, "Connection attempt was refused by host with address #{@host}"
          raise
        rescue Errno::EHOSTUNREACH
          error "Host unreachable for host with address #{@host}"
          error Ghun::Log::Type::REPORT, "Host unreachable for host with address #{@host}"
          raise
        rescue => e
          error "Connection to host with address #{@host} failed with an unknown error: #{e.class} #{e.message}"
          error Ghun::Log::Type::REPORT, "Connection to host with address #{@host} failed with an unknown error: #{e.class} #{e.message}"
          raise
        end
        # run_command in connect_helper will trigger an endless loop
        # when relying on @connected for determining connection status
        # SSH does not rely on the @connected variable but return the expected true
        return true
      end

      def disconnect_helper()
        @session.close unless (@session.nil?() || @session.closed?())
        @session=nil
      rescue Net::SSH::Disconnect => e
        debug "SSH detected disconnect while disconnecting: #{e.message}"
        @session=nil
      end

      def execute_helper(args)
        response=nil
        begin
          response=ssh_exec(args[:command],args[:allow_empty_result])
        rescue Net::SSH::Disconnect => e
          # net-ssh detected disconnect without receiving data
          error "Command execution for command '#{args[:command]}' on host #{@host} failed: #{e.message}"
          raise
        rescue NoMethodError
          # undetected connection failure
          error "Command execution for command '#{args[:command]}' on host #{@host} failed due to a broken connection"
          raise
        rescue IOError
          # closed stream
          error "Command execution for command '#{args[:command]}' on host #{@host} failed due to a broken connection"
          raise
        rescue
          # unknown error
          error "Command execution for command '#{args[:command]}' on host #{@host} failed due to an unknown reason"
          disable_retries()
          raise
        end if response.nil?()

        raise "Command result is nil" if response.nil?()
        warn "Command result stdout is nil" if response[:stdout].nil?()
        warn "Command result stderr is nil" if response[:stderr].nil?()
        warn "Command result exit code is nil" if response[:exit].nil?()

        args[:failed_output_stdout].each do |c|
          if response[:stdout].match(c)
            warn "Command output matched defined 'command failed' output '#{c.to_s}'"
            response[:stdout]=nil
          end unless response[:stdout].nil?()
        end
        args[:failed_output_stderr].each do |c|
          if response[:stderr].match(c)
            warn "Command output matched defined 'command failed' output '#{c.to_s}'"
            response[:stderr]=nil
          end unless response[:stderr].nil?()
        end

        raise "Command result has active failed marker" if response[:exit]==-1

        if @strip_motd && !@motd.nil?()
          if !@motd[:stdout].nil?() && @motd[:stdout]!="" && !response[:stdout].nil?() && response[:stdout]!="" && response[:stdout].index(@motd[:stdout])==0
            debug "Removing motd from stdout of command #{args[:command]} of host #{@host}"
            response[:stdout].sub!(@motd[:stdout],'')
          else
            debug "No motd in stdout of command #{args[:command]} of host #{@host}"
          end
          if !@motd[:stderr].nil?() && @motd[:stderr]!="" && !response[:stderr].nil?() && response[:stderr]!="" && response[:stderr].index(@motd[:stderr])==0
            debug "Removing motd from stderr of command #{args[:command]} of host #{@host}"
            response[:stderr].sub!(@motd[:stderr],'')
          else
            debug "No motd in stderr of command #{args[:command]} of host #{@host}"
          end
        end
        return response
      end

      def ssh_exec(command,allow_empty_result=false)
        result=Hash.new()
        result[:stdout]=""
        result[:stderr]=""
        result[:exit]=nil
        result[:signal]=nil

        channel = @session.open_channel do |chan|
          chan.exec(command) do |ch, success|
            unless success
              error "Could not execute command (ssh.channel.exec) on #{@host}:#{@port}"
            else
              ch.on_data do |c,data|
                result[:stdout]+=data
              end
              ch.on_extended_data do |c,type,data|
                if type==1
                  result[:stderr]+=data
                else
                  warn "SSH extented data contain unknown type #{type} with data #{data}"
                end
              end

              ch.on_request("exit-status") do |c,data|
                result[:exit] = data.read_long
              end

              ch.on_request("exit-signal") do |c, data|
                result[:signal] = data.read_string
              end

              ch.on_eof do |c|
                debug "Channel has send EOF after executing command #{command}"
              end
              ch.on_close do |c|
                debug "Channel is closing after executing command #{command}"
              end
            end
          end
        end
        channel.wait
        return result
      rescue Net::SSH::Disconnect => e
        if command=='' || ((result[:stdout]!='' || result[:stderr]!='') && result[:exit].nil?())
          # net-ssh detected disconnect after receiving data data; assuming success
          debug "SSH detected disconnect after receiving data, but got not exit code: #{e.message}"
          result[:exit]=0
          disconnect
          return result
        elsif command=='' || ((result[:stdout]!='' || result[:stderr]!='') && !result[:exit].nil?())
          # net-ssh detected disconnect after receiving data data; assuming success
          debug "SSH detected disconnect after receiving data: #{e.message}"
          disconnect
          return result
        elsif command!='' && result[:stdout]=='' && allow_empty_result && !result[:exit].nil?()
          # net-ssh detected disconnect after receiving data data; assuming success
          debug "SSH detected disconnect with acceptable empty response: #{e.message}"
          disconnect
          return result
        else
          # net-ssh detected disconnect without receiving data
          error "SSH detected disconnect without receiving data or with incomplete response: #{e.message}"
          debug result.inspect
          disconnect
          raise
        end
      end
    end
  end
end
