# encoding: utf-8
# license: gpl2p 

### BEGIN LICENSE NOTICE
# This file is part of %LONG% (%SHORT%)
# Copyright (C) 2010 - %YEAR%
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
### END LICENSE NOTICE

### BEGIN AUTHOR LIST
#
### END AUTHOR LIST
module Iof
  module SIMU
    class CSVImporter < Ghun::Base::Module
      @@l2eth_suffix="eth"
      @@l3v4_suffix="ipv4"
      @@l3v6_suffix="ipv6"
      Ghun::Base::Blackboard.config.declare_key(:"simu.import.base_dir",:string,File.join(Ghun::Base::Blackboard.local.data_path,'simu','import'))
      Ghun::Base::Blackboard.config.declare_key(:"simu.import.dataset",:string)
      def initialize()
        super(Ghun::Log::Source::SIMU)
        path=config[:"simu.import.base_dir"]
        dataset=config[:"simu.import.dataset"]
        path=File.join(path,dataset)
        raise "Input directory #{path} does not exist or is not readable" unless Dir.exists?(path) && File.readable?(path)
        @dataset=dataset
        @networks_file=File.join(path,'networks.yaml')
        @infrastructure_file=File.join(path,'infrastructure.csv')
        @endpoints_file=File.join(path,'endpoints.csv')
        [@networks_file,@infrastructure_file,@endpoints_file].each do |f|
          raise "Input file #{f} does not exist or is not readable" unless File.readable?(f)
        end
        Iof::Data::Mapper.instance()
        @data_handler=Iof::Data::Handler.new()
        @consistency_check_definitions=Hash.new()
        @components=Hash.new()
        @connections=Array.new()
        @stuff=Array.new()
        @ontology_components=Array.new()
        @ontology_connections=Array.new()
      end

      def import()
        input=Hash.new()
        {:network => @networks_file, :infrastructure => @infrastructure_file, :endpoints => @endpoints_file}.each do |label,file|
          begin
            input[label]=File.read(file)
            raise "Failed to get content from file" if input[label].nil?() || input [label].empty?()
          rescue => e
            error "Failed to read file #{file}", e
            raise
          end
        end
        parse_networks(input[:network])
        parse_infrastructure(input[:infrastructure])
        parse_endpoints(input[:endpoints])
        write_parsed_data()
        postprocess_input()
        send_results()
        @data_handler.write_changes
      rescue => e
        error "Failed to import data", e
        @consistency_check_definitions=Hash.new()
        @components=Hash.new()
        @connections=Array.new()
        @ontology_components=Array.new()
        @ontology_connections=Array.new()
      end

    private

      def parse_networks(input)
        parsed=YAML::load(input)
        @consistency_check_definitions[:vlan]=Hash.new()
        @consistency_check_definitions[:ipv4]=Hash.new()
        @consistency_check_definitions[:ipv6]=Hash.new()
        parsed.each do |vlan|
          if @consistency_check_definitions[:vlan].include?(vlan[:id])
            warn "Duplicate definition for vlan with id #{vlan[:id]}, skipping #{vlan}"
            next
          end
          @consistency_check_definitions[:vlan][vlan[:id]]=Hash.new()
          @consistency_check_definitions[:vlan][vlan[:id]][:id]=vlan[:id]
          @consistency_check_definitions[:vlan][vlan[:id]][:ipv4]=Array.new()
          @consistency_check_definitions[:vlan][vlan[:id]][:ipv6]=Array.new()
          @consistency_check_definitions[:vlan][vlan[:id]][:severity]=vlan[:severity]
          @consistency_check_definitions[:vlan][vlan[:id]][:severity]=:normal unless vlan[:severity]==:high || vlan[:severity]==:low
          vlan[:ipv4].each do |ipv4|
            if ipv4[:type]!=:ipv4
              warn "Found IPv4 network with non-IPv4 type: #{ipv4}"
              next
            end
            network=::IPAddress::IPv4.new("#{ipv4[:network]}/#{ipv4[:netmask]}").network
            if @consistency_check_definitions[:ipv4].include?(network.to_string.intern)
              debug "Duplicate definition for network #{network.to_string}; updating"
              @consistency_check_definitions[:ipv4][network.to_string.intern][:severity]=ipv4[:severity] if ipv4[:severity]==:high || (ipv4[:severity]==:normal && @consistency_check_definitions[:ipv4][network.to_string.intern][:severity]!=:high)
            else
              @consistency_check_definitions[:ipv4][network.to_string.intern]=Hash.new()
              @consistency_check_definitions[:ipv4][network.to_string.intern][:severity]=ipv4[:severity]
              @consistency_check_definitions[:ipv4][network.to_string.intern][:network]=network
            end
            @consistency_check_definitions[:vlan][vlan[:id]][:ipv4] << network.to_string.intern
          end unless vlan[:ipv4].nil?()
          @consistency_check_definitions[:vlan][vlan[:id]][:ipv4]=@consistency_check_definitions[:vlan][vlan[:id]][:ipv4].uniq.compact.sort
          vlan[:ipv6].each do |ipv6|
            if ipv6[:type]!=:ipv6
              warn "Found IPv6 network with non-IPv6 type: #{ipv6}"
              next
            end
            network=::IPAddress::IPv6.new("#{ipv6[:network]}/#{ipv6[:netmask]}").network
            if @consistency_check_definitions[:ipv6].include?(network.to_string.intern)
              debug "Duplicate definition for network #{network.to_string}; updating"
              @consistency_check_definitions[:ipv6][network.to_string.intern][:severity]=ipv6[:severity] if ipv6[:severity]==:high || (ipv6[:severity]==:normal && @consistency_check_definitions[:ipv6][network.to_string.intern][:severity]!=:high)
            else
              @consistency_check_definitions[:ipv6][network.to_string.intern]=Hash.new()
              @consistency_check_definitions[:ipv6][network.to_string.intern][:severity]=ipv6[:severity]
              @consistency_check_definitions[:ipv6][network.to_string.intern][:network]=network
            end
            @consistency_check_definitions[:vlan][vlan[:id]][:ipv6] << network.to_string.intern
          end unless vlan[:ipv6].nil?()
          @consistency_check_definitions[:vlan][vlan[:id]][:ipv6]=@consistency_check_definitions[:vlan][vlan[:id]][:ipv6].uniq.compact.sort
        end
      rescue => e
        error "Failed to interpret data read from #{@networks_file}", e
        raise
      end

      def parse_infrastructure(input)
        last_component=nil
        last_type=nil
        input.each_line do |line|
          begin
            next if line.strip.match(/^\#/)
            next if line.strip.empty?()
            sp_line=line.split(';')
            empty=true
            sp_line.each do |cell|
              empty&&=cell.strip.empty?()
            end
            next if empty
            component=sp_line[0].strip()
            component=last_component if component.empty?()
            if component.nil?() || component.empty?()
              warn "Got line #{line} with no component name and got no component in previous lines"
              next
            end
            last_component=component
            type=sp_line[1].strip()
            type=last_type if type.empty?()
            if type.nil?() || type.empty?()
              warn "Got line #{line} with no type and got no type in previous lines"
              next
            end
            last_type=type
            interface=sp_line[2].strip()
            next if interface.nil?() || interface.empty?()
            mode=sp_line[3].strip()
            mac=sp_line[4].strip()
            native_vlan=sp_line[5].strip()
            tagged_vlans=sp_line[6].strip()
            ipv4=sp_line[7].strip()
            ipv6=sp_line[8].strip()

            c=Hash.new()
            if @components.include?(component.intern)
              c=@components[component.intern]
            else
              c[:name]=component
              @components[component.intern]=c
              c[:interfaces]=Hash.new()
              c[:type]=type.intern if type=='switch' || type=='router' || type=='server' || type=='client' || type=='phone' || type=='ap'
              c[:type]='switch' if type.nil?() || type.empty?()
            end
            if c[:interfaces].include?(interface.intern)
              warn "Duplicate interface line for interface #{interface} at component #{component}"
              next
            end
            c[:interfaces][interface.intern]=Hash.new()
            i=c[:interfaces][interface.intern]
            i[:down]=(mode=='off')
            i[:monitor]=(mode=='monitor')
            i[:layer]=((mode=='routing')?(3):(2)) unless i[:down] || i[:monitor]
            i[:mode]=mode if !i[:down] && !i[:monitor] && i[:layer]==2
            i[:mode]='trunk' unless i[:mode]=="access"
            i[:mac_address]=Iof::Data.mac_plain(mac) unless mac.nil?() || Iof::Data.mac_plain(mac).nil?()
            begin
              vlan_i=native_vlan.to_i
              i[:native_vlan]=vlan_i unless vlan_i.nil?() || vlan_i==0
            rescue => e
              error "Could not parse native vlan #{native_vlan}", e
            end unless native_vlan.nil?() || native_vlan.empty?()
            i[:tagged_vlans]=parse_tagged_vlans(tagged_vlans.split(',')) unless tagged_vlans.nil?() || tagged_vlans.empty?()
            begin
              i[:ipv4_address]=::IPAddress::IPv4.new(ipv4)
            rescue => e
              error "Could not parse IPv4 address #{ipv4}", e
            end unless ipv4.nil?() || ipv4.empty?()
            begin
              i[:ipv6_address]=::IPAddress::IPv6.new(ipv6)
            rescue => e
              error "Could not parse IPv6 address #{ipv6}", e
            end unless ipv6.nil?() || ipv6.empty?()
          rescue => e
            error "Failed to interpret line #{line}", e
            raise
          end
        end
      rescue => e
        error "Failed to interpret data read from #{@networks_file}", e
        raise
      end

      def parse_endpoints(input)
        endpoints=Hash.new()
        input.each_line do |line|
          begin
            next if line.strip.match(/^\#/)
            next if line.strip.empty?()
            sp_line=line.split(';')
            empty=true
            sp_line.each do |cell|
              empty&&=cell.strip.empty?()
            end
            next if empty
            component=sp_line[0].strip()
            next if component.nil?() || component.empty?()
            interface=sp_line[1].strip()
            next if interface.nil?() || interface.empty?()
            mac=sp_line[2].strip()
            vlan=sp_line[3].strip()
            ipv4=sp_line[4].strip()
            ipv6=sp_line[5].strip()
            type=sp_line[6].strip()
            neighbor=sp_line[7].strip()
            neighbor_interface=sp_line[8].strip()
            if @components.include?(component.intern) && @components.include?(neighbor.intern)
              if @components[component.intern][:interfaces].keys.include?(interface.intern) && @components[neighbor.intern][:interfaces].keys.include?(neighbor_interface.intern)
                connection=Hash.new()
                connection[:first_component]=component
                connection[:first_interface]=interface
                connection[:second_component]=neighbor
                connection[:second_interface]=neighbor_interface
                @connections << connection
              else
                warn "At least one interface in the infrastructure definition is missing for the switch connection in line #{line}"
              end
            elsif @components.include?(component.intern) && !@components.include?(neighbor.intern)
              warn "Component #{neighbor} is not part of the infrastructure definition but is referenced in line #{line}"
            else
              c=Hash.new()
              if endpoints.include?(component.intern)
                c=endpoints[component.intern]
              else
                c[:name]=component
                endpoints[component.intern]=c
                c[:interfaces]=Hash.new()
                c[:type]=type.intern if type=='switch' || type=='router' || type=='server' || type=='client' || type=='phone' || type=='ap'
                c[:type]='client' if type.nil?() || type.empty?()
              end
              if c[:interfaces].include?(interface.intern)
                warn "Duplicate interface line for interface #{interface} at component #{component}"
                next
              end
              c[:interfaces][interface.intern]=Hash.new()
              i=c[:interfaces][interface.intern]
              i[:mac_address]=Iof::Data.mac_plain(mac) unless mac.nil?() || Iof::Data.mac_plain(mac).nil?()
              begin
                vlan_i=vlan.to_i
                i[:tagged_vlans]=[vlan_i] unless vlan_i.nil?() || vlan_i==0
              rescue => e
                error "Could not parse native vlan #{vlan}", e
              end unless vlan.nil?() || vlan.empty?()
              begin
                i[:ipv4_address]=::IPAddress::IPv4.new(ipv4)
              rescue => e
                error "Could not parse IPv4 address #{ipv4}", e
              end unless ipv4.nil?() || ipv4.empty?()
              begin
                i[:ipv6_address]=::IPAddress::IPv6.new(ipv6)
              rescue => e
                error "Could not parse IPv6 address #{ipv6}", e
              end unless ipv6.nil?() || ipv6.empty?()
              if @components.include?(neighbor.intern) && @components[neighbor.intern][:interfaces].keys.include?(neighbor_interface.intern)
                connection=Hash.new()
                connection[:first_component]=component
                connection[:first_interface]=interface
                connection[:second_component]=neighbor
                connection[:second_interface]=neighbor_interface
                @connections << connection
              else
                warn "Neighbor definition is not complete in infrastructure for the connection in line #{line}"
              end
            end
          rescue => e
            error "Failed to interpret line #{line}", e
            raise
          end
        end
        @components.merge!(endpoints)
      rescue => e
        error "Failed to interpret data read from #{@networks_file}", e
        raise
      end

      def parse_tagged_vlans(vlans)
        begin
          result_i=Array.new
          result=Array.new()
          parts=Array.new()
          vlans.each do |a|
            parts.concat(a.split(','))
          end
          parts.each do |p|
            if p.match(/-/) then
              first,last,garbage=p.split('-')
              debug "Unexpected format for vlan definition: '#{p}', ignoring '#{garbage.to_s}" unless garbage.nil?()
              first.strip!
              last.strip!
              Range.new(first.to_i,last.to_i).to_a.each do |r|
                next if r.nil?()
                result_i << r
              end
            else
              next if p.nil?() || p==''
              result_i << p.to_i
            end
          end
          result_i=result_i.uniq.sort.compact
          result_i.each do |r|
            result << r
          end
          result
        rescue => e
          warn "Failed to parse #{vlans.to_s}", e
          return []
        end
      end

      def write_parsed_data()
        network={:components => @components, :connections => @connections, :ccdef => @consistency_check_definitions}
        output=Ghun::Output::Versioned.new()
        output.store(network,["simu",@dataset,"network"])
      end

      def postprocess_input()
        ip_mac=Array.new()
        @components.values.each do |component|
          c=nil
          case component[:type]
          when :switch
            c=Iof::Data::NO::Switch.new()
            c[:no_switch]=true
            c.category=0
          when :router
            c=Iof::Data::NO::Switch.new()
            c[:no_router]=true
            c.category=0
          when :phone
            c=Iof::Data::NO::ManagedNetworkComponent.new()
            c[:no_phone]=true
            c.category=2
          when :ap
            c=Iof::Data::NO::ManagedNetworkComponent.new()
            c[:no_access_point]=true
            c.category=2
          else
            c=Iof::Data::NO::ManagedNetworkComponent.new()
            c[:no_endpoint]=true
            c.category=3
          end
          c.name=component[:name]
          component[:interfaces].each do |label,interface|
            l1=Iof::Data::NO::Layer1Interface.new()
            c[:no_simple_layer1_interface]=l1
            l1.name=label.to_s()
            l1[:no_monitor_port]=interface[:monitor] unless interface[:monitor].nil?()
            l2=Iof::Data::NO::EthernetInterface.new()
            l2.name="eth"
            l1[:no_ethernet_interface]=l2
            l2[:no_trunk_mode]=interface[:mode] unless interface[:monitor] || interface[:down]
            l2[:no_mac_address]=@data_handler.get_mac_address_name(interface[:mac_address]) unless interface[:mac_address].nil?()
            l2[:no_native_vlan]=@data_handler.get_vlan_name(interface[:native_vlan]) unless interface[:native_vlan].nil?()
            interface[:tagged_vlans].each do |vlan|
              l2[:no_tagged_vlan]=@data_handler.get_vlan_name(vlan) unless vlan.nil?()
            end unless interface[:tagged_vlans].nil?()
            unless interface[:ipv4_address].nil?()
              v4=Iof::Data::NO::IPv4Interface.new()
              v4.name="ipv4"
              v4_address=nil
              l2[:no_ipv4_interface]=v4
              prefix,network=get_ipv4_network_size_and_name(interface[:ipv4_address])
              if prefix==32
                v4_address=@data_handler.get_ipv4_address_name(interface[:ipv4_address].address.to_s)
                v4[:no_ipv4_address]=v4_address
              else
                interface[:ipv4_address].prefix=prefix.to_i
                v4_address=@data_handler.get_ipv4_address_name(interface[:ipv4_address].address.to_s,interface[:ipv4_address].prefix.to_s)
                v4[:no_ipv4_address]=v4_address
                v4[:no_ipv4_network]=network
              end
              ip_mac << [v4_address,@data_handler.get_mac_address_name(interface[:mac_address]),"arp:192.168.3.254"] unless interface[:mac_address].nil?()
            end
            unless interface[:ipv6_address].nil?()
              v6=Iof::Data::NO::IPv6Interface.new()
              v6.name="ipv6"
              v6_address=nil
              l2[:no_ipv6_interface]=v6
              prefix,network=get_ipv6_network_size_and_name(interface[:ipv6_address])
              if prefix==128
                v6_address=@data_handler.get_ipv6_address_name(interface[:ipv6_address].address.to_s)
                v6[:no_ipv6_address]=v6_address
              else
                interface[:ipv6_address].prefix=prefix.to_i
                v6_address=@data_handler.get_ipv6_address_name(interface[:ipv6_address].address.to_s,interface[:ipv6_address].prefix.to_s)
                v6[:no_ipv6_address]=v6_address
                v6[:no_ipv6_network]=network
              end
              ip_mac << [v6_address,@data_handler.get_mac_address_name(interface[:mac_address]),"arp:192.168.3.254"] unless interface[:mac_address].nil?()
            end
          end
          @ontology_components << c
        end
        @connections.each do |connection|
          c=Iof::Data::NO::IsConnectedTo.new()
          c.subject="#{connection[:first_component]}___#{connection[:first_interface]}"
          c.object="#{connection[:second_component]}___#{connection[:second_interface]}"
          @ontology_connections << c
        end

#        ARES 2015 proof of concept code
#        ip_mac << [:"__ipv4_address_192.168.4.100/24__",:"__mac_00005e005320__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.101/24__",:"__mac_00005e005346__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.102/24__",:"__mac_00005e005344__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.103/24__",:"__mac_00005e005342__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.104/24__",:"__mac_00005e005340__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.105/24__",:"__mac_00005e00533e__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.106/24__",:"__mac_00005e00533c__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.107/24__",:"__mac_00005e00533a__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.108/24__",:"__mac_00005e005338__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.109/24__",:"__mac_00005e005336__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.110/24__",:"__mac_00005e005334__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.111/24__",:"__mac_00005e005330__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.112/24__",:"__mac_00005e00532e__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.113/24__",:"__mac_00005e00532c__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.114/24__",:"__mac_00005e00532a__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.115/24__",:"__mac_00005e005328__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.116/24__",:"__mac_00005e005326__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.117/24__",:"__mac_00005e005324__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.118/24__",:"__mac_00005e005322__","snort:192.168.3.240",(Time.now()-92345).to_s]
#        ip_mac << [:"__ipv4_address_192.168.4.119/24__",:"__mac_00005e005315__","snort:192.168.3.240",(Time.now()-92345).to_s]


        ip_mac.each do |im|
          ipmac=Iof::Data::NO::IpMac.new()
          ipmac.subject=im[0]
          ipmac.object=im[1]


#        ARES 2015 proof of concept code
#          ipmac.annotation=im[2]
#
#          if im.length>3
#            ipmac.timestamp=im[3]
#          else
#            ipmac.timestamp=Time.now().to_s
#          end

          @stuff << ipmac
        end


        @consistency_check_definitions=nil
        @components=nil
        @connections=nil
      end

      def get_ipv4_network_size_and_name(address)
        size=0
        network=nil
        @consistency_check_definitions[:ipv4].values.each do |n|
          if n[:network].include?(address)
            if n[:network].prefix > size
              size=n[:network].prefix
              network=n[:network].to_s
            end
          end
        end
        if size==0
          return 32,nil
        else
          return size,@data_handler.get_ipv4_network_name(network, size)
        end
      end

      def get_ipv6_network_size_and_name(address)
        size=0
        network=nil
        @consistency_check_definitions[:ipv6].values.each do |n|
          if n[:network].include?(address)
            if n[:network].prefix > size
              size=n[:network].prefix
              network=n[:network].to_s
            end
          end
        end
        if size==0
          return 128,nil
        else
          return size,@data_handler.get_ipv6_network_name(network, size)
        end
      end

      def send_results()
        @ontology_components.each do |c|
          @data_handler.set_component(c, c.category)
        end
        @ontology_connections.each do |c|
          @data_handler.set_connection(c)
        end
        @stuff.each do |s|
          @data_handler.set_stuff(s)
        end
        @ontology_components=nil
        @ontology_connections=nil
        @stuff=nil
      end
    end
  end
end
